/* $Id: fsu_dir.c,v 1.6 2008/12/27 20:59:25 stacktic Exp $ */

/*
 * Copyright (c) 2008 Arnaud Ysmal.  All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/stat.h>

#include <assert.h>
#include <dirent.h>
#include <err.h>
#include <stdlib.h>
#include <string.h>

#include <rump/ukfs.h>

#include "fsu_utils.h"

static char *fsu_getdirname(struct ukfs *);

FSU_DIR
*fsu_opendir(struct ukfs *fs, const char *path)
{
	FSU_DIR *dir;
	char *cwd;
	int rv;

	assert(fs != NULL && path != NULL);

	dir = malloc(sizeof(FSU_DIR));
	if (dir == NULL)
		return NULL;

	/* get current absolute path */
	cwd = fsu_getcwd(fs);

	/* get target absolute path */
	rv = ukfs_chdir(fs, path);
	if (rv == -1) {
		free(dir);
		return NULL;
	}
	dir->dd_path = fsu_getcwd(fs);

	/* return to the original directory */
	rv = ukfs_chdir(fs, cwd);
	if (rv == -1) {
		free(dir);
		free(dir->dd_path);
		free(cwd);
		return NULL;
	}

	free(cwd);
	dir->dd_ukfs = fs;
	dir->dd_off = dir->dd_size = 0;
	return dir;
}

void
fsu_closedir(FSU_DIR *dir)
{

	assert(dir != NULL);

	free(dir->dd_path);
	free(dir);
}

struct dirent
*fsu_readdir(FSU_DIR *dir)
{
	struct dirent *dent;

	assert(dir != NULL);

	if (dir->dd_size <= 0) {
		dir->dd_size = ukfs_getdents(dir->dd_ukfs, dir->dd_path,
					     &dir->dd_off,
					     dir->dd_buf, sizeof(dir->dd_buf));
		if (dir->dd_size <= 0)
			return NULL;

		dir->dd_dent = (struct dirent *)dir->dd_buf;
	}

	dent = dir->dd_dent;
	
	dir->dd_size -= _DIRENT_SIZE(dir->dd_dent);
	dir->dd_dent = _DIRENT_NEXT(dir->dd_dent);

	return dent;
}

void
fsu_rewinddir(FSU_DIR *dir)
{
	
	assert(dir != NULL);

	dir->dd_off = 0;
	dir->dd_size = 0;
}

#define ALLOC_SIZE (10)

/* Gets the absolute path of the current directory */
char
*fsu_getcwd(struct ukfs *fs)
{
	int i, rv;
	char **names, **tmpnames;
	char *dirname, *tmp;
	size_t len, clen;

	names = malloc(ALLOC_SIZE * sizeof(char *));
	if (names == NULL)
		return NULL;

	len = 1;

	for (i = 0;; ++i) {
		tmp = fsu_getdirname(fs);
		if (tmp == NULL)
			goto err;
		else if (tmp[0] == '/' && tmp[1] == '\0') {
			free(tmp);
			break;
		}

		names[i] = tmp;
		len += strlen(tmp) + 1;

		rv = ukfs_chdir(fs, "..");
		if (rv == -1)
			goto err;

		if ((i + 1) % ALLOC_SIZE == 0) {
			tmpnames = realloc(names, 
			    (i + 1 + ALLOC_SIZE) * sizeof(char *));
			if (tmpnames == NULL)
				goto err;
			names = tmpnames;
		}
	}

	if (i == 0)
		return strdup("/");

	dirname = malloc(len);
	if (dirname == NULL)
		goto err;

	for (clen = 0, --i; i >= 0; --i) {
		dirname[clen] = '/';
		dirname[clen + 1] = '\0';
		clen = strlcat(dirname, names[i], len);
		free(names[i]);
	}
	free(names);

	ukfs_chdir(fs, dirname);
	return dirname;

err:
	for (--i; i >= 0; --i) {
		ukfs_chdir(fs, names[i]);
		free(names[i]);
	}
	free(names);
	return NULL;
}

static ino_t root_ino = 0;

/* Gets the name of the current directory */
static char
*fsu_getdirname(struct ukfs *fs)
{
	struct stat dot;
	struct dirent *dent;
	int rv, size;
	uint8_t buf[8192];
	off_t off;

	off = 0;

	if (root_ino == 0) {
		rv = ukfs_stat(fs, "/", &dot);
		if (rv == -1)
			return NULL;
		root_ino = dot.st_ino;
	}

	rv = ukfs_stat(fs, ".", &dot);
	if (rv == -1)
		return NULL;

	if (dot.st_ino == root_ino)
		return strdup("/");

	for (;;) {
		size = ukfs_getdents(fs, "..", &off, buf, sizeof(buf));
		if (size == -1) {
			warn("..");
			return NULL;
		} else if (size == 0)
			return NULL;
		dent = (struct dirent *)buf;

		while (size > 0) {
			if (dent->d_fileno == dot.st_ino)
				return strdup(dent->d_name);
			size -= _DIRENT_SIZE(dent);
			dent = _DIRENT_NEXT(dent);
		}
	}
	/* NOTREACHED */
}
