/*
 	Copyright (C) 2001  Sony Computer Entertainment Inc.
 
  This file is subject to the terms and conditions of the GNU Library
  General Public License Version 2. See the file "COPYING.LIB" in the 
  main directory of this archive for more details.
*/

//
// "main.c"
//
//
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>
#include "ps2gs.h"
#include "ps2vpu.h"
#include "ps2dma.h"
#include "ps2vpufile.h"
#include "mathfunc.h"
#include "sjoy.h"

//----------------------------------------------------------------------
#define IMAGE_SIZE 256

//----------------------------------------------------------------------
IMPORT_VPU_SYMBOL(float *, My_matrix, 0);
IMPORT_VPU_SYMBOL(float *, My_rot_trans_matrix, 0);
IMPORT_VPU_SYMBOL(float *, My_light_matrix, 0);
IMPORT_VPU_SYMBOL(float *, My_lcolor_matrix, 0);
IMPORT_VPU_SYMBOL(__u128 *, My_texture1, 0);
IMPORT_VPU_SYMBOL(__u128 *, My_cube_start, 0);
IMPORT_VPU_SYMBOL(__u128 *, My_torus_start, 0);
IMPORT_VPU_SYMBOL(__u128 *, My_dma_start, 0);
IMPORT_VPU_SYMBOL(__u128 *, My_dma_next, 0);

//----------------------------------------------------------------------
typedef struct {
	ps2_giftag giftag;
	ps2_gsreg_test gs_test;
	__u64 gs_test1addr;
	ps2_gs_alphaenv gs_alpha;
	ps2_gs_texenv gs_tex;
} TexEnv;

//----------------------------------------------------------------------
char *g_program_name;

int g_inter;
int g_out_mode;
int g_ff_mode;
int g_resolution;
int g_refresh_rate;
int g_psm;
int g_zpsm;
int g_zbits;

ps2Samp0FVECTOR camera_p = { 0, 0, -25, 0 };
ps2Samp0FVECTOR camera_zd = { 0, 0, 1, 1 };
ps2Samp0FVECTOR camera_yd = { 0, 1, 0, 1 };
ps2Samp0FVECTOR camera_rot = { 0, 0, 0, 0 };

ps2Samp0FVECTOR light0 = { 0.0, 1.5, 0.5, 0 };
ps2Samp0FVECTOR light1 = { 1.5, -0.5, 0.5, 0 };
ps2Samp0FVECTOR light2 = { -1.5, -0.5, 0.5, 0 };

ps2Samp0FVECTOR color0  = { 0.3, 0.3, 0.8, 0 };
ps2Samp0FVECTOR color1  = { 0.8, 0.3, 0.3, 0 };
ps2Samp0FVECTOR color2  = { 0.3, 0.8, 0.3, 0 };

ps2Samp0FVECTOR ambient = { 0.2, 0.2, 0.2, 0 };

ps2Samp0FVECTOR obj_trans = { 0, 0, 0, 0 };
ps2Samp0FVECTOR obj_rot = { 0, 0, 0, 0 };

ps2Samp0FMATRIX local_world;
ps2Samp0FMATRIX world_view;
ps2Samp0FMATRIX view_screen;
ps2Samp0FMATRIX local_screen;

ps2Samp0FMATRIX normal_light;
ps2Samp0FMATRIX light_color;
ps2Samp0FMATRIX local_light;
ps2Samp0FMATRIX local_color;

ps2Samp0FMATRIX work;

int paddata;

ps2_gs_dbuff g_db;
ps2_gs_finish g_finish;
struct ps2_image g_img;
int g_textop64;
TexEnv texenv;

int g_fd_gs;
ps2_vpu *g_vpu0, *g_vpu1;

//----------------------------------------------------------------------
void load_teximages(void)
{
	ps2_gs_set_image(&g_img, g_textop64, IMAGE_SIZE / 64, PS2_GS_PSMCT32,
					 0, 0, IMAGE_SIZE, IMAGE_SIZE, My_texture1);
	ps2_gs_load_image(&g_img);
}

//----------------------------------------------------------------------
int release(void)
{
	if (g_fd_gs >= 0) {
		ps2_gs_close();
		g_fd_gs = -1;
	}
	
	// close vpu1 first
	if (g_vpu1) {
		ps2_vpu_close(g_vpu1);
		g_vpu1 = NULL;
	}
	
	if (g_vpu0) {
		ps2_vpu_close(g_vpu0);
		g_vpu0 = NULL;
	}
	
	sjoy_close();
	
	return PS2_GS_VC_REL_SUCCESS;
}

//----------------------------------------------------------------------
int acquire(void)
{
	assert(g_fd_gs == -1);
	assert(g_vpu0 == NULL);
	assert(g_vpu1 == NULL);
	
	g_fd_gs = ps2_gs_open(-1);
	g_vpu0 = ps2_vpu_open(0); // open vpu0 first
	g_vpu1 = ps2_vpu_open(1);
	
	if (g_fd_gs < 0 || g_vpu0 == NULL || g_vpu1 == NULL) {
		release();
	    return PS2_GS_VC_ACQ_FAILURE;
	}
	
	ps2_vpu_reset(g_vpu1); // reset vpu1 first
	ps2_vpu_reset(g_vpu0);
	
	ps2_gs_reset(0, g_inter, g_out_mode, g_ff_mode, g_resolution,
				 g_refresh_rate);
	
	load_teximages();
	
	sjoy_open();
	
	return PS2_GS_VC_ACQ_SUCCESS;
}

//----------------------------------------------------------------------
void CopySamp0Matrix(float *dst, ps2Samp0FMATRIX *src)
{
	dst[ 0] = (*src)[0][0];
	dst[ 1] = (*src)[1][0];
	dst[ 2] = (*src)[2][0];
	dst[ 3] = (*src)[3][0];
	
	dst[ 4] = (*src)[0][1];
	dst[ 5] = (*src)[1][1];
	dst[ 6] = (*src)[2][1];
	dst[ 7] = (*src)[3][1];
	
	dst[ 8] = (*src)[0][2];
	dst[ 9] = (*src)[1][2];
	dst[10] = (*src)[2][2];
	dst[11] = (*src)[3][2];
	
	dst[12] = (*src)[0][3];
	dst[13] = (*src)[1][3];
	dst[14] = (*src)[2][3];
	dst[15] = (*src)[3][3];
}

//----------------------------------------------------------------------
void SetVu1PacketMatrix(void)
{
	CopySamp0Matrix(My_light_matrix, &local_light);
	CopySamp0Matrix(My_lcolor_matrix, &light_color);
	CopySamp0Matrix(My_rot_trans_matrix, &local_world);
	CopySamp0Matrix(My_matrix, &local_screen);
}

//----------------------------------------------------------------------
void usage(const char *msg1, const char *msg2)
{
	if (msg1) {
		fprintf(stderr, "%s: %s %s\n", g_program_name, msg1, msg2 ? msg2 : "");
	}
	
	fprintf(stderr,
			"Usage: %s [<ii>] [<mode>] [<ff>] [<reso>] [<rate>] [<bpp>] [<z-bpp>]\n",
			g_program_name);
	fprintf(stderr,
			"    ii: interlace or noninterlace\n"
			"        -inter, -nointer\n"
			"    mode: video mode\n"
			"        -ntsc, -pal, -dtv, -vesa\n"
			"    ff: field/frame mode\n"
			"        -frame, -field\n"
			"    reso: resolution\n"
			"        -640x480, -800x600, -1024x768\n"
			"    rate: refresh rate\n"
			"        -60, -75\n"
			"    bpp: color bits per pixel\n"
			"        -16, -24, -32\n"
			"    z-bpp: depth bits per pixel\n"
			"        -z0, -z16, -z24\n"
			);
	
	exit(1);
}

//----------------------------------------------------------------------
void options(int argc, char *argv[])
{
	int i;
	
	g_inter = PS2_GS_NOINTERLACE;
	g_out_mode = PS2_GS_VESA;
	g_ff_mode = PS2_GS_FRAME;
	g_resolution = PS2_GS_640x480;
	g_refresh_rate = PS2_GS_60Hz;
	g_psm = PS2_GS_PSMCT32;
	g_zpsm = PS2_GS_PSMZ24;
	g_zbits = 24;
	
	for (i = 1; i < argc; i++) {
		char *p = argv[i];
		
		if (strcmp(p, "-inter") == 0) {
			g_inter = PS2_GS_INTERLACE;
		} else if (strcmp(p, "-nointer") == 0) {
			g_inter = PS2_GS_NOINTERLACE;
			
		} else if (strcmp(p, "-ntsc") == 0) {
			g_out_mode = PS2_GS_NTSC;
		} else if (strcmp(p, "-pal") == 0) {
			g_out_mode = PS2_GS_PAL;
		} else if (strcmp(p, "-dtv") == 0) {
			g_out_mode = PS2_GS_DTV;
		} else if (strcmp(p, "-vesa") == 0) {
			g_out_mode = PS2_GS_VESA;
			
		} else if (strcmp(p, "-frame") == 0) {
			g_ff_mode = PS2_GS_FRAME;
		} else if (strcmp(p, "-field") == 0) {
			g_ff_mode = PS2_GS_FIELD;
			
		} else if (strcmp(p, "-640x480") == 0) {
			g_resolution = PS2_GS_640x480;
		} else if (strcmp(p, "-800x600") == 0) {
			g_resolution = PS2_GS_800x600;
		} else if (strcmp(p, "-1024x768") == 0) {
			g_resolution = PS2_GS_1024x768;
		} else if (strcmp(p, "-1280x1024") == 0) {
			g_resolution = PS2_GS_1280x1024;
			
		} else if (strcmp(p, "-60") == 0) {
			g_refresh_rate = PS2_GS_60Hz;
		} else if (strcmp(p, "-75") == 0) {
			g_refresh_rate = PS2_GS_75Hz;
			
		} else if (strcmp(p, "-32") == 0) {
			g_psm = PS2_GS_PSMCT32;
		} else if (strcmp(p, "-24") == 0) {
			g_psm = PS2_GS_PSMCT24;
		} else if (strcmp(p, "-16") == 0) {
			g_psm = PS2_GS_PSMCT16S;
			
		} else if (strcmp(p, "-z24") == 0) {
			g_zpsm = PS2_GS_PSMZ24;
			g_zbits = 24;
		} else if (strcmp(p, "-z16") == 0) {
			g_zpsm = PS2_GS_PSMZ16S;
			g_zbits = 16;
		} else if (strcmp(p, "-z0") == 0) {
			g_zpsm = PS2_GS_PSMZ24;
			g_zbits = 0;
			
		} else {
			usage("illegal optopn", p);
		}
	}
	
	if (g_psm == PS2_GS_PSMCT16S && g_zpsm == PS2_GS_PSMZ16) {
		g_psm = PS2_GS_PSMCT16;
		g_zpsm = PS2_GS_PSMZ16;
	}
}

//----------------------------------------------------------------------
int main(int argc, char *argv[])
{
	int frame, odev, obj_switch, next2k;
	float delta;
	int toggle = 0;
	VPUFILE *vfd;
	ps2_gs_gparam *gp = ps2_gs_get_gparam();
	
	g_program_name = argv[0];
	
	options(argc, argv);
	
	g_fd_gs = ps2_gs_open(-1);
	g_vpu0 = ps2_vpu_open(0);
	g_vpu1 = ps2_vpu_open(1);
	
	ps2_gs_vc_graphicsmode();
	
	vfd = vpuobj_open("flower.elf", O_DATA_PS2MEM);
	if (vfd == NULL) {
		perror("vpuobj_open");
		exit(1);
	}
	
	obj_switch = 0;
	delta = 1.0f * (float)M_PI / 180.0f;
	
	ps2_gs_reset(0, g_inter, g_out_mode, g_ff_mode, g_resolution,
				 g_refresh_rate);	
	
	next2k = ps2_gs_set_dbuff(&g_db, g_psm, gp->width, gp->height,
							  (g_zbits == 0) ? 0 : PS2_GS_ZGREATER,
							  g_zpsm, 1);
	*(__u64 *)&g_db.clear0.rgbaq = PS2_GS_SETREG_RGBAQ(0x10, 0x10, 0x20, 0x80,
													 0x3f800000);
	*(__u64 *)&g_db.clear1.rgbaq = PS2_GS_SETREG_RGBAQ(0x10, 0x10, 0x20, 0x80,
													 0x3f800000);
	
	g_textop64 = next2k * 2048 / 64;
	
	load_teximages();
	
	sjoy_open();
	
	frame = 0;
	
	// clear back buffer
	ps2_gs_put_drawenv(&g_db.giftag1);
	
	// wait clear done
	ps2_gs_set_finish(&g_finish);
	ps2_gs_wait_finish(&g_finish);
	
	odev = !ps2_gs_sync_v(0);
	
	ps2_gs_start_display(1);
	
	ps2_gs_vc_enablevcswitch(acquire, release);
	
	for (; ;) {
		const float pi = (float)M_PI;
		const float pi2 = pi * 2;
		
		ps2_gs_vc_lock();
		
		ps2_gs_set_half_offset((frame & 1) ? &g_db.draw1 : &g_db.draw0, odev);
		ps2_gs_swap_dbuff(&g_db, frame);
		
		// --- read pad ---
		sjoy_poll();
		paddata = sjoy_get_ps2_button(0);
		
		// --- object rotate & change view point ---
		if (paddata & SJOY_PS2_L_DOWN) {
			obj_rot[0] += delta;
			if (obj_rot[0] > pi) {
				obj_rot[0] -= pi2;
			}
		}
		
		if (paddata & SJOY_PS2_L_UP) {
			obj_rot[0] -= delta;
			if (obj_rot[0] < -pi) {
				obj_rot[0] += pi2;
			}
		}
		
		if (paddata & SJOY_PS2_L_RIGHT) {
			obj_rot[1] += delta;
			if (obj_rot[1] > pi) {
				obj_rot[1] -= pi2;
			}
		}
		
		if (paddata & SJOY_PS2_L_LEFT) {
			obj_rot[1] -= delta;
			if (obj_rot[1] < -pi) {
				obj_rot[1] += pi2;
			}
		}
		
		if (paddata & SJOY_PS2_L1) {
			obj_rot[2] += delta;
			if (obj_rot[2] > pi) {
				obj_rot[2] -= pi2;
			}
		}
		
		if (paddata & SJOY_PS2_L2) {
			obj_rot[2] -= delta;
			if (obj_rot[2] < -pi) {
				obj_rot[2] += pi2;
			}
		}
		
		if (paddata & SJOY_PS2_R_DOWN) {
			camera_rot[0] += delta;
			if (camera_rot[0] > 0.4f * pi) {
				camera_rot[0] = 0.4f * pi;
			}
		}
		
		if (paddata & SJOY_PS2_R_UP) {
			camera_rot[0] -= delta;
			if (camera_rot[0] < -0.4f * pi) {
				camera_rot[0] = -0.4f * pi;
			}
		}
		
		if (paddata & SJOY_PS2_R_RIGHT) {
			camera_rot[1] += delta;
			if (camera_rot[1] > pi) {
				camera_rot[1] -= pi2;
			}
		}
		
		if (paddata & SJOY_PS2_R_LEFT) {
			camera_rot[1] -= delta;
			if (camera_rot[1] < -pi) {
				camera_rot[1] += pi2;
			}
		}
		
		if (paddata & SJOY_PS2_R1) {
			camera_p[2] -= delta * 5;
			if (camera_p[2] < -100) {
				camera_p[2] = -100;
			}
		}
		
		if (paddata & SJOY_PS2_R2) {
			camera_p[2] += delta*5;
			if (camera_p[2] > -10) {
				camera_p[2] = -10;
			}
		}
		
		if (!toggle && (paddata & SJOY_PS2_SELECT)) {
			obj_switch ^= 1;
			toggle = 1;
		} else if (!(paddata & SJOY_PS2_SELECT)) {
			toggle = 0;
		}
		
		// set texture env
		ps2_gs_set_texenv(&texenv.gs_tex, 0, &g_img, 8, 8, 0, 0, 0, 0, 1);
		
		PS2_GIFTAG_CLEAR_TAG(&texenv.giftag);
		texenv.giftag.NLOOP = sizeof(texenv) / 16 - 1;
		texenv.giftag.EOP = 1;
		texenv.giftag.PRE = 0;
		texenv.giftag.NREG = 1;
		texenv.giftag.REGS0 = PS2_GIFTAG_REGS_AD;
		
		ps2_gs_set_alphaenv(&texenv.gs_alpha, 0);
		*(__u64 *)&texenv.gs_alpha.alpha1 = PS2_GS_SETREG_ALPHA(0, 1, 0, 1, 0);
	
		texenv.gs_test1addr = PS2_GS_TEST_1;
		*(__u64 *)&texenv.gs_test =  PS2_GS_SETREG_TEST(0, 0, 0, 0, 0, 0, 1,
														(g_zbits == 0) ? PS2_GS_ZALWAYS : PS2_GS_ZGREATER);
		
		*(__u64 *)&texenv.gs_tex.clamp1 = PS2_GS_SETREG_CLAMP(0, 0, 0, 0, 0, 0);
	
		ps2_gs_put_drawenv(&texenv.giftag);
	
		// local -> world (rotate)matrix
		ps2Samp0UnitMatrix(work);
		ps2Samp0RotMatrix(local_world, work, obj_rot);
		
		// color&normal matrix setting
		ps2Samp0NormalLightMatrix(normal_light, light0, light1, light2);
		ps2Samp0LightColorMatrix(light_color, color0, color1, color2, ambient);
		
		// light(normal) -> local_light matrix
		ps2Samp0MulMatrix(local_light, normal_light, local_world);
		
		// local -> world (rotate&translate)matrix
		ps2Samp0TransMatrix(local_world, local_world, obj_trans);
		
		// world -> view matrix
		ps2Samp0RotCameraMatrix(world_view, camera_p, camera_zd, camera_yd,
								camera_rot);
		
		// view -> screen matrix
		ps2Samp0ViewScreenMatrix(view_screen, 512.0f, 1.0f, gp->pixel_ratio,
								 gp->center_x, gp->center_y,
								 1.0f, (g_zbits == 0) ? 2 : ((1 << g_zbits) - 1),
								 1.0f, 65536.0f);
		
		// local -> screen matrix
		ps2Samp0MulMatrix(local_screen, view_screen, world_view);
		
		SetVu1PacketMatrix();
		
		// use ioctl
		{
			if (obj_switch == 0) {
				((ps2_dmatag *)My_dma_next)->ADDR = (__u32)My_cube_start;
			} else if (obj_switch == 1) {
				((ps2_dmatag *)My_dma_next)->ADDR = (__u32)My_torus_start;
			}
			ps2_dma_start(ps2_vpu_fd(g_vpu1), vfd, (ps2_dmatag *)My_dma_start);
		}
		
		frame++;
		odev = !ps2_gs_sync_v(0);
		
		ps2_gs_vc_unlock();
	}
	
	vpuobj_close(vfd);
	
	ps2_gs_close();
	ps2_vpu_close(g_vpu0);
	ps2_vpu_close(g_vpu1);
	
	return 0;
}
