/*
 	Copyright (C) 2001  Sony Computer Entertainment Inc.
 
  This file is subject to the terms and conditions of the GNU Library
  General Public License Version 2. See the file "COPYING.LIB" in the 
  main directory of this archive for more details.
*/

// 
// "blow.c"
// 
// 
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <libvu0.h>
#include <linux/ps2/gs.h>
#include "ps2gs.h"
#include "ps2dma.h"
#include "ps2vpu.h"
#include "ps2vpufile.h"
#include "libvu0.h"
#include "gas.h"
#include "blow.h"
#include "physics.h"
#include "sjoy.h"

//----------------------------------------------------------------------
IMPORT_VPU_SYMBOL(u_int *, My_part_src, 0)
IMPORT_VPU_SYMBOL(float *, My_ground_matrix, 0)
IMPORT_VPU_SYMBOL(__u128 *, My_texture1, 0)
IMPORT_VPU_SYMBOL(__u128 *, My_texture3, 0)
IMPORT_VPU_SYMBOL(__u128 *, My_dma_start, 0)

IMPORT_VPU_SYMBOL(__u128 *, My_dma_start_part_fire, 0)
IMPORT_VPU_SYMBOL(__u128 *, My_dma_start_shadow_part_fire, 0)
IMPORT_VPU_SYMBOL(float *, My_matrix_fire, 0)
IMPORT_VPU_SYMBOL(float *, My_shadow_matrix_fire, 0)
IMPORT_VPU_SYMBOL(float *, My_offset_fire, 0)
IMPORT_VPU_SYMBOL(float *, My_shadow_offset_fire, 0)
IMPORT_VPU_SYMBOL(__u128 *, My_texture2_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part1_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part_second_pole_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part_third_pole_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part1_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part_pole2_fire, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part_pole3_fire, 0)

IMPORT_VPU_SYMBOL(__u128 *, My_dma_start_part_gas, 0)
IMPORT_VPU_SYMBOL(__u128 *, My_dma_start_shadow_part_gas, 0)
IMPORT_VPU_SYMBOL(float *, My_matrix_gas, 0)
IMPORT_VPU_SYMBOL(float *, My_shadow_matrix_gas, 0)
IMPORT_VPU_SYMBOL(float *, My_offset_gas, 0)
IMPORT_VPU_SYMBOL(float *, My_shadow_offset_gas, 0)
IMPORT_VPU_SYMBOL(__u128 *, My_texture2_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part1_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part_second_pole_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_part_third_pole_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part1_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part_pole2_gas, 0)
IMPORT_VPU_SYMBOL(u_int *, My_shadow_part_pole3_gas, 0)

//----------------------------------------------------------------------
char *g_program_name;

int g_inter;
int g_out_mode;
int g_ff_mode;
int g_resolution;
int g_refresh_rate;
int g_psm;
int g_zpsm;
int g_zbits;

int g_fire;
int g_image_size;
float g_part_size;
float g_part_size_shadow;

float *g_my_offset;
float *g_my_shadow_offset;
float *g_my_matrix;
float *g_my_shadow_matrix;
__u128 *g_my_texture2;

__u128 *g_my_dma_start_part;
__u128 *g_my_dma_start_shadow_part;

float g_fov = 42.0f;
float screenOffset[16];
u_int fire_bound[FIRE_NUM_BLOCK * FIRE_NUM_PART * FIRE_NUM_EXPLODE];

ps2_gs_dbuff g_db;
ps2_gs_finish g_finish;

struct {
	ps2_giftag giftag;
	//
	ps2_gsreg_test gs_test;
	ps2_gsreg_addr gs_test1addr;
	//
	ps2_gs_alphaenv gs_alpha;
	ps2_gs_texenv gs_tex;
} texenv1, texenv2;

struct ps2_image g_img1, g_img2, g_img3;
int g_textop64;

int g_fd_gs;
ps2_vpu *g_vpu0, *g_vpu1;

//----------------------------------------------------------------------
void load_teximages(void)
{
	int texad64 = g_textop64;
	int size;
	
	// image 1
	ps2_gs_set_image(&g_img1, texad64, g_image_size / 64, PS2_GS_PSMCT32,
					 0, 0, g_image_size, g_image_size, My_texture1);
	size = ps2_gs_load_image(&g_img1);
	texad64 += size / 64;
	
	// image 2
	ps2_gs_set_image(&g_img2, texad64, 1, PS2_GS_PSMCT32,
					 0, 0, 32, 32, g_my_texture2);
	size = ps2_gs_load_image(&g_img2);
	texad64 += size / 64;
	
	// image 3
	ps2_gs_set_image(&g_img3, texad64, 1, PS2_GS_PSMCT32,
					 0, 0, 32, 32, My_texture3);
	size = ps2_gs_load_image(&g_img3);
	texad64 += size / 64;
}

//----------------------------------------------------------------------
int release(void)
{
	if (g_vpu1) {
		ps2_vpu_close(g_vpu1);
		g_vpu1 = NULL;
	}
	
	if (g_vpu0) {
		ps2_vpu_close(g_vpu0);
		g_vpu0 = NULL;
	}
	
	if (g_fd_gs >= 0) {
		ps2_gs_close();
		g_fd_gs = -1;
	}
	
	return PS2_GS_VC_REL_SUCCESS;
}

//----------------------------------------------------------------------
int acquire(void)
{
	g_fd_gs = ps2_gs_open(-1);
	g_vpu0 = ps2_vpu_open(0);
	g_vpu1 = ps2_vpu_open(1);
	
	if (g_fd_gs < 0 || g_vpu0 == NULL || g_vpu1 == NULL) {
	    release();
	    return PS2_GS_VC_ACQ_FAILURE;
	}
	
	ps2_vpu_reset(g_vpu1);
	ps2_vpu_reset(g_vpu0);
	
	ps2_gs_reset(0, g_inter, g_out_mode, g_ff_mode, g_resolution,
				 g_refresh_rate);
	
	load_teximages();
	
	return PS2_GS_VC_ACQ_SUCCESS;
}

//----------------------------------------------------------------------
void CreateViewingMatrix(ps2_vu0_fvector view, 
						 ps2_vu0_fvector interest,
						 ps2_vu0_fmatrix persMat)
{
	ps2_gs_gparam *gp = ps2_gs_get_gparam();
	float viewLength, upLength, rightLength; 
	float fFOV; 
	ps2_vu0_fmatrix viewMat, transMat, projMat, screenMat;
	ps2_vu0_fvector viewVector, upVector, rightVector, tmpVector1;
	ps2_vu0_fvector tmpVector2;
	
	ps2_vu0_sub_vector(viewVector, interest, view);
	viewLength = sqrtf(viewVector[0] * viewVector[0] + 
					   viewVector[1] * viewVector[1] +
					   viewVector[2] * viewVector[2]);	
	ps2_vu0_normalize(viewVector, viewVector);
	
	upVector[0] = 0.0f; 
	upVector[1] = 1.0f; 
	upVector[2] = 0.0f; 
	upVector[3] = 1.0f; 
	
	ps2_vu0_outer_product(tmpVector1, viewVector, upVector);
	ps2_vu0_normalize(tmpVector1, tmpVector1);
	ps2_vu0_outer_product(upVector, tmpVector1, viewVector);
	ps2_vu0_outer_product(rightVector, upVector, viewVector);
	
	// ------ set offset for screen ------
	ps2_vu0_scale_vector(tmpVector2, rightVector, -1.0f);
	ps2_vu0_sub_vector(tmpVector1, tmpVector2, upVector);
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size);
	screenOffset[0] = tmpVector1[0];
	screenOffset[1] = tmpVector1[1];
	screenOffset[2] = tmpVector1[2];
	
	ps2_vu0_sub_vector(tmpVector1, upVector, rightVector);
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size);
	screenOffset[4] = tmpVector1[0];
	screenOffset[5] = tmpVector1[1];
	screenOffset[6] = tmpVector1[2];
	
	ps2_vu0_sub_vector(tmpVector1, rightVector, upVector);
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size);
	screenOffset[8] = tmpVector1[0];
	screenOffset[9] = tmpVector1[1];
	screenOffset[10] = tmpVector1[2];
	
	ps2_vu0_add_vector(tmpVector1, rightVector, upVector);
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size);
	screenOffset[12] = tmpVector1[0];
	screenOffset[13] = tmpVector1[1];
	screenOffset[14] = tmpVector1[2];
	
	((u_int *)g_my_offset)[0] = *(u_int *)&(screenOffset[0]);
	((u_int *)g_my_offset)[1] = *(u_int *)&(screenOffset[1]);
	((u_int *)g_my_offset)[2] = *(u_int *)&(screenOffset[2]);
	
	((u_int *)g_my_offset)[4] = *(u_int *)&(screenOffset[4]);
	((u_int *)g_my_offset)[5] = *(u_int *)&(screenOffset[5]);
	((u_int *)g_my_offset)[6] = *(u_int *)&(screenOffset[6]);
	
	((u_int *)g_my_offset)[8] = *(u_int *)&(screenOffset[8]);
	((u_int *)g_my_offset)[9] = *(u_int *)&(screenOffset[9]);
	((u_int *)g_my_offset)[10] = *(u_int *)&(screenOffset[10]);
	
	((u_int *)g_my_offset)[12] = *(u_int *)&(screenOffset[12]);
	((u_int *)g_my_offset)[13] = *(u_int *)&(screenOffset[13]);
	((u_int *)g_my_offset)[14] = *(u_int *)&(screenOffset[14]);
	
	// ========== set offset for ground =========
	tmpVector1[0] = 1.0f;
	tmpVector1[1] = 0.0f;
	tmpVector1[2] = 1.0f;
	tmpVector1[3] = 0.0f;
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size_shadow);
	((u_int *)g_my_shadow_offset)[0] = *(u_int *)&(tmpVector1[0]);
	((u_int *)g_my_shadow_offset)[1] = *(u_int *)&(tmpVector1[1]);
	((u_int *)g_my_shadow_offset)[2] = *(u_int *)&(tmpVector1[2]);
	
	tmpVector1[0] = -1.0f;
	tmpVector1[2] = 1.0f;
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size_shadow);
	((u_int *)g_my_shadow_offset)[4] = *(u_int *)&(tmpVector1[0]);
	((u_int *)g_my_shadow_offset)[5] = *(u_int *)&(tmpVector1[1]);
	((u_int *)g_my_shadow_offset)[6] = *(u_int *)&(tmpVector1[2]);
	
	tmpVector1[0] = 1.0f;
	tmpVector1[2] = -1.0f;
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size_shadow);
	((u_int *)g_my_shadow_offset)[8] = *(u_int *)&(tmpVector1[0]);
	((u_int *)g_my_shadow_offset)[9] = *(u_int *)&(tmpVector1[1]);
	((u_int *)g_my_shadow_offset)[10] = *(u_int *)&(tmpVector1[2]);
	
	tmpVector1[0] = -1.0f;
	tmpVector1[2] = -1.0f;
	ps2_vu0_scale_vector(tmpVector1, tmpVector1, g_part_size_shadow);
	((u_int *)g_my_shadow_offset)[12] = *(u_int *)&(tmpVector1[0]);
	((u_int *)g_my_shadow_offset)[13] = *(u_int *)&(tmpVector1[1]);
	((u_int *)g_my_shadow_offset)[14] = *(u_int *)&(tmpVector1[2]);
	
	fFOV = g_fov * (float)M_PI / 180.0f;
	upLength = viewLength * sinf(fFOV * 0.5f) / cosf(fFOV * 0.5f);
	rightLength = upLength * (float)gp->width / gp->height * gp->pixel_ratio;
	
	ps2_vu0_scale_vector(viewVector, viewVector, 1.0f / viewLength);
	ps2_vu0_scale_vector(upVector, upVector, 1.0f / upLength);
	ps2_vu0_scale_vector(rightVector, rightVector, 1.0f / rightLength);
	
	memset(viewMat, 0, sizeof(ps2_vu0_fmatrix));
	
	viewMat[0][0] = rightVector[0];
	viewMat[1][0] = rightVector[1];
	viewMat[2][0] = rightVector[2];
	
	viewMat[0][1] = upVector[0];
	viewMat[1][1] = upVector[1];
	viewMat[2][1] = upVector[2];
	
	viewMat[0][2] = viewVector[0];
	viewMat[1][2] = viewVector[1];
	viewMat[2][2] = viewVector[2];
	
	viewMat[3][3] = 1.0f;
	
	memset(transMat, 0, sizeof(ps2_vu0_fmatrix));
	transMat[0][0] = 1.0f;
	transMat[1][1] = 1.0f;
	transMat[2][2] = 1.0f;
	transMat[3][3] = 1.0f;
	
	transMat[3][0] = - interest[0];
	transMat[3][1] = - interest[1];
	transMat[3][2] = - interest[2];
	
	ps2_vu0_mul_matrix(projMat, transMat, viewMat);
	
	projMat[0][3] = projMat[0][2]; 
	projMat[1][3] = projMat[1][2]; 
	projMat[2][3] = projMat[2][2]; 
	
	screenMat[0][0] = gp->width / 2.0f;
	screenMat[1][0] = 0.0f;
	screenMat[2][0] = 0.0f;
	screenMat[3][0] = gp->width / 2.0f + gp->offset_x;
	
	screenMat[0][1] = 0.0f;
	screenMat[1][1] = -gp->height / 2.0f;
	screenMat[2][1] = 0.0f;
	screenMat[3][1] = gp->height / 2.0f + gp->offset_y;
	
	screenMat[0][2] = 0.0f;
	screenMat[1][2] = 0.0f;
	screenMat[2][2] = -100000.0f;
	screenMat[3][2] =  100000000.0f;
	
	screenMat[0][3] = 0.0f;
	screenMat[1][3] = 0.0f;
	screenMat[2][3] = 0.0f;
	screenMat[3][3] = 1.0f;
	
	ps2_vu0_mul_matrix(persMat, screenMat, projMat);

	memcpy(g_my_matrix, persMat, sizeof(ps2_vu0_fmatrix));
	memcpy(g_my_shadow_matrix, persMat, sizeof(ps2_vu0_fmatrix));
	memcpy(My_ground_matrix, persMat, sizeof(ps2_vu0_fmatrix));
}

//----------------------------------------------------------------------
void SetViewPosition(ps2_vu0_fvector view, float theta, float phi)
{
#define RADIUS 65.0f
	
	view[0] = RADIUS * cosf(phi) * cosf(theta);
	view[1] = RADIUS * sinf(phi);
	view[2] = RADIUS * cosf(phi) * sinf(theta);
	view[3] = 1.0f;
}

//----------------------------------------------------------------------
void usage(const char *msg1, const char *msg2)
{
	if (msg1) {
		fprintf(stderr, "%s: %s %s\n", g_program_name, msg1, msg2 ? msg2 : "");
	}
	
	fprintf(stderr,
			"Usage: %s [-gas] [<ii>] [<mode>] [<ff>] [<reso>] [<rate>] [<bpp>] [<z-bpp>]\n",
			g_program_name);
	fprintf(stderr,
			"    ii: interlace or noninterlace\n"
			"        -inter, -nointer\n"
			"    mode: video mode\n"
			"        -ntsc, -pal, -dtv, -vesa\n"
			"    ff: field/frame mode\n"
			"        -frame, -field\n"
			"    reso: resolution\n"
			"        -640x480, -800x600, -1024x768\n"
			"    rate: refresh rate\n"
			"        -60, -75\n"
			"    bpp: color bits per pixel\n"
			"        -16, -24, -32\n"
			"    z-bpp: depth bits per pixel\n"
			"        -z0, -z16, -z24\n"
			);
	
	exit(1);
}

//----------------------------------------------------------------------
void options(int argc, char *argv[])
{
	int i;
	
	g_fire = 1;
	
	g_inter = PS2_GS_NOINTERLACE;
	g_out_mode = PS2_GS_VESA;
	g_ff_mode = PS2_GS_FRAME;
	g_resolution = PS2_GS_640x480;
	g_refresh_rate = PS2_GS_60Hz;
	g_psm = PS2_GS_PSMCT32;
	g_zpsm = PS2_GS_PSMZ24;
	g_zbits = 24;
	
	for (i = 1; i < argc; i++) {
		char *p = argv[i];
		
		if (strcmp(p, "-gas") == 0) {
			g_fire = 0;
		} else if (strcmp(p, "-fire") == 0) {
			g_fire = 1;
			
		} else if (strcmp(p, "-inter") == 0) {
			g_inter = PS2_GS_INTERLACE;
		} else if (strcmp(p, "-nointer") == 0) {
			g_inter = PS2_GS_NOINTERLACE;
			
		} else if (strcmp(p, "-ntsc") == 0) {
			g_out_mode = PS2_GS_NTSC;
		} else if (strcmp(p, "-pal") == 0) {
			g_out_mode = PS2_GS_PAL;
		} else if (strcmp(p, "-dtv") == 0) {
			g_out_mode = PS2_GS_DTV;
		} else if (strcmp(p, "-vesa") == 0) {
			g_out_mode = PS2_GS_VESA;
			
		} else if (strcmp(p, "-frame") == 0) {
			g_ff_mode = PS2_GS_FRAME;
		} else if (strcmp(p, "-field") == 0) {
			g_ff_mode = PS2_GS_FIELD;
			
		} else if (strcmp(p, "-640x480") == 0) {
			g_resolution = PS2_GS_640x480;
		} else if (strcmp(p, "-800x600") == 0) {
			g_resolution = PS2_GS_800x600;
		} else if (strcmp(p, "-1024x768") == 0) {
			g_resolution = PS2_GS_1024x768;
		} else if (strcmp(p, "-1280x1024") == 0) {
			g_resolution = PS2_GS_1280x1024;
			
		} else if (strcmp(p, "-60") == 0) {
			g_refresh_rate = PS2_GS_60Hz;
		} else if (strcmp(p, "-75") == 0) {
			g_refresh_rate = PS2_GS_75Hz;
			
		} else if (strcmp(p, "-32") == 0) {
			g_psm = PS2_GS_PSMCT32;
		} else if (strcmp(p, "-24") == 0) {
			g_psm = PS2_GS_PSMCT24;
		} else if (strcmp(p, "-16") == 0) {
			g_psm = PS2_GS_PSMCT16S;
			
		} else if (strcmp(p, "-z24") == 0) {
			g_zpsm = PS2_GS_PSMZ24;
			g_zbits = 24;
		} else if (strcmp(p, "-z16") == 0) {
			g_zpsm = PS2_GS_PSMZ16S;
			g_zbits = 16;
		} else if (strcmp(p, "-z0") == 0) {
			g_zpsm = PS2_GS_PSMZ24;
			g_zbits = 0;
			
		} else {
			usage("illegal optopn", p);
		}
	}
	
	if (g_psm == PS2_GS_PSMCT16S && g_zpsm == PS2_GS_PSMZ16) {
		g_psm = PS2_GS_PSMCT16;
		g_zpsm = PS2_GS_PSMZ16;
	}
}
	
//----------------------------------------------------------------------
int main(int argc, char *argv[])
{
	int odev;
	u_int i, frame;
	float theta, dtheta, phi;
	ps2_vu0_fvector view;
	ps2_vu0_fvector interest;
	ps2_vu0_fmatrix viewmat;
	int paddata = 0;
	VPUFILE *vfd;
	int next2k;
	ps2_gs_gparam *gp = ps2_gs_get_gparam();
	
	g_program_name = argv[0];
	
	options(argc, argv);
	
	g_fd_gs = ps2_gs_open(-1);
	g_vpu0 = ps2_vpu_open(0); // must be opened for vpu1...
	g_vpu1 = ps2_vpu_open(1);
	
	ps2_gs_vc_graphicsmode();
	
	vfd = vpuobj_open("vpu.elf", O_DATA_PS2MEM);
	if (vfd < 0) {
		perror("vpu_open");
		exit(1);
	}
	
	if (g_fire) {
		g_image_size = FIRE_IMAGE_SIZE;
		g_part_size = FIRE_PART_SIZE;
		g_part_size_shadow = FIRE_PART_SIZE_SHADOW;
		g_my_offset = My_offset_fire;
		g_my_shadow_offset = My_shadow_offset_fire;
		g_my_matrix = My_matrix_fire;
		g_my_shadow_matrix = My_shadow_matrix_fire;
		g_my_texture2 = My_texture2_fire;
		g_my_dma_start_part = My_dma_start_part_fire;
		g_my_dma_start_shadow_part = My_dma_start_shadow_part_fire;
	} else {
		g_image_size = GAS_IMAGE_SIZE;
		g_part_size = GAS_PART_SIZE;
		g_part_size_shadow = GAS_PART_SIZE_SHADOW;
		g_my_offset = My_offset_gas;
		g_my_shadow_offset = My_shadow_offset_gas;
		g_my_matrix = My_matrix_gas;
		g_my_shadow_matrix = My_shadow_matrix_gas;
		g_my_texture2 = My_texture2_gas;
		g_my_dma_start_part = My_dma_start_part_gas;
		g_my_dma_start_shadow_part = My_dma_start_shadow_part_gas;
	}
	
	ps2_gs_reset(0, g_inter, g_out_mode, g_ff_mode, g_resolution,
				 g_refresh_rate);
	
	// set double buffer
	next2k = ps2_gs_set_dbuff(&g_db, g_psm, gp->width, gp->height,
							  (g_zbits == 0) ? 0 : PS2_GS_ZGREATER,
							  g_zpsm, 1);
	*(__u64 *)&g_db.clear0.rgbaq = PS2_GS_SETREG_RGBAQ(0x10, 0x10, 0x10, 0x80,
													   0x3f800000);
	*(__u64 *)&g_db.clear1.rgbaq = PS2_GS_SETREG_RGBAQ(0x10, 0x10, 0x10, 0x80,
													   0x3f800000);
	
	// texture buffer
	g_textop64 = next2k * 2048 / 64;
	
	load_teximages();
	
	if (g_fire) {
		// init bound flags
		int n = FIRE_NUM_BLOCK * FIRE_NUM_PART * FIRE_NUM_EXPLODE;
		for (i = 0; i < n; i++) {
			fire_bound[i] = 0;
		}
	}
	
	interest[0] = 0.0f;
	interest[1] = 0.7f;
	interest[2] = 0.0f;
	interest[3] = 0.0f;
	
	// defaults theta
	theta = 0.0f;
	phi = 0.0f;
	dtheta = 2.0f * (float)M_PI / 180.0f;
	
	frame = 0;
	
	if (g_fire) {
		SetParticlePositionFire(frame++);
	} else {
		SetParticlePositionGas(frame++);
	}
	
	sjoy_open();
	
	// clear back buffer
	if (frame & 1) {
		ps2_gs_put_drawenv(&g_db.giftag0);
	} else {
		ps2_gs_put_drawenv(&g_db.giftag1);
	}
	
	// wait clear done
	ps2_gs_set_finish(&g_finish);
	ps2_gs_wait_finish(&g_finish);
	
	odev = !ps2_gs_sync_v(0);
	
	ps2_gs_start_display(1);
	
	ps2_gs_vc_enablevcswitch(acquire, release);
	
	for (; ;) {
		const float pi = (float)M_PI;
		
		ps2_gs_vc_lock();
		
		ps2_gs_set_half_offset((frame & 1) ? &g_db.draw1 : &g_db.draw0, odev);
		ps2_gs_swap_dbuff(&g_db, frame);
		
		// --- read pad ---
		sjoy_poll();
		paddata = sjoy_get_ps2_button(0);
		
		// --- theta direction ---
		if (paddata & SJOY_PS2_L_LEFT) {
			theta += dtheta;
		} else if (paddata & SJOY_PS2_L_RIGHT) {
			theta -= dtheta;
		}
		
		// --- phi direction ---
		if (paddata & SJOY_PS2_L_UP) {
			phi += dtheta;
		} else if (paddata & SJOY_PS2_L_DOWN) {
			phi -= dtheta;
		}
		
		// --- adjust invalid angle ---
		if (theta > 2.0f * pi) {
			theta -= 2.0f * pi;
		} else if (theta < 0.0f) {
			theta += 2.0f * pi;
		}
		
		if (phi >= 0.49f * pi) {
			phi = 0.49f * pi;
		} else if (phi <= -0.49f * pi) {
			phi = -0.49f * pi;
		}
		
		// --- Viewing Matrix ---
		SetViewPosition(view, theta, phi);
		CreateViewingMatrix(view, interest, viewmat);
		
		// ---- draw ground ----
		ps2_gs_set_texenv(&texenv2.gs_tex, 0, &g_img1, 6, 6, 0, 0, 0, 0, 1);
		
		PS2_GIFTAG_CLEAR_TAG(&texenv2.giftag);
		texenv2.giftag.NLOOP = (sizeof(texenv2) - sizeof(ps2_giftag)) / 16;
		texenv2.giftag.EOP = 1;
		texenv2.giftag.NREG = 1;
		texenv2.giftag.REGS0 = PS2_GIFTAG_REGS_AD;
		
		ps2_gs_set_alphaenv(&texenv2.gs_alpha, 0);
		*(__u64 *)&texenv2.gs_alpha.alpha1 = PS2_GS_SETREG_ALPHA(0, 2, 0, 1, 0);
		*(__u64 *)&texenv2.gs_test = PS2_GS_SETREG_TEST(0, 0, 0, 0, 0, 0, 1, PS2_GS_ZALWAYS);
		texenv2.gs_test1addr = PS2_GS_TEST_1;
		ps2_gs_put_drawenv(&texenv2.giftag);
		
		ps2_dma_send(ps2_vpu_fd(g_vpu1), vfd, (ps2_dmatag *)My_dma_start);
		
		// --- particle setup ---
		if (!(paddata & SJOY_PS2_R_RIGHT)) {
			if (g_fire) {
				SetParticlePositionFire(frame);
			} else {
				SetParticlePositionGas(frame);
			}
		}
		
		// ---- DRAW PARTICLE SHADOW ----
		// ==== set texture env ===
		ps2_gs_set_texenv(&texenv1.gs_tex, 0, &g_img3, 5, 5, 0, 0, 0, 0, 1);
		
		PS2_GIFTAG_CLEAR_TAG(&texenv1.giftag);
		texenv1.giftag.NLOOP = (sizeof(texenv2) - sizeof(ps2_giftag)) / 16;
		texenv1.giftag.EOP = 1;
		texenv1.giftag.NREG = 1;
		texenv1.giftag.REGS0 = PS2_GIFTAG_REGS_AD;
		
		ps2_gs_set_alphaenv(&texenv1.gs_alpha, 0);
		
		if (g_fire) {
			*(__u64 *)&texenv1.gs_alpha.alpha1 =
				PS2_GS_SETREG_ALPHA(0, 2, 0, 1, 0);
		} else {
			*(__u64 *)&texenv1.gs_alpha.alpha1 =
				PS2_GS_SETREG_ALPHA(2, 0, 0, 1, 0);
		}
		
		*(__u64 *)&texenv1.gs_test =
			PS2_GS_SETREG_TEST(0, 0, 0, 0, 0, 0, 1, PS2_GS_TEST_ZTST_ALWAYS);
		texenv1.gs_test1addr = PS2_GS_TEST_1;
		ps2_gs_put_drawenv(&texenv1.giftag);
		
		ps2_dma_send(ps2_vpu_fd(g_vpu1), vfd, (ps2_dmatag *)g_my_dma_start_shadow_part);
		
		// ---- DRAW PARTICLE ----
		// ==== set texture env ===
		ps2_gs_set_texenv(&texenv1.gs_tex, 0, &g_img2, 5, 5, 0, 0, 0, 0, 1);
		
		PS2_GIFTAG_CLEAR_TAG(&texenv1.giftag);
		texenv1.giftag.NLOOP = (sizeof(texenv2) - sizeof(ps2_giftag)) / 16;
		texenv1.giftag.EOP = 1;
		texenv1.giftag.NREG = 1;
		texenv1.giftag.REGS0 = PS2_GIFTAG_REGS_AD;
		
		ps2_gs_set_alphaenv(&texenv1.gs_alpha, 0);
		*(__u64 *)&texenv1.gs_alpha.alpha1 =
			PS2_GS_SETREG_ALPHA(0, 2, 0, 1, 0);
		
		*(__u64 *)&texenv1.gs_test =
			PS2_GS_SETREG_TEST(0, 0, 0, 0, 0, 0, 1, PS2_GS_TEST_ZTST_ALWAYS);
		texenv1.gs_test1addr = PS2_GS_TEST_1;
		
		ps2_gs_put_drawenv(&texenv1.giftag);
		
		ps2_dma_send(ps2_vpu_fd(g_vpu1), vfd, (ps2_dmatag *)g_my_dma_start_part);
		
		frame++;
		odev = !ps2_gs_sync_v(0);

		ps2_gs_vc_unlock();
	}
}
