/* **********************************************************
 * Copyright (C) 1998-2000 VMware, Inc.
 * All Rights Reserved
 * $Id: vm_time.h,v 1.3 2003/02/16 15:32:34 bad Exp $
 * **********************************************************/


/*
 * vm_time.h  --
 *
 *    Time management functions.
 *    Part of driver-only distribution
 *
 *    see comment in poll.c
 */


#ifndef VM_TIME_H
#define VM_TIME_H

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_MONITOR
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#include "includeCheck.h"


#include "vm_types.h"

/* TS stands for "timestamp", which is in units of "cycles" */
/* Use these types to express time, Get_TSC should return a
   VmAbsoluteTS -- converting to US results in a VmAbsoluteUS.
   Subtracting two VmAbsoluteTS's results in a VmRelativeTS, likewise
   for VmAbsoluteUS and VmRelativeUS */
/* I use these instead of VmTimeRealClock and VmTimeVirtualClock
   because those types are not used consistently in our code (cycles vs
   us) */
typedef uint64 VmAbsoluteTS; //a particular point in time (in cycles)
typedef int64 VmRelativeTS; //a signed delta in cycles
typedef uint64 VmAbsoluteUS; //a particular point in time (in us)
typedef int64 VmRelativeUS; //a signed delta in us

struct VmTimeVirtualRealClock;
typedef struct VmTimeVirtualRealClock VmTimeVirtualRealClock;

#define VMTIME_VIRTUAL_INFINITE  ((((VmTimeVirtualClock)1)<<62)-1)

#ifdef VMX86_SERVER
/* ESX server uses more accurate khz estimate to reduce time drift with
 * respect to the vmkernel timer interrupts. */
#define CYCLES_TO_USECS(vm, _c)  (((_c)*1000LL) / (vm)->miscShared->khzEstimate)
#define USECS_TO_CYCLES(vm, _us) ((((uint64)(_us)) * (vm)->miscShared->khzEstimate)/1000)
#else
#define CYCLES_TO_USECS(vm, _c)  ((_c) / (vm)->miscShared->mhzEstimate)
#define USECS_TO_CYCLES(vm, _us) ((_us) * (vm)->miscShared->mhzEstimate)
#endif

#ifdef MONITOR
#define GET_REALCYCLES()         (GetTSC())
#define GET_REALTIME_IN_USECS(vm) (CYCLES_TO_USECS(vm, GET_REALCYCLES()))
#else
#define GET_REALCYCLES()         GetTSC()
#define GET_REALTIME_IN_USECS(vm) (Hostinfo_FastRealTime() / \
				  (Hostinfo_FastRealTimeFrequency() / 1000000))
#endif

#ifdef USERLEVEL
struct VM;
extern VmTimeType VmTime_ReadVirtualTime(void);
extern void VmTime_IncVirtualTime(struct VM *vm, VmTimeType amount);
extern void VmTime_IncVirtualTimeInCycles(VmTimeVirtualClock amount);
extern VmTimeVirtualRealClock *VmTime_NewVirtualRealClock(struct VM *);
extern void VmTime_StartVirtualRealClock(VmTimeVirtualRealClock *, double);
extern void VmTime_ResetVirtualRealClock(VmTimeVirtualRealClock *);
extern VmTimeType VmTime_ReadVirtualRealTime(VmTimeVirtualRealClock *);
extern VmTimeType VmTime_RemainingVirtualRealTime(VmTimeVirtualRealClock *,
						  VmTimeType realTime);
extern void VmTime_UpdateVirtualRealTime(VmTimeVirtualRealClock *clock,
                                         VmTimeType realTime,
                                         VmTimeType virtualTime);
#endif
#endif /* VM_TIME_H */

