"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateCustomError = generateCustomError;
exports.getConnectorsFromResponses = getConnectorsFromResponses;
exports.getModelsFromResponses = getModelsFromResponses;
exports.getResourcesCreatedFromResponse = getResourcesCreatedFromResponse;
exports.getWorkflowStateFromResponse = getWorkflowStateFromResponse;
exports.getWorkflowsFromResponses = getWorkflowsFromResponses;
exports.isIgnorableError = isIgnorableError;
exports.toWorkflowObj = toWorkflowObj;
var _common = require("../../common");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// OSD does not provide an interface for this response, but this is following the suggested
// implementations. To prevent typescript complaining, leaving as loosely-typed 'any'
function generateCustomError(res, err) {
  var _err$body;
  if (isDatasourceError(err)) {
    return res.customError({
      statusCode: 404,
      body: {
        message: 'Data source not found'
      }
    });
  }
  return res.customError({
    statusCode: err.statusCode || 500,
    body: {
      message: err.message,
      attributes: {
        error: ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.error) || err.message
      }
    }
  });
}
function isDatasourceError(err) {
  if (err.message !== undefined && typeof err.message === 'string') {
    return err.message.includes(_common.INVALID_DATASOURCE_MSG);
  }
}

// Helper fn to filter out backend errors that we don't want to propagate on the frontend.
function isIgnorableError(error) {
  var _error$body, _error$body2;
  return ((_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type) === _common.INDEX_NOT_FOUND_EXCEPTION || ((_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : _error$body2.error) === _common.NO_MODIFICATIONS_FOUND_TEXT;
}

// Convert backend workflow into frontend workflow obj
function toWorkflowObj(hitSource, id) {
  return {
    id,
    name: hitSource.name,
    use_case: hitSource.use_case,
    description: hitSource.description || '',
    version: hitSource.version,
    workflows: hitSource.workflows,
    ui_metadata: hitSource.ui_metadata,
    lastUpdated: hitSource.last_updated_time,
    lastLaunched: hitSource.last_provisioned_time
  };
}

// TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
// https://github.com/opensearch-project/flow-framework/issues/171
// Current implementation combines 2 search responses to create a single set of workflows with
// static information + state information
function getWorkflowsFromResponses(workflowHits, workflowStateHits) {
  const workflowDict = {};
  workflowHits.forEach(workflowHit => {
    var _workflowStateHit$_so, _workflowStateHit$_so2, _workflowStateHit$_so3;
    const hitSource = workflowHit._source;
    workflowDict[workflowHit._id] = toWorkflowObj(hitSource, workflowHit._id);
    const workflowStateHit = workflowStateHits.find(workflowStateHit => workflowStateHit._id === workflowHit._id);
    const workflowState = getWorkflowStateFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so = workflowStateHit._source) === null || _workflowStateHit$_so === void 0 ? void 0 : _workflowStateHit$_so.state);
    const workflowError = workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so2 = workflowStateHit._source) === null || _workflowStateHit$_so2 === void 0 ? void 0 : _workflowStateHit$_so2.error;
    const workflowResourcesCreated = getResourcesCreatedFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so3 = workflowStateHit._source) === null || _workflowStateHit$_so3 === void 0 ? void 0 : _workflowStateHit$_so3.resources_created);
    workflowDict[workflowHit._id] = {
      ...workflowDict[workflowHit._id],
      // @ts-ignore
      state: workflowState,
      error: workflowError,
      resourcesCreated: workflowResourcesCreated
    };
  });
  return workflowDict;
}
function getModelsFromResponses(modelHits) {
  const modelDict = {};
  modelHits.forEach(modelHit => {
    // search model API returns hits for each deployed model chunk. ignore these hits
    if (modelHit._source.chunk_number === undefined) {
      var _modelHit$_source, _modelHit$_source2, _modelHit$_source3, _modelHit$_source4, _modelHit$_source5, _modelHit$_source6;
      const modelId = modelHit._id;

      // the persisted model interface (if available) is a mix of an obj and string.
      // We parse the string values for input/output to have a complete
      // end-to-end JSONSchema obj
      let indexedModelInterface = modelHit._source.interface;
      let modelInterface = undefined;
      if (indexedModelInterface !== undefined) {
        let parsedInput = undefined;
        let parsedOutput = undefined;
        try {
          parsedInput = JSON.parse(indexedModelInterface.input);
        } catch {}
        try {
          parsedOutput = JSON.parse(indexedModelInterface.output);
        } catch {}
        modelInterface = {
          input: parsedInput,
          output: parsedOutput
        };
      }

      // in case of schema changes from ML plugin, this may crash. That is ok, as the error
      // produced will help expose the root cause
      modelDict[modelId] = {
        id: modelId,
        name: (_modelHit$_source = modelHit._source) === null || _modelHit$_source === void 0 ? void 0 : _modelHit$_source.name,
        // @ts-ignore
        algorithm: _common.MODEL_ALGORITHM[(_modelHit$_source2 = modelHit._source) === null || _modelHit$_source2 === void 0 ? void 0 : _modelHit$_source2.algorithm],
        // @ts-ignore
        state: _common.MODEL_STATE[(_modelHit$_source3 = modelHit._source) === null || _modelHit$_source3 === void 0 ? void 0 : _modelHit$_source3.model_state],
        modelConfig: {
          modelType: (_modelHit$_source4 = modelHit._source) === null || _modelHit$_source4 === void 0 || (_modelHit$_source4 = _modelHit$_source4.model_config) === null || _modelHit$_source4 === void 0 ? void 0 : _modelHit$_source4.model_type,
          embeddingDimension: (_modelHit$_source5 = modelHit._source) === null || _modelHit$_source5 === void 0 || (_modelHit$_source5 = _modelHit$_source5.model_config) === null || _modelHit$_source5 === void 0 ? void 0 : _modelHit$_source5.embedding_dimension
        },
        interface: modelInterface,
        connectorId: (_modelHit$_source6 = modelHit._source) === null || _modelHit$_source6 === void 0 ? void 0 : _modelHit$_source6.connector_id
      };
    }
  });
  return modelDict;
}
function getConnectorsFromResponses(modelHits) {
  const connectorDict = {};
  modelHits.forEach(connectorHit => {
    var _connectorHit$_source, _connectorHit$_source2, _connectorHit$_source3;
    const connectorId = connectorHit._id;

    // in case of schema changes from ML plugin, this may crash. That is ok, as the error
    // produced will help expose the root cause
    connectorDict[connectorId] = {
      id: connectorId,
      name: (_connectorHit$_source = connectorHit._source) === null || _connectorHit$_source === void 0 ? void 0 : _connectorHit$_source.name,
      parameters: {
        model: (_connectorHit$_source2 = connectorHit._source) === null || _connectorHit$_source2 === void 0 || (_connectorHit$_source2 = _connectorHit$_source2.parameters) === null || _connectorHit$_source2 === void 0 ? void 0 : _connectorHit$_source2.model,
        dimensions: (_connectorHit$_source3 = connectorHit._source) === null || _connectorHit$_source3 === void 0 ? void 0 : _connectorHit$_source3.parameters.dimensions
      }
    };
  });
  return connectorDict;
}

// Convert the workflow state into a readable/presentable state on frontend
function getWorkflowStateFromResponse(state) {
  const finalState = state || _common.DEFAULT_NEW_WORKFLOW_STATE_TYPE;
  // @ts-ignore
  return _common.WORKFLOW_STATE[finalState];
}

// Convert the workflow resources into a readable/presentable state on frontend
function getResourcesCreatedFromResponse(resourcesCreated) {
  const finalResources = [];
  if (resourcesCreated) {
    resourcesCreated.forEach(backendResource => {
      finalResources.push({
        id: backendResource.resource_id,
        stepType: backendResource.workflow_step_name,
        type:
        // @ts-ignore
        _common.WORKFLOW_RESOURCE_TYPE[
        // the backend persists the types in lowercase. e.g., "pipeline_id"
        backendResource.resource_type.toUpperCase()]
      });
    });
  }
  return finalResources;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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