/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.plugin.insights.core.exporter;

import java.io.IOException;
import java.nio.charset.Charset;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Objects;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.ExceptionsHelper;
import org.opensearch.ResourceAlreadyExistsException;
import org.opensearch.action.admin.indices.create.CreateIndexRequest;
import org.opensearch.action.admin.indices.create.CreateIndexResponse;
import org.opensearch.action.admin.indices.delete.DeleteIndexRequest;
import org.opensearch.action.bulk.BulkRequestBuilder;
import org.opensearch.action.bulk.BulkResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.support.master.AcknowledgedResponse;
import org.opensearch.client.Client;
import org.opensearch.cluster.ClusterState;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.xcontent.XContentFactory;
import org.opensearch.core.action.ActionListener;
import org.opensearch.core.xcontent.ToXContent;
import org.opensearch.index.IndexNotFoundException;
import org.opensearch.plugin.insights.core.exporter.QueryInsightsExporter;
import org.opensearch.plugin.insights.core.metrics.OperationalMetric;
import org.opensearch.plugin.insights.core.metrics.OperationalMetricsCounter;
import org.opensearch.plugin.insights.core.utils.ExporterReaderUtils;
import org.opensearch.plugin.insights.rules.model.SearchQueryRecord;

public class LocalIndexExporter
implements QueryInsightsExporter {
    private final Logger logger = LogManager.getLogger();
    private final Client client;
    private final ClusterService clusterService;
    private final String indexMapping;
    private DateTimeFormatter indexPattern;
    private int deleteAfter;
    private final String id;
    private static final int DEFAULT_NUMBER_OF_REPLICA = 1;
    private static final int DEFAULT_NUMBER_OF_SHARDS = 1;
    private static final List<String> DEFAULT_SORTED_FIELDS = List.of("measurements.latency.number", "measurements.cpu.number", "measurements.memory.number");
    private static final List<String> DEFAULT_SORTED_ORDERS = List.of("desc", "desc", "desc");

    public LocalIndexExporter(Client client, ClusterService clusterService, DateTimeFormatter indexPattern, String indexMapping, String id) {
        this.indexPattern = indexPattern;
        this.client = client;
        this.clusterService = clusterService;
        this.indexMapping = indexMapping;
        this.deleteAfter = 7;
        this.id = id;
    }

    @Override
    public String getId() {
        return this.id;
    }

    public DateTimeFormatter getIndexPattern() {
        return this.indexPattern;
    }

    public void setIndexPattern(DateTimeFormatter indexPattern) {
        this.indexPattern = indexPattern;
    }

    @Override
    public void export(final List<SearchQueryRecord> records) {
        if (records == null || records.isEmpty()) {
            return;
        }
        try {
            final String indexName = this.buildLocalIndexName();
            if (!this.checkIndexExists(indexName)) {
                CreateIndexRequest createIndexRequest = new CreateIndexRequest(indexName);
                createIndexRequest.settings(Settings.builder().putList("index.sort.field", DEFAULT_SORTED_FIELDS).putList("index.sort.order", DEFAULT_SORTED_ORDERS).put("index.number_of_shards", 1).put("index.number_of_replicas", 1));
                createIndexRequest.mapping(this.readIndexMappings());
                this.client.admin().indices().create(createIndexRequest, (ActionListener)new ActionListener<CreateIndexResponse>(){

                    public void onResponse(CreateIndexResponse createIndexResponse) {
                        if (createIndexResponse.isAcknowledged()) {
                            try {
                                LocalIndexExporter.this.bulk(indexName, records);
                            }
                            catch (IOException e) {
                                OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                                LocalIndexExporter.this.logger.error("Unable to index query insights data: ", (Throwable)e);
                            }
                        }
                    }

                    public void onFailure(Exception e) {
                        Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                        if (cause instanceof ResourceAlreadyExistsException) {
                            try {
                                LocalIndexExporter.this.bulk(indexName, records);
                            }
                            catch (IOException ex) {
                                OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                                LocalIndexExporter.this.logger.error("Unable to index query insights data: ", (Throwable)e);
                            }
                        } else {
                            OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                            LocalIndexExporter.this.logger.error("Unable to create query insights index: ", (Throwable)e);
                        }
                    }
                });
            } else {
                this.bulk(indexName, records);
            }
        }
        catch (Exception e) {
            OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
            this.logger.error("Unable to index query insights data: ", (Throwable)e);
        }
    }

    private void bulk(String indexName, List<SearchQueryRecord> records) throws IOException {
        BulkRequestBuilder bulkRequestBuilder = this.client.prepareBulk().setTimeout(TimeValue.timeValueMinutes((long)1L));
        for (SearchQueryRecord record : records) {
            bulkRequestBuilder.add(new IndexRequest(indexName).id(record.getId()).source(record.toXContent(XContentFactory.jsonBuilder(), ToXContent.EMPTY_PARAMS)));
        }
        bulkRequestBuilder.execute((ActionListener)new ActionListener<BulkResponse>(){

            public void onResponse(BulkResponse bulkItemResponses) {
            }

            public void onFailure(Exception e) {
                OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_BULK_FAILURES);
                LocalIndexExporter.this.logger.error("Failed to execute bulk operation for query insights data: ", (Throwable)e);
            }
        });
    }

    @Override
    public void close() {
        this.logger.debug("Closing the LocalIndexExporter..");
    }

    String buildLocalIndexName() {
        ZonedDateTime currentTime = ZonedDateTime.now(ZoneOffset.UTC);
        return this.indexPattern.format(currentTime) + "-" + ExporterReaderUtils.generateLocalIndexDateHash(currentTime.toLocalDate());
    }

    public void setDeleteAfter(int deleteAfter) {
        this.deleteAfter = deleteAfter;
    }

    public int getDeleteAfter() {
        return this.deleteAfter;
    }

    public void deleteSingleIndex(final String indexName, Client client) {
        final Logger logger = LogManager.getLogger();
        client.admin().indices().delete(new DeleteIndexRequest(indexName), (ActionListener)new ActionListener<AcknowledgedResponse>(){

            public void onResponse(AcknowledgedResponse acknowledgedResponse) {
            }

            public void onFailure(Exception e) {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                if (cause instanceof IndexNotFoundException) {
                    return;
                }
                OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_DELETE_FAILURES);
                logger.error("Failed to delete index '{}': ", (Object)indexName, (Object)e);
            }
        });
    }

    private boolean checkIndexExists(String indexName) {
        ClusterState clusterState = this.clusterService.state();
        return clusterState.getRoutingTable().hasIndex(indexName);
    }

    private String readIndexMappings() throws IOException {
        return new String(Objects.requireNonNull(LocalIndexExporter.class.getClassLoader().getResourceAsStream(this.indexMapping)).readAllBytes(), Charset.defaultCharset());
    }
}

