/*
 *  playlist-mp4.c
 *  mod_musicindex
 *
 *  $Id: playlist-mp4.c 754 2007-04-29 00:57:53Z varenet $
 *
 *  Created by Thibaut VARENE on Tue Sep  2 2004.
 *  Copyright (c) 2004-2005,2007 Thibaut VARENE
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 2.1,
 *  as published by the Free Software Foundation.
 *
 */

/**
 * @file
 * MP4 files management system.
 *
 * @author Thibaut Varene
 * @version $Revision: 754 $
 * @date 2004-2005
 * @date 2007
 *
 * This file contains everything needed to produce music entries from
 * MP4 files.
 *
 * @warning I'm not actually sure raw AAC files are properly handled.
 */

#include "playlist.h"
#include "playlist-mp4.h"

#include <mp4.h>        /* libmp4v2 */

/**
 * Checks for valid MP4 filename extension.
 *
 * @param filename The filename to check.
 * @todo see if we can just rely on the trackType test.
 *
 * @return TRUE when the extension is correct, FALSE otherwise.
 */
static inline short mp4_ext_check(const char *const filename)
{
	const char *const ext = strrchr(filename, '.');
	if(ext && (!strncasecmp(ext, ".mp4", 4)	|| /* official extention */
		!strncasecmp(ext, ".m4a", 4)	|| /* Apple mp4 file */
		!strncasecmp(ext, ".aac", 4))	)  /* old MPEG2/4 extention */
		return TRUE;
	return FALSE;
}

/**
 * Fills in the information fields about MP4 data.
 *
 * This function reads the mp4 metadata (using libmp4) from the MP4 file
 * and fills in the struct mu_ent fields accordingly.
 * The code is based on sample files from mpeg4ip.
 *
 * @param r Apache request_rec struct to handle log writings (debugging)
 * @param pool Pool
 * @param in MP4 file to parse (closed on normal exit)
 * @param conf MusicIndex configuration paramaters struct
 * @param names Names
 *
 * @return When possible, struct mu_ent correctly set up, file stream closed.
 *
 * @todo handle old AAC files that use id3 tags.
 */
mu_ent *make_mp4_entry(request_rec *r, apr_pool_t *pool, FILE *const in,
	const mu_config *const conf, mu_ent_names *const names)
{
	mu_ent 			*p = NULL;

	struct stat		filestat;
	MP4FileHandle		mp4file;
	char *			temp = NULL;
	uint32_t		numTracks;
	uint16_t		tmpi1, tmpi2; 
	register unsigned short i;
	
	/* XXX We could probably rely on the trackType test only though */
	if (!mp4_ext_check(names->filename))
		goto exit;

	/* XXX va savoir ce qui pourrait faire foirer le read, je me souviens plus
	des heritages en C++ ca me gave. Si une sous fonction chie, la fonction parente chie? */
	if (!(mp4file = MP4Read(names->filename, 0)))
		goto exit;
	
	numTracks = MP4GetNumberOfTracks(mp4file, NULL, 0);
	
	for (i=0; i<numTracks; i++) {
		MP4TrackId trackID = MP4FindTrackId(mp4file, i, NULL, 0);
		const char *trackType = MP4GetTrackType(mp4file, trackID);
	
		if (!strcmp(trackType, MP4_AUDIO_TRACK_TYPE)) {	/* We have an Audio Track */
			p = NEW_ENT(pool);
			if (p == NULL) {
				MP4Close(mp4file);
				return NULL;
			}
				
			p->filetype = FT_MP4;
			p->flags |= EF_VBR;
		
			fstat(fileno(in), &filestat);
			p->size = filestat.st_size;
			p->mtime = filestat.st_mtime;
			
			fclose(in);	/* we won't use the file stream as provided anymore */
			
			if (conf->options & (MI_QUICKPL))
				p->bitrate = p->length = p->freq = 0;
			else {
				unsigned short timeScale = MP4GetTrackTimeScale(mp4file, trackID); /* Hz */
				MP4Duration trackDuration = MP4GetTrackDuration(mp4file, trackID);
				unsigned long msDuration = MP4ConvertFromTrackDuration(mp4file, trackID, 
							trackDuration, MP4_MSECS_TIME_SCALE) / 1000;	/* s */	
				unsigned long avgBitRate = ( MP4GetTrackBitRate(mp4file, trackID) + 500 ) / 1000;	/* kbps */
					
				p->length = msDuration;
				p->bitrate = avgBitRate << 10; 
				p->freq = timeScale;
			}
		
			if (MP4GetMetadataArtist(mp4file, &temp)) {
				p->artist = ap_pstrdup(pool, temp);
				free(temp);
			}
			
			if (MP4GetMetadataName(mp4file, &temp)) {
				p->title = ap_pstrdup(pool, temp);
				free(temp);
			}
			
			if (MP4GetMetadataAlbum(mp4file, &temp)) {
				p->album = ap_pstrdup(pool, temp);
				free(temp);
			}
			
			if (MP4GetMetadataYear(mp4file, &temp)) {
				p->date = atoi(temp);
				free(temp);
			}
		
			if (MP4GetMetadataGenre(mp4file, &temp)) {
				p->genre = ap_pstrdup(pool, temp);
				free(temp);
			}
		
			if (MP4GetMetadataTrack(mp4file, &tmpi1, &tmpi2))
				p->track = tmpi1;
		
			if (MP4GetMetadataDisk(mp4file, &tmpi1, &tmpi2))
				p->posn = tmpi1;
		
			break;	/* Get out of the tracks loop */
		}
	}
	
	MP4Close(mp4file);

exit:
	return p;
}
