/**
 * ================================================
 * LibLoader : a free Java resource loading library
 * ================================================
 *
 * Project Info:  http://reporting.pentaho.org/libloader/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * EHResourceBundleDataCache.java
 * ------------
 */
package org.pentaho.reporting.libraries.resourceloader.modules.cache.ehcache;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.Element;
import org.pentaho.reporting.libraries.resourceloader.ResourceBundleData;
import org.pentaho.reporting.libraries.resourceloader.ResourceKey;
import org.pentaho.reporting.libraries.resourceloader.ResourceLoadingException;
import org.pentaho.reporting.libraries.resourceloader.ResourceManager;
import org.pentaho.reporting.libraries.resourceloader.cache.CachingResourceBundleData;
import org.pentaho.reporting.libraries.resourceloader.cache.DefaultResourceBundleDataCacheEntry;
import org.pentaho.reporting.libraries.resourceloader.cache.ResourceBundleDataCache;
import org.pentaho.reporting.libraries.resourceloader.cache.ResourceBundleDataCacheEntry;

/**
 * Creation-Date: 13.04.2006, 16:30:34
 *
 * @author Thomas Morgner
 */
public class EHResourceBundleDataCache implements ResourceBundleDataCache
{
  private Cache dataCache;

  public EHResourceBundleDataCache(final Cache dataCache)
  {
    if (dataCache == null)
    {
      throw new NullPointerException();
    }
    this.dataCache = dataCache;
  }

  /**
   * Retrieves the given data from the cache.
   *
   * @param key the resource key for the data.
   */
  public ResourceBundleDataCacheEntry get(final ResourceKey key)
  {
    if (key == null)
    {
      throw new NullPointerException();
    }

    try
    {
      final Element element = dataCache.get(key);
      return element != null ? (ResourceBundleDataCacheEntry) element.getObjectValue() : null;
    }
    catch (CacheException e)
    {
      return null;
    }
  }

  public ResourceBundleData put(final ResourceManager caller, final ResourceBundleData data)
      throws ResourceLoadingException
  {
    if (caller == null)
    {
      throw new NullPointerException();
    }
    if (data == null)
    {
      throw new NullPointerException();
    }

    final ResourceBundleData cdata = CachingResourceBundleData.createCached(data);
    dataCache.put(new Element(data.getKey(), new DefaultResourceBundleDataCacheEntry(cdata, caller)));
    return cdata;
  }

  public void remove(final ResourceBundleData data)
  {
    if (data == null)
    {
      throw new NullPointerException();
    }

    dataCache.remove(data.getKey());
  }

  /**
   * Remove all cached entries. This should be called after the cache has become
   * invalid or after it has been removed from a resource manager.
   */
  public void clear()
  {
    try
    {
      dataCache.removeAll();
    }
    catch (Exception e)
    {
      // ignore it ..
    }
  }

  public void shutdown()
  {
    try
    {
      dataCache.getCacheManager().shutdown();
    }
    catch (Exception e)
    {
      // ignore it ..
    }
  }
}