/**
 * ========================================
 * JFreeReport : a free Java report library
 * ========================================
 *
 * Project Info:  http://www.object-refinery.com/jfreereport/index.html
 * Project Lead:  Thomas Morgner (taquera@sherito.org);
 *
 * (C) Copyright 2000-2003, by Simba Management Limited and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * ------------------------------
 * EventOrderFunction.java
 * ------------------------------
 * (C)opyright 2003, by Thomas Morgner and Contributors.
 *
 * Original Author:  Thomas Morgner;
 * Contributor(s):   David Gilbert (for Simba Management Limited);
 *
 * $Id: EventOrderFunction.java,v 1.1 2003/06/13 22:58:25 taqua Exp $
 *
 * Changes 
 * -------------------------
 * 12.06.2003 : Initial version
 *  
 */

package com.jrefinery.report.ext.junit.base.functionality;

import com.jrefinery.report.function.AbstractFunction;
import com.jrefinery.report.event.ReportEvent;
import com.jrefinery.report.util.Log;

public class EventOrderFunction extends AbstractFunction
{
  private int lastEventType;
  private boolean pageOpen;

  public EventOrderFunction()
  {
  }

  public EventOrderFunction(String name)
  {
    setName(name);
  }


  /**
   * Receives notification that report generation initializes the current run.
   * <P>
   * The event carries a ReportState.Started state.  Use this to initialize the report.
   *
   * @param event The event.
   */
  public void reportInitialized(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.REPORT_INITIALIZED) != ReportEvent.REPORT_INITIALIZED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type REPORT_INITIALIZED");
    }

    if (lastEventType != ReportEvent.REPORT_DONE && lastEventType != 0)
    {
      throw new IllegalStateException("Unexpected Event: ReportInitialized: " + lastEventType);
    }

    lastEventType = ReportEvent.REPORT_INITIALIZED;
  }

  /**
   * Receives notification that the report has started.
   *
   * @param event  the event.
   */
  public void reportStarted(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.REPORT_STARTED) != ReportEvent.REPORT_STARTED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type REPORT_STARTED");
    }

    if (lastEventType != ReportEvent.REPORT_INITIALIZED)
    {
      throw new IllegalStateException("Unexpected Event: ReportStarted: " + lastEventType);
    }

    lastEventType = ReportEvent.REPORT_STARTED;
  }

  /**
   * Receives notification that the report has finished.
   *
   * @param event  the event.
   */
  public void reportFinished(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.REPORT_FINISHED) != ReportEvent.REPORT_FINISHED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type REPORT_FINISHED");
    }

    if (lastEventType != ReportEvent.GROUP_FINISHED)
    {
      throw new IllegalStateException("Unexpected Event: ReportFinished: " + lastEventType);
    }

    lastEventType = ReportEvent.REPORT_FINISHED;
  }

  /**
   * Receives notification that a page has started.
   *
   * @param event  the event.
   */
  public void pageStarted(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.PAGE_STARTED) != ReportEvent.PAGE_STARTED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type PAGE_STARTED");
    }

    if (pageOpen)
    {
      throw new IllegalStateException("Unexpected Event: PageStarted: " + lastEventType);
    }
    pageOpen = true;
  }

  /**
   * Receives notification that a page has ended.
   *
   * @param event  the event.
   */
  public void pageFinished(ReportEvent event)
  {
    Log.error ("! EventOrderFunction: Page Finished called !");
    if ((event.getType() & ReportEvent.PAGE_FINISHED) != ReportEvent.PAGE_FINISHED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type PAGE_FINISHED: " +
          event.getType());
    }

    if (pageOpen == false)
    {
      throw new IllegalStateException("Unexpected Event: PageFinished: " + lastEventType);
    }
    pageOpen = false;
  }

  /**
   * Receives notification that a group has started.
   *
   * @param event  the event.
   */
  public void groupStarted(ReportEvent event)
  {
    Log.error ("! EventOrderFunction: Group Started called !");
    if ((event.getType() & ReportEvent.GROUP_STARTED) != ReportEvent.GROUP_STARTED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type GROUP_STARTED");
    }
    Log.error ("! EventOrderFunction: Group Started called !");

    if (lastEventType != ReportEvent.GROUP_STARTED && lastEventType != ReportEvent.REPORT_STARTED)
    {
      Log.error (" ++! EventOrderFunction: Group Started called !", new Exception());
      throw new IllegalStateException("Unexpected Event: GroupStarted: " + lastEventType);
    }
    Log.error ("! EventOrderFunction: Group Started called !");

    lastEventType = ReportEvent.GROUP_STARTED;
    Log.error ("! EventOrderFunction: Group Started called !");
  }

  /**
   * Receives notification that a group has finished.
   *
   * @param event  the event.
   */
  public void groupFinished(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.GROUP_FINISHED) != ReportEvent.GROUP_FINISHED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type GROUP_FINISHED");
    }

    if (lastEventType != ReportEvent.GROUP_FINISHED && lastEventType != ReportEvent.ITEMS_FINISHED)
    {
      throw new IllegalStateException("Unexpected Event: GroupFinished: " + lastEventType);
    }

    lastEventType = ReportEvent.GROUP_FINISHED;
  }

  /**
   * Receives notification that a row of data is being processed.
   *
   * @param event  the event.
   */
  public void itemsAdvanced(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.ITEMS_ADVANCED) != ReportEvent.ITEMS_ADVANCED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type ITEMS_ADVANCED");
    }

    if (lastEventType != ReportEvent.ITEMS_STARTED && lastEventType != ReportEvent.ITEMS_ADVANCED)
    {
      throw new IllegalStateException("Unexpected Event: ReportDone: " + lastEventType);
    }

    lastEventType = ReportEvent.ITEMS_ADVANCED;

  }

  /**
   * Receives notification that a group of item bands is about to be processed.
   * <P>
   * The next events will be itemsAdvanced events until the itemsFinished event is raised.
   *
   * @param event The event.
   */
  public void itemsStarted(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.ITEMS_STARTED) != ReportEvent.ITEMS_STARTED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type ITEMS_STARTED");
    }

    if (lastEventType != ReportEvent.GROUP_STARTED)
    {
      throw new IllegalStateException("Unexpected Event: ItemsStarted: " + lastEventType);
    }

    lastEventType = ReportEvent.ITEMS_STARTED;

  }

  /**
   * Receives notification that a group of item bands has been completed.
   * <P>
   * The itemBand is finished, the report starts to close open groups.
   *
   * @param event The event.
   */
  public void itemsFinished(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.ITEMS_FINISHED) != ReportEvent.ITEMS_FINISHED)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type ITEMS_FINISHED");
    }

    if (lastEventType != ReportEvent.ITEMS_ADVANCED)
    {
      throw new IllegalStateException("Unexpected Event: ItemsFinished: " + lastEventType);
    }

    lastEventType = ReportEvent.ITEMS_FINISHED;
  }

  /**
   * Receives notification that report generation has completed, the report footer was printed,
   * no more output is done. This is a helper event to shut down the output service.
   *
   * @param event The event.
   */
  public void reportDone(ReportEvent event)
  {
    if ((event.getType() & ReportEvent.REPORT_DONE) != ReportEvent.REPORT_DONE)
    {
      throw new IllegalArgumentException("ReportEvent was expected to be of type REPORT_DONE");
    }

    if (lastEventType != ReportEvent.REPORT_FINISHED)
    {
      throw new IllegalStateException("Unexpected Event: ReportDone: " + lastEventType);
    }

    lastEventType = ReportEvent.REPORT_DONE;
  }

  /**
   * Return the current expression value.
   * <P>
   * The value depends (obviously) on the expression implementation.
   *
   * @return the value of the function.
   */
  public Object getValue()
  {
    return new Integer(lastEventType);
  }
}
