/*
 * snes9express
 * rom.cc
 * Copyright (C) 1998-2004  David Nordlund
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * For further details, please read the included COPYING file,
 * or go to http://www.gnu.org/copyleft/gpl.html
 */

#include <cstdio>
#include <cstring>
#include <sstream>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "rom.h"

#ifdef HAVE_ZLIB_H
# include <zlib.h>
#endif

// characters which may be in the game title in the rom file
static const char*ROMcharacters =
  "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
  "0123456789 !#$&*()_+-/,.?:"
  "abcdefghijklmnopqrstuvwxyz";

s9x_ROM::s9x_ROM(fr_Notebook*parent):
s9x_Notepage(parent, "ROM"),
RomBtn(this, new fr_Label(this, "ROM Selector")),
File(this, Name, S9X_ROMENV, S9X_ROMDIR, false),
MemoryMap(this, "Memory Map"),
Format(this, "Format"),
Bnr(0),
Selector(this)
{
   fr_MenuItem*MI;

   SetGridSize(3, 3, true);
   //SetPadding(2, 6);
   //SetStretch(Normal, Fill);

   Pack(File, 3, 1);
   
   RomBtn.SetTooltip("ROM Selector");
   RomBtn.AddListener(this);
   Pack(RomBtn, 3, 1);

   MI = new fr_MenuItem(&MemoryMap, "auto");
   MI = new fr_MenuItem(&MemoryMap, "Force Lo-ROM");
   MI->Args << fr_CaseInsensitive << "-fl" << "-lorom" << "-lr";
   MI->SetTooltip("Force Lo-ROM memory map");

   MI = new fr_MenuItem(&MemoryMap, "Force Hi-ROM");
   MI->Args << fr_CaseInsensitive << "-fh" << "-hirom" << "-hr";
   MI->SetTooltip("Force Hi-ROM memory map");

   Pack(MemoryMap, 1, 2, 2, 3);

   MI = new fr_MenuItem(&Format, "auto");
   MI = new fr_MenuItem(&Format, "Interleaved");
   MI->Args << fr_CaseInsensitive << "-i" << "-interleaved";
   MI->SetTooltip("Interleaved format");

   MI = new fr_MenuItem(&Format, "Interleaved 2");
   MI->Args << fr_CaseInsensitive << "-i2" << "-interleaved2";
   MI->SetTooltip("Interleaved 2 format.  This format can't be auto-deteced.  Many Super-FX games use this format.");

   Pack(Format, 2, 2, 3, 3);
   addOption(MemoryMap);
   addOption(Format);
   addOption(File);

   Selector.AddListener(this);
   AddListener(this);
}

void s9x_ROM::Set9xVersion(float version)
{
}

void s9x_ROM::SiftArgs(fr_ArgList& L)
{
   MemoryMap.SiftArgs(L);
   Format.SiftArgs(L);

   for(int a=L.CountArgs(); a--;)
   {
      const char*A = L[a];
      if(L.IsMarked(a))
	continue;
      if(Selector.hasROMdata(Selector.getFileID(A)) || fr_Exists(A))
      {
         SetFileName(A);
         L.Mark(a);
	 break;
      }
   }
}

void s9x_ROM::SetFileName(const std::string& file)
{
   File.SetFileName(file);
}

std::string s9x_ROM::GetFileName()
{
   return File.GetFileName();
}

void s9x_ROM::FilePopup()
{
   File.FilePopup();
}

void s9x_ROM::ApplySkin(s9x_Skin*S)
{
  s9x_Notepage::ApplySkin(S);
  s9x_SkinSection* section = S?S->getSection("icons"):NULL;
  fr_Image* label = section?section->getImage("label"):NULL;
  if(Bnr)
  {
    Remove(*Bnr);
    delete Bnr;
    Bnr = (fr_Element*)0;
  }
  if(label)
  {
    Bnr = new fr_Element(this, *label);
    Pack(*Bnr, 0, 2, 1, 3);
  }
}

void s9x_ROM::EventOccurred(fr_Event*e) {
   if(e->Is(RomBtn, fr_Click) || e->Is(this, fr_MenuClick))
   {
      Selector.SetVisibility(true);
   } else if(e->Is(Selector, fr_Destroy)) {
      Selector.SetVisibility(false);
   }
}

/* ############################# s9x_ROMdata ############################## */

s9x_ROMdata::s9x_ROMdata(const std::string& file_ID):
fileid(file_ID)
{
}

s9x_ROMdata::~s9x_ROMdata()
{
}

void s9x_ROMdata::setData(const std::string& key, time_t val)
{
  std::ostringstream s;
  s << val;
  setData(key, s.str());
}

std::string s9x_ROMdata::getData(const std::string& key, const std::string& fallback)
{
  strmap::iterator i = data.find(key), e = data.end();
  if(i==e)
    return fallback;
  return i->second;
}

time_t s9x_ROMdata::getData(const std::string& key, time_t fallback)
{
  strmap::iterator i = data.find(key), e = data.end();
  time_t t;
  if(i==e)
    return fallback;
  std::istringstream s(i->second);
  if(s >> t);
    return t;
  return fallback;
}


/// Extract some rom data from a rom file.
/// @return true if valid rom data loaded, false otherwise.
bool s9x_ROMdata::extractROMdata(const std::string& dir, const std::string& file)
{
   bool valid = false;
   unsigned int i, o;
   char title[32];
   char *rombytes;
   std::string filename;
   static const long offsets[] =
   {
     0x7fc0, 0x81c0, 0x101c0
   };
#ifdef ZLIB_VERSION
# define romopen(f) (gzFile*)gzopen(f.c_str(), "rb")
# define romseek gzseek
# define romread gzread
# define romclose gzclose
   gzFile *fptr;
#else
# define romopen(f) fopen(f.c_str(), "rb")
# define romseek fseek
# define romread(f, b, s) fread(b, 1, s, f)
# define romclose fclose
   FILE *fptr;
#endif

   if(file[0]=='.')
     return false;

   i = dir.size();
   if(i)
   {
     filename = dir;
     if(dir[i-1]!='/')
       filename += '/';
   }
   else
     filename = "./";
   filename += file;

   fptr = romopen(filename);
   if(!fptr)
     return false;

   for(o=0; o < sizeof(offsets); o++)
   {
     if(romseek(fptr, offsets[o], SEEK_SET) < 0)
     {
       romclose(fptr);
       return false;
     }
     if(romread(fptr, title, sizeof(title))!=sizeof(title))
     {
       romclose(fptr);
       return false;
     }

     rombytes = title - 0xc0;
     if(
	((rombytes[0xd5] >= '0' )&&(rombytes[0xd5] <='9'))
	||((rombytes[0xd7] >= 0x0A)&&(rombytes[0xd7] <= 0x0f))
	||((rombytes[0xd9] >= 0x01)&&(rombytes[0xd9] <= 0x09))
	) {
	 valid = true;
	 for(i=0; i<21; i++)
	   if(!strchr(ROMcharacters, title[i])) {
	      valid = false;
	      break;
	   };
      }
      if(valid) break;
   }
   romclose(fptr);

   if(!valid)
     return false;

   title[21] = 0;
   for(o=21; o && title[--o] <= ' ';) //rtrim
       title[o] = 0;
   setData("title", title);
   
   return true;

# undef romopen
# undef romseek
# undef romread
# undef romclose
}

std::istream& operator>>(std::istream& i, s9x_ROMdata& rd)
{
  std::string buf;
  i >> buf;
  fr_UrlDecode(buf, rd.data);
  return i;
}

std::ostream& operator<<(std::ostream& out, const s9x_ROMdata& rd)
{
  std::string s = fr_UrlEncode(rd.data);
  if(!s.size())
    s = "-";
  return out << s;
}

/*
bool s9x_ROMdata::setFile(const std::string& dir, const std::string& file) {
   struct stat statbuf;
   char*dot;

   FileName = dir;
   if(dir[dir.size()-1]!='/') FileName += '/';
   FileName += file;
   snprintf(Name, sizeof(Name), "%s", file.c_str());
   if((dot = strrchr(Name, '.')))
     dot[0] = 0;

   if(stat(FileName.c_str(), &statbuf)!=0)
     return -1;
   filesize = statbuf.st_size;

   is_a_rom = true;
   return 0;
}
*/

/* ############################# s9x_ROMselector ########################## */

s9x_ROMselector::s9x_ROMselector(fr_Element*parent):
fr_Window(parent, "ROM Selector"),
RomIcon(this, "ROM"),
ROMlist(this, 1),
ROMdetails(this, "ROM Details"),
BtnOK(this, "Ok"),
BtnCancel(this, "Cancel"),
BtnBox(this),
romdatafile("roms")
{
   SetGridSize(2, 2, false);
   SetPadding(3, 3);
   //SetSize(630, 400);

   CountFiles = 0;
   ROMlist.AddListener(this);
   ROMlist.SetColumnWidth(0, 180);
   ROMlist.SetSize(200, 330);
   ROMlist.SetSortColumn(0);
   ROMlist.SetSortOnInsert(true);
   Pack(ROMlist);

   ROMdetails.SetSize(420, 330);
   ROMdetails.setDefaultColors(fr_Colors(0xffffff, 0x000000));
   Pack(ROMdetails);

   BtnOK.AddListener(this);
   BtnCancel.AddListener(this);
   BtnBox.AddButton(BtnOK, true);
   BtnBox.AddButton(BtnCancel);
   Pack(BtnBox, 2, 1);

   loadROMdata();
}

s9x_ROMselector::~s9x_ROMselector()
{
  saveROMdata();
}

void s9x_ROMselector::loadROMdata()
{
  std::ifstream infile;
  std::string buf;
  s9x_ROMdata *r;
  
  if(!romdatafile.exists())
    return;

  try
  {
    romdatafile.open(infile);
    std::getline(infile, buf);
    while(infile >> buf)
    {
      if((buf.size() < 3) || (buf[0]!=':'))
        continue;
      r = new s9x_ROMdata(buf.substr(1));
      infile >> *r;
      romdatamap[r->getID()] = r;
      //std::cerr << "added rom to cache: [" << r->getID() << "] = " << r << std::endl;
      //std::cerr << " --- titled: \"" << r->getROMtitle() << "\"" << std::endl;
    }
  }
  catch(std::string errmsg)
  {
    Mesg("Error loading rom data:\n" + romdatafile + "\n" + errmsg, true);
  }
  catch(...)
  {
    Mesg("An unknown error occured loading rom data file:\n" + romdatafile, true);
  }
  romdatafile.close(infile);
}

void s9x_ROMselector::saveROMdata()
{
  std::string s;
  std::ofstream out;

  if(!romdatamap.size())
  {
    romdatafile.remove();
    return;
  }
  try
  {
    romdatafile.open(out);
    out << PROG << " roms" << std::endl;
    ROMDataMap::iterator i = romdatamap.begin(), e = romdatamap.end();
    for(; i != e; i++)
    {
        s9x_ROMdata *r = i->second;
        s = r->getID();
        if(!s.size())
          continue;
        out << ':' << s << ' ' << *r << std::endl;
    }
  }
  catch(std::string errmsg)
  {
    Mesg("Error saving rom data:\n" + romdatafile + "\n" + errmsg, true);
  }
  catch(...)
  {
    Mesg("An unknown error occured saving rom data file:\n" + romdatafile, true);
  }
  //romdatafile.close(out);
}

void s9x_ROMselector::loadDirectory(const std::string& directory) {
  std::string dir(directory), ext, matchfile;
  std::string::iterator b;
  const char *dot, *c_ext, **f;
  int i, r;
  bool findmatch = false;

  static const char*knownformats[] =
  {
    ".smc", ".sfc", ".fig",
#ifdef ZLIB_VERSION
    ".smc.gz", ".sfc.gz", ".gif.gz",
#endif
    NULL
  };
   
  static const char*unknownformats[] =
  {
    ".zip",
#ifndef ZLIB_VERSION
    ".smc.gz", ".sfc.gz", ".fig.gz",
#endif
    ".smc.z", ".sfc.z", ".fig.z",
    NULL
  };

  struct dirent *DirEntry;
  DIR *d;

  if(!dir.size())
    dir = ".";

  d = opendir(dir.c_str());
  if(!d)
  {
    std::string msg("Unable to read directory:\n");
    msg += dir + "\n" + fr_syserr();
    Mesg(msg, true);
    return;
  }

  if(dir == ((s9x_ROM*)Parent)->getROMdir())
  {
    matchfile = fr_Basename(((s9x_ROM*)Parent)->GetFileName());
    if(matchfile.size())
      findmatch = true;
  }
  for(DirEntry = readdir(d);  DirEntry;  DirEntry = readdir(d))
  {
    if((!DirEntry->d_name[0])||(DirEntry->d_name[0]=='.'))
      continue;
    dot = strchr(DirEntry->d_name, '.');
    if(!dot)
      continue;
    ext = dot;
    b = ext.begin();
    transform(b, ext.end(), b, tolower);
    c_ext = ext.c_str();
    f = NULL;
    for(i=0; !f && knownformats[i]; i++)
      if(ext==knownformats[i])
        f = knownformats;
    for(i=0; !f && unknownformats[i]; i++)
      if(ext==unknownformats[i])
        f = unknownformats;
    r = -1;
    if(f==knownformats)
      r = addROM(dir, DirEntry->d_name, true);
    else if(f==unknownformats)
      r = addROM(dir, DirEntry->d_name, false);
    if(findmatch && (r!=-1) && (matchfile==DirEntry->d_name))
    {
      ROMlist.Select(r);
      RowSelected(r);
      findmatch = false;
    }
   }
   closedir(d);
}


/// Add a SNES ROM to the ROMlist
int s9x_ROMselector::addROM(const std::string& dir, const std::string& file, bool extractdata)
{
   const char* newdata[] =
   {
      "-", ""
   };
   int r;
   std::string title, fileid;
   ROMDataMap::iterator cache, notfound = romdatamap.end();
   s9x_ROMdata *d;

   fileid = getFileID(file);
   if(!fileid.size())
     return -1;


//   std::cerr << "+ adding rom to ROMlist:  " << file << " -> " << newdata[0] << " (" << r << ")" << std::endl;
   cache = romdatamap.find(fileid);
   if(cache==notfound)
   {
        d = new s9x_ROMdata(fileid);
        romdatamap[fileid] = d;
//        std::cerr << " - cache miss " << std::endl;
   }
   else
   {
        d = cache->second;
//        std::cerr << " - cache hit! " << std::endl;
   }
   

   title = d->getROMtitle();
   if(!title.size() && extractdata)
   {
//     std::cerr << " - extracting data from rom" <<  std::endl;
     d->extractROMdata(dir, file);
     title = d->getROMtitle();
   }
   if(!title.size())
   {
//     std::cerr << " - getting title from filename" <<  std::endl;
     title = getTitleFromFilename(file);
     d->setROMtitle(title);
   }
   if(!title.size())
   {
//     std::cerr << " ! no title, rejecting." <<  std::endl;
     return -1;
   }
//   std::cerr << " - adding \"" << title << '"' <<  std::endl;

   //newdata[0] = file.c_str();
   newdata[0] = title.c_str();
   r = ROMlist.AddRow(newdata);
   //ROMlist.SetCell(r, 0, title);
   ROMlist.AssociateData(r, strdup(file.c_str()));
   CountFiles++;
   return r;   
}

std::string s9x_ROMselector::getSelectedFilename() {
   int r;
   char *fn;

   if((r=ROMlist.GetSelected()) < 0)
     return "";

   fn = (char*)ROMlist.AssociatedData(r);
   if(!fn)
     return "";

   return fn;
}

std::string s9x_ROMselector::getSelectedFilepath()
{
  std::string f = getSelectedFilename(), p(romdir);
  int s = romdir.size();
  if(!f.size() || !s)
    return f;
  if(romdir[s-1]!='/')
    p += '/';
  return p + f;
}

void s9x_ROMselector::EventOccurred(fr_Event*e) {
   if(e->Is(BtnOK)||e->Is(ROMlist, fr_DoubleClick)) {
      ((s9x_ROM*)Parent)->SetFileName(getSelectedFilepath());
      SetVisibility(false);
   } else if(e->Is(ROMlist, fr_Select))
	RowSelected(e->intArg);
   else if(e->Is(BtnCancel))
     SetVisibility(false);
}

void s9x_ROMselector::RowSelected(int row)
{
   showROMdetails(row);
}

bool s9x_ROMselector::hasROMdata(const std::string& fid)
{
  ROMDataMap::iterator i = romdatamap.find(fid), e = romdatamap.end();
  if(i==e)
    return false;
  return true;
}

s9x_ROMdata* s9x_ROMselector::getROMdata(const std::string& fid)
{
  ROMDataMap::iterator i = romdatamap.find(fid), e = romdatamap.end();
  if(i==e)
    return NULL;
  return i->second;
}

s9x_ROMdata* s9x_ROMselector::getROMdata(int row)
{
  const char*fn = (const char*)ROMlist.AssociatedData(row);
  if(!fn)
    return NULL;
  return getROMdata(getFileID(fn));
}

void s9x_ROMselector::showROMdetails(int row)
{
  fr_TextArea &r(ROMdetails);
  std::string sec(" *** "), sep(" >>> "), ind(" ");
  s9x_ROMdata *romdata;
  std::string fn, fileid, title;

  r.clear();
  r.print("\n");

  static const fr_Colors n_clr(0xcccccc, 0x000000),
                         sec_clr(0x000000, 0xff0000),
                         lbl_clr(0xffffff, 0x000000);
  static const fr_Color dec_clr(0x9966cc), val_clr(0x00deda), err_clr(0xffff00);

  if(CountFiles < 1)
  {
    r << err_clr << ind << "No ROMs were found.\n"
      << ind << "Make sure you have set\n"
      << ind << "the correct ROM directory\n"
      << ind << "in the Preferences window.";
    return;
  }
  
  fn = (const char*)ROMlist.AssociatedData(row);
  fileid = getFileID(fn);
  romdata = getROMdata(fileid);
  title = ROMlist.GetCell(row, 0);
  

  r << ind << sec_clr << sec << " ROM Information " << sec << n_clr << "\n\n";
  r << ind << lbl_clr << "Title" << dec_clr << sep << val_clr << title << "\n";
  r << ind << lbl_clr << "File " << dec_clr << sep << val_clr << fn << "\n";
  r << ind << lbl_clr << "Dir  " << dec_clr << sep << val_clr << romdir << "\n\n";

  if(!romdata)
  {
    r << err_clr << "\nNo ROM data available.\n";
    return;
  }
  s9x_ROMdata &d(*romdata);
  int played = d.getData("played", 0);
  if(played < 2)
    return;

  time_t total = d.getData("total", 0), t;
  r << "\n" << ind << sec_clr << sec << " Play Statistics " << sec << n_clr << "\n\n";
    r << ind << lbl_clr << "Played       " << dec_clr
      << sep << val_clr << played << " times\n";
  t = total / played;
  if(t)
    r << ind << lbl_clr << "Avg play time" << dec_clr
      << sep << val_clr << timestr(t) << "\n";
    r << ind << lbl_clr << "Total played " << dec_clr
      << sep << val_clr << timestr(total) << "\n";
  t = d.getData("first", 0);
  if(t)
    r << ind << lbl_clr << "First played " << dec_clr
      << sep << val_clr << timestr(t) << "\n";
  t = d.getData("last", 0);
  if(t)
    r << ind << lbl_clr << "Last played  " << dec_clr
      << sep << val_clr << timestr(t) << "\n";
}

std::string s9x_ROMselector::timestr(time_t t)
{
  std::ostringstream s;
  if(t==1)
    s << "1 second";
  else if(t < 120)
    s << t << " seconds";
  else if(t < 600)
    s << (t / 60) << "m " << (t % 60) << "s";
  else if(t < 5400)
    s << (t / 60) << " minutes";
  else if(t < 1000000)
    s << (t / 3600) << "h " << ((t / 60) % 60) << "m";
  else
  {
    char buf[64];
    struct tm tmdata;
    localtime_r(&t, &tmdata);
    strftime(buf, sizeof(buf), "%a %b %d %Y, %X", &tmdata);
    buf[sizeof(buf)-1] = 0;
    s << buf;
  }
  return s.str();
}

void s9x_ROMselector::doPrePlay(const std::string& f)
{
  nowplaying = f;
  time(&playtime);
}

void s9x_ROMselector::doPostPlay()
{
  int played = 0;
  std::string fid = getFileID(nowplaying);
  if(!fid.size())
    return;
  time_t stoptime, total, thistime, shortest, longest;
  s9x_ROMdata *rd;
  time(&stoptime);
  ROMDataMap::iterator cache, notfound = romdatamap.end();
  cache = romdatamap.find(fid);
  if(cache==notfound)
  {
        rd = new s9x_ROMdata(fid);
        romdatamap[fid] = rd;
  }
  else
        rd = cache->second;
  thistime = stoptime - playtime;
  total = rd->getData("total", 0) + thistime;
  played = rd->getData("played", 0);
  shortest = rd->getData("shortest", thistime);
  longest = rd->getData("longest", thistime);
  if(!played)
    rd->setData("first", playtime);
  rd->setData("played", ++played);
  rd->setData("total", total);
  rd->setData("shortest", std::min(shortest, thistime));
  rd->setData("longest", std::max(longest, thistime));
  rd->setData("last", playtime);
}

void s9x_ROMselector::setROMdir(const std::string& d)
{
  romdir = d;
  if(isVisible())
    RefreshContent();
}

void s9x_ROMselector::SetVisibility(bool v)
{
	Parent->SetEditable(!v);
        fr_Window::SetVisibility(v);
        ROMdetails.clear();
	fr_Flush();
	if(v)
	  RefreshContent();
	else
	  DeleteContent();
}

bool s9x_ROMselector::RefreshContent() {
   if(CountFiles>0)
     DeleteContent();

   ROMlist.SetSortColumn(0);
   ROMlist.SetSortOnInsert(true);
   if(romdir.size())
	loadDirectory(romdir);
   if(CountFiles<1)
   {
      BtnOK.SetEditable(false);
      showROMdetails(-1);
   }
   else if(CountFiles>0)
   {
     BtnOK.SetEditable(true);
     //ROMlist.Sort();
   }
   return true;
}

void s9x_ROMselector::DeleteContent() {
   char*fn;

   Parent->SetEditable(true);
   if(!CountFiles)
     return;
   for(int r=0; r<CountFiles; r++)
   {
      fn = (char*)ROMlist.AssociatedData(r);
      delete[] fn;
   }
   ROMlist.RemoveAll();
   CountFiles = 0;
}

/// Given a filename, create an id token for it
std::string s9x_ROMselector::getFileID(const std::string& filename)
{
  char c;
  bool ignore = false;
  unsigned int slash, dot;
  slash = filename.rfind('/');
  if(slash==filename.npos)
    slash = 0;
  else
    slash++;
  dot = filename.find('.', slash);
  std::string f(filename, slash, dot==filename.npos?dot:(dot-slash)), s;
  std::string::iterator i = f.begin(), e = f.end();
  
  for(; i != e; i++)
  {
    c = *i;
    if(strchr(")]}>", c))
    {
      ignore = false;
      continue;
    }
    else if(ignore)
      continue;
    
    if(isalpha(c))
      s += tolower(c);
    else if(isdigit(c))
      s += c;
    else if(strchr("<{[(", c))
      ignore = true;
  }
  return s;
}

/// Given a filename, create a title for it
std::string s9x_ROMselector::getTitleFromFilename(const std::string& filename)
{
  char c;
  bool ignore = false;
  unsigned int slash, dot;
  slash = filename.rfind('/');
  if(slash==filename.npos)
    slash = 0;
  else
    slash++;
  dot = filename.find('.', slash);
  std::string f(filename, slash, dot==filename.npos?dot:(dot-slash)), s;
  std::string::iterator i = f.begin(), e = f.end();

  for(; i != e; i++)
  {
    c = *i;
    if(strchr(")]}>", c))
    {
      ignore = false;
      continue;
    }
    else if(ignore)
      continue;

    if(strchr("<{[(", c))
      ignore = true;
    else if(strchr(ROMcharacters, c))
      s += c;
  }
  return s;
}
