// ---------------------------------------------------------------------------
// - Argument.cpp                                                            -
// - aleph engine - argument class implementation                            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Stack.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "Argument.hpp"
#include "Exception.hpp"

namespace aleph {

  // the argument supported quarks
  static const long QUARK_GETCONST  = String::intern ("get-const");
  static const long QUARK_GETOBJECT = String::intern ("get-object");
  static const long QUARK_SETCONST  = String::intern ("set-const");
  static const long QUARK_SETOBJECT = String::intern ("set-object");

  // create a argument by quark with an index

  Argument::Argument (const long quark, const long index) {
    d_quark = quark;
    d_index = index;
    d_const = false;
  }

  // copy construct this argument

  Argument::Argument (const Argument& that) {
    d_quark = that.d_quark;
    d_index = that.d_index;
    d_const = that.d_const;
  }

  // return the class name

  String Argument::repr (void) const {
    return "Argument";
  }

  // return a clone of this argument

  Object* Argument::clone (void) const {
    return new Argument (*this);
  }

  // get a literal representation of this argument

  String Argument::toliteral (void) const {
    return String::qmap (d_quark);
  }
    
  // get a string representation of this argument

  String Argument::tostring (void) const {
    return String::qmap (d_quark);
  }

  // set the const flag for this argument

  void Argument::setconst (const bool flag) {
    d_const = flag;
  }

  // get the const flag for this argument

  bool Argument::getconst (void) const {
    return d_const;
  }

  // set the argument object

  void Argument::setobj (Runnable* robj, Object* object) {
    // check for the const flag
    if (d_const == true)
      throw Exception ("const-error", "const violation for argument", 
		       String::qmap (d_quark));
    // get the stack runnable and set the object
    Stack* stk = robj->getstk ();
    stk->set (d_index, object);
  }

  // set this argument as a const object

  Object* Argument::cdef (Runnable* robj, Nameset* nset, Object* object) {
    setobj (robj, object);
    d_const = true;
    return object;
  }

  // set this argument with an object

  Object* Argument::vdef (Runnable* robj, Nameset* nset, Object* object) {
    setobj (robj, object);
    return object;
  }

  // evaluate this object - this is like get object

  Object* Argument::eval (Runnable* robj, Nameset* nset) {
    Stack* stk = robj->getstk ();
    return stk->get (d_index);
  }

  // apply this argument with a set of arguments and a quark

  Object* Argument::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETCONST)  return new Boolean (getconst ());
      if (quark == QUARK_GETOBJECT) {
	Object* result = eval (robj, nset);
	robj->post (result);
	return result;
      }
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETCONST) {
	bool bval = argv->getbool (0);
	setconst (bval);
	return nilp;
      }
      if (quark == QUARK_SETOBJECT) {
	setobj (robj, argv->get (0));
	return nilp;
      }
    }
    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
