// ---------------------------------------------------------------------------
// - Class.cpp                                                               -
// - aleph engine - class class implementation                               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Class.hpp"
#include "Lexical.hpp"
#include "Boolean.hpp"
#include "Builtin.hpp"
#include "Instance.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a new class object

  Class::Class (void) {
    p_cset  = new Localset;
    Object::iref (p_cset);
  }

  // destroy this class object

  Class::~Class (void) {
    Object::dref (p_cset);
  }

  // return the class name

  String Class::repr (void) const {
    return "Class";
  }

  // make this class a shared object

  void Class::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_cset != nilp) p_cset->mksho ();
  }

  // add a new data member by quark
  
  void Class::madd (const long quark) {
    if (d_mdata.exists (quark) == true)
      throw Exception ("duplicate-error", "duplicate data member name", 
		       String::qmap (quark));
    d_mdata.add (quark);
  }

  // operate this class with another object

  Object* Class::oper (Runnable* robj, t_oper type, Object* object) {
    Class* cobj = dynamic_cast <Class*> (object);
    switch (type) {
    case Object::EQL:
      if (cobj != nilp) return new Boolean (this == cobj);
      break;
    case Object::NEQ:
      if (cobj != nilp) return new Boolean (this != cobj);
      break;
    default:
      throw Exception ("operator-error", "unsupported class operator");
    }
    throw Exception ("type-error", "invalid operand with class",
		     Object::repr (object));
  }

  // create a new static const symbol by quark

  Object* Class::cdef (Runnable* robj, Nameset* nset, const long quark, 
		       Object* object) {
    return p_cset->cdef (robj, nset, quark, object);
  }

  // create or set a new static symbol by quark

  Object* Class::vdef (Runnable* robj, Nameset* nset, const long quark, 
		       Object* object) {
    return p_cset->vdef (robj, nset, quark, object);
  }

  // evaluate a static member by quark

  Object* Class::eval (Runnable* robj, Nameset* nset, const long quark) {
    // look in the local set
    Object* obj = p_cset->find (quark);
    if (obj != nilp) return obj->eval (robj, nset);
    // go to the object
    return Object::eval (robj, nset, quark);
  }

  // create a new instance with a set of arguments

  Object* Class::apply (Runnable* robj, Nameset* nset, Cons* args) {
    return new Instance (robj, nset, args, this);
  }

  // apply a member function with a set of arguments by quark

  Object* Class::apply (Runnable* robj, Nameset* nset, const long quark,
			Cons* args) {
    Object* obj = eval (robj, nset, quark);
    return (obj == nilp) ? nilp : obj->apply (robj, nset, args);
  }

  // create a new class object - this is the builtin function

  Object* builtin_class (Runnable* robj, Nameset* nset, Cons* args) {
    long len  = (args == nilp) ? 0 : (args->length ());
    // process 0 arguments
    if (len == 0) return new Class;
    // process data member list
    if (len == 1) {
      Class* cls = new Class;
      Cons* cons = dynamic_cast <Cons*> (args->getcar ());
      if (cons == nilp) throw Exception ("argument-error",
					 "only data member list with class");
      while (cons != nilp) {
	Lexical* lex = dynamic_cast <Lexical*> (cons->getcar ());
	if (lex == nilp) 
	  throw Exception ("argument-error",
			   "only lexical name with class data memeber list");
	cls->madd (lex->toquark ());
	cons = cons->getcdr ();
      }
      return cls;
    }
    throw Exception ("argument-error",
		     "too many arguments with class definition");
  }
}
