// ---------------------------------------------------------------------------
// - Collection.cpp                                                          -
// - aleph:odb library - collection class implementation                     -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Integer.hpp"
#include "Runnable.hpp"
#include "Collection.hpp"
#include "Exception.hpp"

namespace aleph {
  // the collection supported quarks
  static const long QUARK_TADD    = String::intern ("table-add");
  static const long QUARK_TGET    = String::intern ("table-get");
  static const long QUARK_TSET    = String::intern ("table-set");
  static const long QUARK_LENGTH  = String::intern ("length");
  static const long QUARK_GETNAME = String::intern ("get-name");

  // the collection magic number
  const long ODB_MSIZE   = 4;
  const char ODB_MAGIC[] = {'\377', 'O', 'D', 'B'};

  // this function write the collection header
  static void write_collection_magic (Output& os) {
    for (long i = 0; i < ODB_MSIZE; i++) os.write (ODB_MAGIC[i]);
  }

  // this function check that the header matches the edc magic number
  static bool check_collection_magic (Input* is) {
    // check for nil and reset
    if (is == nilp) return false;
    // read in the magic number
    char mbuf[ODB_MSIZE];
    for (long i = 0; i < ODB_MSIZE; i++) {
      mbuf[i] = is->read ();
      if (mbuf[i] != ODB_MAGIC[i]) {
	is->pushback (mbuf, i+1);
	return false;
      }
    }
    return true;
  }

  // create a nil collection

  Collection::Collection (void) {
  }

  // create a new collection by name

  Collection::Collection (const String& name) {
    d_name = name;
  }

  // create a new collection by stream

  Collection::Collection (Input* is) {
    // check for valid header
    if (check_collection_magic (is) == false)
      throw Exception ("collection-error", "invalid stream header");
    // read-in data
    rdstream (*is);
  }

  // return the object name

  String Collection::repr (void) const {
    return "Collection";
  }

  // write the collection to an output stream
  
  void Collection::write (Output& os) {
    rdlock ();
    try {
      // write the header
      write_collection_magic (os);
      // serialize the collection
      wrstream (os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // serialize a collection

  void Collection::wrstream (Output& os) const {
    rdlock ();
    // save the collection name
    d_name.wrstream (os);
    // save the vector
    d_vtbls.wrstream (os);
    unlock ();
  }

  // deserialize this collection

  void Collection::rdstream (Input& is) {
    wrlock ();
    // get the collection name
    d_name.rdstream (is);
    // get the vector
    d_vtbls.rdstream (is);
    unlock ();
  }

  // return the collection name

  String Collection::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // return the length of the collection

  long Collection::length (void) const {
    rdlock ();
    long result = d_vtbls.length ();
    unlock ();
    return result;
  }

  // add a table in this collection

  void Collection::tadd (Table* table) {
    wrlock ();
    d_vtbls.append (table);
    unlock ();
  }

  // get a table by index

  Table* Collection::tget (const long index) const {
    rdlock ();
    try {
      Table* result = dynamic_cast <Table*> (d_vtbls.get (index));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set a table in this collection by index

  void Collection::tset (const long index, Table* table) {
    wrlock ();
    d_vtbls.set (index, table);
    unlock ();
  }

  // create a new collection in a generic way

  Object* Collection::mknew (Vector* argv) {
    // get number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Collection;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Collection (name);
    }
    throw Exception ("argument-error", "too many argument with collection");
  }

  // apply a collection method with a set of arguments and a quark

  Object* Collection::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH ) return new Integer (length ());
      if (quark == QUARK_GETNAME) return new String  (getname ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_TGET) {
	long idx = argv->getint (0);
	rdlock ();
	try {
	  Object* result = tget (idx);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_TADD) {
	Object* obj = argv->get (0);
	Table*  sht = dynamic_cast <Table*> (obj);
	if ((obj != nilp) && (sht == nilp)) 
	  throw Exception ("type-error", "invalid object to add in collection",
			   obj->repr ());
	tadd (sht);
	return nilp;
      }
    }
    // dispatch 2 argument
    if (argc == 2) {
      if (quark == QUARK_TSET) {
	long    idx = argv->getint (0);
	Object* obj = argv->get (1);
	Table*  sht = dynamic_cast <Table*> (obj);
	if ((obj != nilp) && (sht == nilp)) 
	  throw Exception ("type-error", "invalid object to set in collection",
			   obj->repr ());
	tset (idx, sht);
	return nilp;
      }
    }
    // call the persist method
    return Persist::apply (robj, nset, quark, argv);
  }
}
