/*
 * Copyright 2002 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "bakery/Document/Document.h"
#include <libgnomevfsmm.h>
//#include <fstream>
#include <libintl.h>

namespace Bakery
{

const guint BYTES_TO_PROCESS = 256;

Document::Document()
{
  m_bIsNew = true;
  m_bModified = false;
  m_bReadOnly = false;
  m_pView = 0;
}

Document::~Document()
{
  //Tell views to forget the document -  to null their pointers to it. We should maybe use the Document via a sharing smartpointer instead.
  signal_forget_.emit();
}

Glib::ustring Document::get_file_uri() const
{
  return m_file_uri;
}

void Document::set_file_uri(const Glib::ustring& file_uri, bool bEnforceFileExtension /* = false */)
{
  if(file_uri != m_file_uri)
    set_modified(); //Ready to save() for a Save As.

  m_file_uri = file_uri;

  //Enfore file extension:
  if(bEnforceFileExtension)
  {
    if(m_strFileExtension.size() != 0)  //If there is an extension to enforce.
    {
      bool bAddExt = false;
      Glib::ustring strExt = "." + get_file_extension();
      
      if(m_file_uri.size() < strExt.size()) //It can't have the ext already if it's not long enough.
      {
        bAddExt = true; //It isn't there already.
      }
      else
      {
        Glib::ustring strEnd = m_file_uri.substr(m_file_uri.size()-strExt.size());
        if(strEnd != strExt) //If it doesn't already have the extension
          bAddExt = true;
      }
 
      //Add extension if necessay.
      if(bAddExt)
        m_file_uri += strExt;
        
      //Note that this does not replace existing extensions, so it could be e.g. 'something.blah.theext'
    }
  }
}

void Document::set_contents(const Glib::ustring& strVal)
{
  m_strContents = strVal;
}

Glib::ustring Document::get_contents() const
{
  return m_strContents;
}

void Document::set_modified(bool bVal /* = true */)
{
  m_bModified = bVal;

  if(m_bModified)
  {
    m_bIsNew = false; //Can't be new if it's been modified.
  }

  //Allow the application or view to update it's UI accordingly:
  signal_modified().emit(m_bModified);
}

bool Document::get_modified() const
{
  return m_bModified;
}

bool Document::load()
{
  bool bTest = read_from_disk();
  if(bTest)
  {
    bTest = load_after(); //may be overridden.
    if(bTest)
    {
      //Tell the View to show the new data:
      if(m_pView)
        m_pView->load_from_document();
    }
  }

  return bTest;
}

bool Document::load_after()
{
  //Called after text is read from disk, but before updating view.

  //Override this if necessary.
  //For instance, Document_XML parses the XML.

  return true;
}

bool Document::save()
{
  //Tell the view to update the data in this document.
  if(m_pView)
    m_pView->save_to_document();

  bool bTest = save_before(); //This could be overridden.
  if(bTest)
    return write_to_disk();

  return bTest;
}

bool Document::save_before()
{
  //Called after view saves itself to document, but before writing to disk.

  //Override this if necessary.
  //For instance, Document_XML serializes its XML to text.

  return true;
}

bool Document::read_from_disk()
{
  m_strContents.erase();

  // open the input file for read access
  Gnome::Vfs::Handle read_handle;

  try
  {
    read_handle.open(m_file_uri, Gnome::Vfs::OPEN_READ);
  }
  catch(const Gnome::Vfs::exception& ex)
  {
    // If the operation was not successful, print the error and abort
    return false; //print_error(ex, input_uri_string);
  }

  // Read data from the input uri:
  guint buffer[BYTES_TO_PROCESS] = {0, }; // For each chunk.
  Gnome::Vfs::FileSize bytes_read = 0;
  std::string data; //We use a std::string because we might not get whole UTF8 characters at a time. This might not be necessary.

  try
  {
    bool bContinue = true;
    while(bContinue)
    {
      bytes_read = read_handle.read(buffer, BYTES_TO_PROCESS);

      if(bytes_read == 0)
        bContinue = false; //stop because we reached the end.
      else
      {
        // Add the text to the string:
        data += std::string((char*)buffer, bytes_read);
      }
    }
  }
  catch(const Gnome::Vfs::exception& ex)
  {
    // If the operation was not successful, print the error and abort
    return false; //print_error(ex, input_uri_string);
  }

  m_strContents = data;

  set_modified(false);

  return true; //Success.
}

bool Document::write_to_disk()
{
  //Write the changed data to disk:
  if(get_modified())
  {
    /* we use create instead of open, because open will not create the file if it does
    not already exist. The last argument is the permissions to use if the file is created,
    the second to last tells GnomeVFS that its ok if the file already exists, and just open it */

    Gnome::Vfs::Handle write_handle;
      
    try
    {
      //0660 means "this user and his group can read and write this non-executable file".
      //The 0 prefix means that this is octal.
      write_handle.create(m_file_uri, Gnome::Vfs::OPEN_WRITE, false, 0660 /* leading zero means octal */);
    }
    catch(const Gnome::Vfs::exception& ex)
    {
     // If the operation was not successful, print the error and abort
     return false; // print_error(ex, output_uri_string);
    }

    try
    {
      //Write the data to the output uri
      GnomeVFSFileSize bytes_written = write_handle.write(m_strContents.data(), m_strContents.bytes());
    }
    catch(const Gnome::Vfs::exception& ex)
    {
      // If the operation was not successful, print the error and abort
      return false; //print_error(ex, output_uri_string);
    }
  
    return true; //Success. (At doing nothing, because nothing needed to be done.)
  }
  else
    return true; //Success. (At doing nothing, because nothing needed to be done.)
}

Glib::ustring Document::get_name() const
{
  return util_file_uri_get_name(m_file_uri);
}

Glib::ustring Document::util_file_uri_get_name(const Glib::ustring& file_uri)
{
  Glib::ustring strResult;

  if(file_uri.size())
  {
    //Find position of last '/' character.
    Glib::ustring::size_type posLastSlash = file_uri.find_last_of("/");
    if(posLastSlash == Glib::ustring::npos)
    {
      //No '/' found. The file must be in the current or top-level directory.
      strResult = file_uri;
    }
    else
    {
      //'/' found. The filename is the text after this: e.g. sub/sub/filename:
      strResult = file_uri.substr(posLastSlash+1);
    }
  }

  //Show untitled explicitly:
  //Also happens for file_uris with path but no name. e.g. /sub/sub/, which shouldn't happen.
  if(strResult.size() == 0)
    strResult = gettext("Untitled");

  return strResult;
}

void Document::set_view(ViewBase* pView)
{
  m_pView = pView;
}

ViewBase* Document::get_view()
{
  return m_pView;
}

bool Document::get_read_only() const
{
  if(m_bReadOnly)
  {
    //An application might have use set_read_only() to make this document explicitly read_only, regardless of the positions of the storage location.
    return true;
  }
  else
  {
    if(m_file_uri.empty())
      return false; //It must be a default empty document, not yet saved, so it is not read-only.
    else
    {
      try
      {
        Gnome::Vfs::FilePermissions permissions = Gnome::Vfs::Handle::get_file_info(m_file_uri, Gnome::Vfs::FILE_INFO_GET_ACCESS_RIGHTS)->get_permissions();
        bool read_only = ((permissions & Gnome::Vfs::PERM_ACCESS_WRITABLE) != Gnome::Vfs::PERM_ACCESS_WRITABLE);
        return read_only;
      }
      catch(const Gnome::Vfs::exception& ex)
      {
        return false; //We should at least be able to read the permissions, so maybe the location is invalid. I'm not sure what the best return result here is.
      }
    }
  }
  
  return m_bReadOnly;
}

void Document::set_read_only(bool bVal)
{
  m_bReadOnly = bVal;
}

bool Document::get_is_new() const
{
  return m_bIsNew;
}

void Document::set_is_new(bool bVal)
{
  if(bVal)
    set_modified(false); //can't be modified if it is new.
    
  m_bIsNew = bVal; 
}

void Document::set_file_extension(const Glib::ustring& strVal)
{
  m_strFileExtension = strVal;
}

Glib::ustring Document::get_file_extension() const
{
  return m_strFileExtension; //TODO: get it from the mime-type system?
}

Document::type_signal_modified& Document::signal_modified()
{
  return signal_modified_;
}

Document::type_signal_forget& Document::signal_forget()
{
  return signal_forget_;
}




} //namespace
