/***************************************************************************
                          bkutil.cpp  -  description
                             -------------------
    begin                : Sun Nov 24 2002
    copyright            : (C) 2002 by Ken Schenke
    email                : kenschenke at yahoo dot com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   In addition, as a special exception, Ken Schenke gives permission to  *
 *   link the code of this program with the Qt non-commercial edition (or  *
 *   with modified versions of the Qt non-commercial edition that use the  *
 *   same license as the Qt non-commercial edition, and distribute linked  *
 *   combinations including the two.  You must obey the GNU General Public *
 *   License in all respects for all of the code used other than the Qt    *
 *   Non-Commercial edition.  If you modify this file, you may extend this *
 *   exception to your version of the file, but you are not obligated to   *
 *   do so.  If you do not wish to do so, delete this exception statement  *
 *   from your version.                                                    *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   See bkutil.h for an explanation of this class's purpose.              *
 *                                                                         *
 ***************************************************************************/

#include "bkutil.h"
#include "browserbk.h"
#include "browserlist.h"

#include <vector>
#include <algorithm>

extern BROWSERLIST BrowserList[];

using namespace std;

/***************************************************************************
 *                                                                         *
 *   BkUtil::BkUtil()                                                      *
 *                                                                         *
 *   Parameters:                                                           *
 *      BridgeCfg &                                                        *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkNode class constructor.                                          *
 *                                                                         *
 ***************************************************************************/

BkUtil::BkUtil(BridgeCfg & cfg) : m_cfg(cfg)
{
	m_root = NULL;
}

/***************************************************************************
 *                                                                         *
 *   BkUtil::~BkUtil()                                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      void                                                               *
 *   Description:                                                          *
 *      BkNode class destructor.                                           *
 *                                                                         *
 ***************************************************************************/

BkUtil::~BkUtil()
{
	delete m_root;
}

/***************************************************************************
 *                                                                         *
 *   BkUtil::readBookmarksFromBrowsers()                                   *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      false for success, true for failure                                *
 *   Description:                                                          *
 *      This function loops through each browser listed as a source in the *
 *      configuration (BridgeCfg class) and constructs an instance of each *
 *      browser's implentation class.  For each, it asks the class to read *
 *      that browser's bookmark database.  Once the bookmarks are read, it *
 *      merge's the browser's bookmarks into the a tree holding bookmarks  *
 *      for all the browsers.  This central bookmark tree is read from the *
 *      XBEL file, if found.                                               *
 *                                                                         *
 ***************************************************************************/

void BkUtil::readBookmarksFromBrowsers(void) throw(BkException)
{
	vector<BRWSNUM>::const_iterator sit;
	vector<BrowserCfg>::const_iterator fit;

	// If we don't have a root folder yet, create one.

	BrowserBk	*bk = NULL;
	
	if(m_root == NULL)
	{
		if((m_root = new BkFolder) == NULL)
			BKEXCEPT("Unable to Allocate Memory");
	}

	// loop through the list of browsers defined as sources
	
	try
	{
		for(sit=m_cfg.sources.begin(); sit!=m_cfg.sources.end(); ++sit)
		{
			// find the browser in the list of configured browsers
			
			fit = std::find(m_cfg.browsers.begin(), m_cfg.browsers.end(), *sit);
			if(fit == m_cfg.browsers.end())
				BKEXCEPT("Unable to find information on a browser configured as a source");

			// loop through the list of configured browsers
			
			// create an instance of the browser's implementation class
			
			bk = BrowserList[fit->browserNum()-1].Factory();

			// is the browser running right now?

			if(bk->IsBrowserRunning())
			{
				QString msg;
				msg.sprintf("The %s browser is running right now.  Unable to merge.",
					BrowserList[fit->browserNum()-1].Name);
				delete bk;
				bk = NULL;
				BKEXCEPT(msg);
			}

			// ask the class to read that browser's bookmarks
			
			bk->readBookmarks(fit->location().c_str(), *sit);

			// clean up the bookmarks

			bk->scrubBookmarks();

			// merge the browser's bookmarks into the central bookmark tree
			
			bk->findDifferences(*m_root);

			// we're done with the browser class
			
			delete bk;
			bk = NULL;
		}
	}
	catch(BkException)
	{
		delete bk;
		throw;		// with resources cleaned up, re-throw the exception
	}

	// Each node in the bookmark tree knows which browsers in which it is
	// saved.  If a bookmark or folder is stored in a browser but that
	// browser is not configured as a source, BookmarkBridge will not
	// know that the bookmark actually exists in that browser.  For that reason,
	// we need to step through each configured browser not defined as a source
	// and mark it has having been found in that browser.  This will prevent
	// BookmarkBridge from thinking the bookmark or folder has been deleted.

	for(fit=m_cfg.browsers.begin(); fit!=m_cfg.browsers.end(); ++fit)
	{
		sit = std::find(m_cfg.sources.begin(), m_cfg.sources.end(), fit->ordinal());
		if(sit == m_cfg.sources.end())
			setBrowserAsSource(*m_root, fit->ordinal());
	}
}

/***************************************************************************
 *                                                                         *
 *   BkUtil::saveBookmarksToBrowsers()                                     *
 *                                                                         *
 *   Parameters:                                                           *
 *      None                                                               *
 *   Return:                                                               *
 *      false for success, true for failure                                *
 *   Description:                                                          *
 *      This function loops through the browsers defined as destinations   *
 *      and asks each browser's class to save the bookmarks from the       *
 *      master bookmark tree back to the browser's bookmark database.      *
 *                                                                         *
 ***************************************************************************/

void BkUtil::saveBookmarksToBrowsers(void) const throw(BkException)
{
	vector<BRWSNUM>::const_iterator sit;
	vector<BrowserCfg>::const_iterator fit;

	// Loop through the browsers configured as destinations

	for(sit=m_cfg.destinations.begin(); sit!=m_cfg.destinations.end(); ++sit)
	{
		// figure out which browser this is
		
		fit = std::find(m_cfg.browsers.begin(), m_cfg.browsers.end(), *sit);
		if(fit == m_cfg.browsers.end())
			BKEXCEPT("Unable to Find Information on Browser Configured as Destination");

		// save the bookmarks
		
		BrowserList[fit->browserNum()-1].Save(fit->location().c_str(), *m_root, fit->ordinal());
	}
}

void BkUtil::setBrowserAsSource(BkFolder &folder, BRWSNUM browserOrd)
{
	FolderLst::iterator fit;
	BookmarkLst::iterator bit;

	for(bit=folder.BookmarksBegin(); bit!=folder.BookmarksEnd(); ++bit)
	{
		if(bit->browserSaved(browserOrd))
			bit->setBrowserFound(browserOrd);
	}

	for(fit=folder.FoldersBegin(); fit!=folder.FoldersEnd(); ++fit)
	{
		if(fit->browserSaved(browserOrd))
			fit->setBrowserFound(browserOrd);
		setBrowserAsSource(*fit, browserOrd);
	}
}
