
/*
 *  Diverse Bristol audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <math.h>

#include "brightoninternals.h"

#define SQRT_TAB_SIZE 128
double sqrttab[SQRT_TAB_SIZE][SQRT_TAB_SIZE];

initSqrtTab()
{
	int i, j, pindex;

	for (j = 0 ; j < SQRT_TAB_SIZE; j++)
	{
		for (i = 0 ; i < SQRT_TAB_SIZE; i++)
		{
			if ((sqrttab[i][j] = sqrt((double) (i * i + j * j))) == 0)
				sqrttab[i][j] = 0.0001;
		}
	}
}

static int
brightonInitBitmap(brightonBitmap *bitmap, int value)
{
	register int index, total = bitmap->width * bitmap->height;

	for (index = 0; index < total; index++)
		bitmap->pixels[index] = value;
}

/*
 * This is dead slow.
 */
brightonWorldChanged(brightonWindow *bwin, int width, int height)
{
	float aspect;

printf("brightonWorldChanged(%i %i %i %i)\n", width, height,
bwin->width, bwin->height);
	/*
	 * We need to maintain aspect ratio. First take the new width, and build
	 * a corresponding height. If the height is outside the realms of the
	 * display then do the reverse. If, however, the width and height are
	 * within a reasonable amount of the desired aspect ratio, just use
	 * them.
	 */
	aspect = ((float) width) / height;

	if (bwin->height == height)
	{
		/*
		 * width has changed, calculate new height. If height is our max,
		 * for this screen, then let it be max, and calculate alternative
		 * width.
		 */
		if ((width / bwin->aspect)
			< ((brightonDisplay *) bwin->display)->height)
		{
			bwin->width = width;
			bwin->height = bwin->width / bwin->aspect;
		} else {
			bwin->height = ((brightonDisplay *) bwin->display)->height - 10;
			bwin->width = bwin->height * bwin->aspect;
		}
	} else {
		if ((height * bwin->aspect)
			< ((brightonDisplay *) bwin->display)->width)
		{
			bwin->height = height;
			bwin->width = height * bwin->aspect;
		} else {
			bwin->width = ((brightonDisplay *) bwin->display)->width - 10;
			bwin->height = bwin->width / bwin->aspect;
		}
	}
	if ((aspect / bwin->aspect > 0.98) && ((aspect / bwin->aspect) < 1.02))
	{
		bwin->width = width;
		bwin->height = height;
	} else {
		bwin->flags |= BRIGHTON_SET_SIZE;
	}

	brightonFreeBitmap(bwin, bwin->canvas);
	brightonFreeBitmap(bwin, bwin->render);
	brightonFreeBitmap(bwin, bwin->dlayer);
	brightonFreeBitmap(bwin, bwin->slayer);

	bwin->canvas = brightonCreateBitmap(bwin, bwin->width, bwin->height);
	bwin->dlayer = brightonCreateBitmap(bwin, bwin->width, bwin->height);
	bwin->slayer = brightonCreateBitmap(bwin, bwin->width, bwin->height);
	bwin->render = brightonCreateBitmap(bwin, bwin->width, bwin->height);
	/*
	 * Our device and shadow layer need to be initialised with -1
	 */
	brightonInitBitmap(bwin->dlayer, -1);
	brightonInitBitmap(bwin->slayer, -1);

	bwin->lightX = -1000;
	bwin->lightY = -1000;
	bwin->lightH = 3000;
	bwin->lightI = 0.95;

	/*
	 * We now have a configured size. We need to render the surface onto the
	 * window background, and render the image if we have any configured 
	 * blueprints. These are rendered onto our canvas.
	 */
	if (bwin->app->flags & BRIGHTON_STRETCH)
		brightonStretch(bwin, bwin->surface, bwin->canvas, 0, 0,
			bwin->width, bwin->height, bwin->app->flags);
	else
		brightonTesselate(bwin, bwin->surface, bwin->canvas, 0, 0,
			bwin->width, bwin->height, bwin->app->flags);

	brightonStretch(bwin, bwin->image, bwin->canvas, 0, 0,
		bwin->width, bwin->height, 0);

	brightonRender(bwin, bwin->canvas, bwin->render, 0, 0,
		bwin->width, bwin->height, 0);

	bwin->flags |= BRIGHTON_NO_DRAW;

	if  (bwin->app)
	{
		int panel;
		brightonEvent event;

		for (panel = 0; panel < bwin->app->nresources; panel++)
		{
			/*
			 * We need to configure the device to resize itself
			 */
			event.command = BRIGHTON_RESIZE;
			event.x = (int) bwin->app->resources[panel].x
				* bwin->width / 1000;
			event.y = (int) bwin->app->resources[panel].y
				* bwin->height / 1000;
			event.w = (int) bwin->app->resources[panel].width
				* bwin->width / 1000;
			event.h = (int) bwin->app->resources[panel].height
				* bwin->height / 1000;

			bwin->app->resources[panel].configure(bwin,
				&bwin->app->resources[panel], &event);
		}
	}

	bwin->flags &= ~BRIGHTON_NO_DRAW;

	brightonFinalRender(bwin, 0, 0, bwin->width, bwin->height);

	/*
	 * Call the application to see if it wants to alter anything after the
	 * world changed.
	 */
	if (bwin->template->configure)
		bwin->template->configure(bwin);
}

/*
 * This is the final render algorithm.
 * It will take the window canvas, device layer and shadow layer at a given
 * offset, put them all onto the rendering plane, and update our screen 
 * interface library.
 */
brightonFinalRender(brightonWindow *bwin,
register int x, register int y, register int width, register int height)
{
	register int i, j, dy, index;
	register int *src = bwin->canvas->pixels;
	register int *dest = bwin->render->pixels;
	register int *device = bwin->dlayer->pixels;
	register int *shadow = bwin->slayer->pixels;

//printf("	finalRender(%i, %i, %i, %i)\n", x, y, width, height); 

	/* 
	 * Speed up on resizing - prevent excess redraws since we will have to
	 * scan the whole lot again anyway
	 */
	if (bwin->flags & BRIGHTON_NO_DRAW)
		return;

	for (j = y; j < (y + height); j++)
	{
		if (j >= bwin->height)
			break;

		dy = j * bwin->width;

		for (i = x; i < (x + width); i++)
		{
			if (i >= bwin->width)
				break;

			//index = i + j * bwin->width;
			index = i + dy;

			/*
			 * We have 3 layers, the topmost device layer, the middle shadow
			 * layer, and the lower canvas. The colors from each layer are
			 * moved into the render layer in order of priority. Where any
			 * layer has a negative brightonColor reference it is passed over
			 * to a lower plane. Other planes may be introduced at a later date
			 * as other features are incorporated.
			 */
			if (device[index] >= 0)
				dest[i + dy] = device[index];
			else if (shadow[index] >= 0)
				dest[i + dy] = shadow[index];
			else
				dest[i + dy] = src[index];
		}
	}
	/*
	 * Finally call the B library, of which currently only one verstion, the
	 * B11 interface to X11.
	 */
	BDrawArea(bwin->display, bwin->render, x, y, width, height, x, y);
}

/*
 * A tesselation algorithm, takes a bitmap and tiles it onto the specified area.
 */
brightonTesselate(register brightonWindow *bwin, register brightonBitmap *src,
register brightonBitmap *dest, register int x, register int y,
register int width, register int height, int direction)
{
	register int i, j, dy, w, h;
	register brightonPalette *palette = bwin->display->palette;
	register int *pixels;

	if ((src == 0) || (dest == 0))
		return;

//printf("brightonTesselate()\n");

	pixels = src->pixels;

	for (j = 0; j < height; j += src->height)
	{
		if (((j + y) >= dest->height) || (j >= height))
			break;

		dy = (j + y) * dest->width;

		/*
		 * Bounds check for reduction of last image to be rendered.
		 */
		if (j + src->height >= height)
			h = height - j;
		else
			h = src->height;

		for (i = 0; i < width; i += src->width)
		{
			if (((i + x) >= dest->width) || (i >= width))
				break;

			if (i + src->width >= width)
				w = width - i;
			else
				w = src->width;

			brightonRender(bwin, src, dest, i + x, j + y, w, h, direction);
		}
	}
}

/*
 * This is a render algorithm. Takes a bitmap and copies it to the
 * brightonBitmap area. We should do some bounds checking first?
 */
brightonRender(register brightonWindow *bwin, register brightonBitmap *src,
register brightonBitmap *dest, register int x, register int y,
register int width, register int height, int direction)
{
	register int i, j, dy;
	register int *pixels;

	if ((src == 0) || (dest == 0))
		return;

//printf("	render %i %i %i %i, %i %i, %i %i\n", x, y, width, height,
//src->width, src->height, dest->width, dest->height);

	pixels = src->pixels;

	for (j = 0; j < height; j++)
	{
		if (((j + y) >= dest->height) || (j >= height))
			break;

		dy = (j + y) * dest->width + x;

		for (i = 0; i < width; i++)
		{
			if (((i + x) >= dest->width) || (i >= width))
				break;

			dest->pixels[i + dy] = src->pixels[i + j * src->width];
		}
	}
}

//#define SMOOTHING

/*
 * This is a scaling render algorithm. Takes a bitmap and speads it across
 * the brightonWindow canvas area. We should do some bounds checking first.
 */
brightonStretch(register brightonWindow *bwin, brightonBitmap *src,
register brightonBitmap *dest, register int x, register int y,
register int width, register int height, int direction)
{
	register float i, j, x1 = 0;
	register int y1 = 0, cindex, drawcount = 0, ix1;
	register brightonPalette *palette = bwin->display->palette;
	register int *pixels;

	if ((src == 0) || (dest == 0))
		return;

//printf("	stretch %i %i %i %i, %i %i: flags %x\n", x, y, width, height,
//src->width, src->height, direction);

	pixels = src->pixels;
	/*
	 * Do not support rendering outside of the panel area, excepting where we
	 * go over outer edges.
	 */
	if ((x < 0) || (x >= bwin->width) || (y < 0) || (y >= bwin->height))
		return(0);

#ifdef SMOOTHING
	if (y == 0)
	{
		if (height == dest->height)
			height = height - 2;
		y = 1;
	}
	if (x == 0)
	{
		x = 1;
		if (width == dest->width)
			width = width - 2;
	}
#endif /* smoothing */

	/*
	 * We now have the test bitmap, and it should have generated a palette.
	 * render the bits onto our background, and copy it over.
	 */
	for (i = y; i < (y + height); i++)
	{
		switch (direction & BRIGHTON_DIRECTION_MASK) {
			default:
			// Default case and reverse take same values here
			case BRIGHTON_REVERSE:
				y1 = (int) ((i - y) * src->height / height) * src->width;
				break;
			case BRIGHTON_VERTICAL:
				y1 = (i - y) / height * src->width;
				break;
			case (BRIGHTON_VERTICAL|BRIGHTON_REVERSE):
				if (direction & BRIGHTON_HALF_REVERSE)
				{
					y1 = (i - y) / height * src->width;
					if (y1 < src->width / 2)
						y1 += src->width / 2;
					else
						y1 -= src->width / 2;
				} else {
					y1 = src->width - (i - y) / height * src->width;
				}
				break;
		}

		for (j = x; j < (x + width); j++)
		{
			switch (direction & BRIGHTON_DIRECTION_MASK) {
				default:
				// Default case and reverse take same values here
				case 0:
					x1 = y1 + (j - x) * src->width / width;
					break;
				case BRIGHTON_REVERSE:
					/*
					 * This is for button operations, and is only "half"
					 * reversed. We should make this a rendering option?
					 */
					if (direction & BRIGHTON_HALF_REVERSE)
					{
						x1 = (j - x) / width * src->width;
						if (x1 < src->width / 2)
							x1 += y1 + src->width / 2;
						else
							x1 += y1 - src->width / 2;

						if (x1 >= src->width * src->height)
							x1 = src->width * src->height - 1;
					} else {
						x1 = y1 + src->width - (j - x) * src->width / width;
					}
					break;
				case BRIGHTON_VERTICAL:
				case (BRIGHTON_VERTICAL|BRIGHTON_REVERSE):
					x1 = y1
						+ (int) ((j - x) * src->height / width) * src->width;
					break;
			}

			ix1 = x1;
//printf("r1: %f %f %i %f %i\n", i, j, y1, x1, ix1);

#ifndef SMOOTHING
			/* 
			 * Set pixel. Since we are using brightonBitmaps we do not have to
			 * go around using X library calls to set the pixel states.
			 */
			if ((palette[pixels[ix1]].red != 0) || 
				(palette[pixels[ix1]].green != 0)  ||
				(palette[pixels[ix1]].blue != 0xffff))
			{
				//printf("filling in %i from %i (%i %i %i)\n",
				//	(int) ((y + i) * dest->width + j + x), ix1,
				//	dest->width * dest->height, src->width * src->height, ix1);
				dest->pixels[(int) (i * dest->width + j)]
					= src->pixels[ix1];
			}
#else /* smoothing else */
#define isblue(x,y,z) ((y[z[x]].red == 0) && \
				(y[z[x]].green == 0) && \
				(y[z[x]].blue == 0xffff))
			/*
			 * We are only going to smooth from blueprint edges, not within
			 * the blueprint itself?
			if ((palette[pixels[ix1]].red == 0) &&
				(palette[pixels[ix1]].green == 0) &&
				(palette[pixels[ix1]].blue == 0xffff))
			 */
			if ((ix1 == 0) || (ix1 == src->width * src->height))
				continue;

			if (isblue(ix1 - 1, palette, pixels)
				&& isblue(ix1 + 1, palette, pixels))
				continue;
			else {
				int r, g, b;

printf("smoothing at %i, %f %f\n", ix1, i, j);
/*
continue;
*/
				if (isblue(ix1, palette, pixels))
					g = palette[pixels[(int) (i * dest->width + j)]].green * 7 / 8;
				else
					g = palette[pixels[ix1]].green * 7 / 8;
				if (isblue(ix1 - 1, palette, pixels))
					g +=
					palette[dest->pixels[(int) (i * dest->width + j - 1)]].green
						/ 8;
				if (isblue(ix1 + 1, palette, pixels))
					g +=
					palette[dest->pixels[(int) (i * dest->width + j + 1)]].green
						/ 8;

				if (isblue(ix1, palette, pixels))
					r = palette[pixels[(int) (i * dest->width + j)]].red * 7 / 8;
				else
					r = palette[pixels[ix1]].red * 7 / 8;
				if (isblue(ix1 - 1, palette, pixels))
					r +=
					palette[dest->pixels[(int) (i * dest->width + j - 1)]].red
						/ 8;
				if (isblue(ix1 + 1, palette, pixels))
					r +=
					palette[dest->pixels[(int) (i * dest->width + j + 1)]].red
						/ 8;

				if (isblue(ix1, palette, pixels))
					b = palette[pixels[(int) (i * dest->width + j)]].blue * 7 / 8;
				else
					b = palette[pixels[ix1]].blue * 7 / 8;
				if (isblue(ix1 - 1, palette, pixels))
					b +=
					palette[dest->pixels[(int) (i * dest->width + j - 1)]].blue
						/ 8;
				if (isblue(ix1 + 1, palette, pixels))
					b +=
					palette[dest->pixels[(int) (i * dest->width + j + 1)]].blue
						/ 8;

				dest->pixels[(int) (i * dest->width + j)] =
					brightonGetGC(bwin, r, g, b);
			}
#endif /* smoothing */
		}
	}
}

static int
brightonround(float x)
{
	int ix = x;
	float fx = ix;

	if (x - -fx >= 0.5)
		return(ix + 1);

	return(ix);
}

/*
 * This could do with some XDrawPoints() optimisation as well.
 */
brightonRotate(register brightonWindow *bwin, register brightonBitmap *src,
register brightonBitmap *dest, register int dx, register int dy,
register int width, register int height, register double rotation)
{
	register int py, px;
	register double i, j;
	register int adjx, adjy, natx, naty;
	register double r, ho2, angle;
	register int x, y, ax, ay, index, cindex;
	register brightonPalette *palette = bwin->display->palette;
	register int *pixels;

//printf("brightonRotate(%i, %i, %i, %i)\n",
//dx, dy, src->width, src->height);

	if ((src == 0) || (dest == 0))
		return;

	pixels = src->pixels;
	ho2 = src->height / 2;
	/*
	 * Do not support rendering outside of the panel area, excepting where we
	 * go over outer edges.
	 */
	if ((dx < 0) || (dx >= bwin->width) || (dy < 0) || (dy >= bwin->height))
	{
		printf("bounds fail\n");
		return(0);
	}

	if (width & 0x01)
		width--;
	if (height & 0x01)
		height--;

	/*
	 * We now have the test bitmap, and it should have generated a palette.
	 * render the bits onto our background, and copy it over.
	 */
	for (py = 0; py < height; py+=1)
	{
		j = py * src->height / height;
		naty = j - ho2;

		if (py >= dest->height)
			break;

		for (px = 0; px < width; px++)
		{
			i = px * src->width / width;
			natx = i - ho2;

			if ((r = sqrttab[natx < 0?-natx:natx][naty < 0?-naty: naty]) > ho2)
				continue;

			/*
			 * Rotate this point, and then put them back into native coords.
			 * We need our own rounding algorithm - otherwise we get truncation.
			 */
			if (r < src->istatic) {
				x = brightonround(natx + ho2);
				y = brightonround(naty + ho2);
			} else if (r < src->ostatic) {
				/*
				 * Take the angle of this point to the origin.
				 */
				if (naty < 0.0) {
					angle = asin(natx / r) + rotation;
					adjx = r * sin(angle);
					adjy = -r * cos(angle);
				} else {
					angle = -asin(natx / r) + rotation;
					adjx = -r * sin(angle);
					adjy = r * cos(angle);
				}
				x = brightonround(adjx + ho2);
				y = brightonround(adjy + ho2);
			} else {
				x = brightonround(natx + ho2);
				y = brightonround(naty + ho2);
			}

			if ((x < 0) || (x >= src->height) || (y < 0) || (y >= src->height))
				continue;
			index = y * src->width + x;

			dest->pixels[(int) (((dy + py) * dest->width) + dx + px)]
				= src->pixels[(int) (y * src->width + x)];
		}
	}
}

