(**************************************************************************)
(*                   Cameleon                                             *)
(*                                                                        *)
(*      Copyright (C) 2002 Institut National de Recherche en Informatique et   *)
(*      en Automatique. All rights reserved.                              *)
(*                                                                        *)
(*      This program is free software; you can redistribute it and/or modify  *)
(*      it under the terms of the GNU General Public License as published by  *)
(*      the Free Software Foundation; either version 2 of the License, or  *)
(*      any later version.                                                *)
(*                                                                        *)
(*      This program is distributed in the hope that it will be useful,   *)
(*      but WITHOUT ANY WARRANTY; without even the implied warranty of    *)
(*      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the     *)
(*      GNU General Public License for more details.                      *)
(*                                                                        *)
(*      You should have received a copy of the GNU General Public License  *)
(*      along with this program; if not, write to the Free Software       *)
(*      Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA          *)
(*      02111-1307  USA                                                   *)
(*                                                                        *)
(*      Contact: Maxence.Guesdon@inria.fr                                *)
(**************************************************************************)

(** HTML doc generation. *)

open Dbf_types.Current

(** Make sure that we have all the needed dbms information
   and only information on these for each column. 
   Also sort tables and columns by name.*)
let check_schema schema dbms_list =
  let f_dbms c dbms =
    try
      ignore (List.assoc dbms#dbms c.col_dbms)
    with
      Not_found ->
	let cdbms = { col_type_sql = ("", None, None) ;
		      col_2ml = "";
		      col_ml2 = "";
		      col_key = None ;
		      col_default = None ;
		      col_atts = []
		    } 
	in
	c.col_dbms <- (dbms#dbms, cdbms) :: c.col_dbms
  in
  let dbms_types = List.map (fun d -> d#dbms) dbms_list in
  let f_column c =
    List.iter (f_dbms c) dbms_list ;
    c.col_dbms <- List.filter
	(fun (dbms,_) -> List.mem dbms dbms_types) c.col_dbms
  in
  let f_table t = 
    List.iter f_column t.ta_columns;
    t.ta_columns <- List.sort
	(fun c1 -> fun c2 -> compare c1.col_name  c2.col_name)
	t.ta_columns
  in
  List.iter f_table schema.sch_tables;
  schema.sch_tables <- List.sort
      (fun t1 -> fun t2 -> compare t1.ta_name  t2.ta_name)
      schema.sch_tables

let column_def_by_dbms c dbms =
  let gen = 
    match dbms#dbms with
      Dbf_types.Current.Odbc -> new Dbf_odbc.odbc_gen dbms
    | Dbf_types.Current.Mysql -> new Dbf_mysql.mysql_gen dbms
    | Dbf_types.Current.Postgres -> new Dbf_postgres.postgres_gen dbms
  in
  gen#column_def c

let generate_doc file dbms_list schema =
  let objdbms_list = List.map
      (fun d ->
	match d with
	| Odbc -> Dbf_odbc.spec
	| Mysql -> Dbf_mysql.spec
	| Postgres -> Dbf_postgres.spec
      )
      dbms_list
  in
  check_schema schema objdbms_list;

  Dbf_html_rep.print_file ~html:true file 
    (if !Dbf_args.with_html_header then
      (Dbf_messages.html_header 
	 (match !Dbf_args.title with None -> "" | Some s -> s))
    else
      ""
    )
    (if !Dbf_args.with_html_header then
      Dbf_messages.html_footer
    else
      ""
    )    
    schema
    objdbms_list
    column_def_by_dbms
  
  
