(**************************************************************************)
(*  The CDuce compiler                                                    *)
(*  Alain Frisch <Alain.Frisch@inria.fr> and the CDuce team               *)
(*  Copyright CNRS,INRIA, 2003,2004 (see LICENSE for details)             *)
(**************************************************************************)

open Ident

type t
val serialize: t Serialize.Put.f
val deserialize: t Serialize.Get.f

exception NonExhaustive of Types.descr
exception Constraint of Types.descr * Types.descr
exception ShouldHave of Types.descr * string
exception WrongLabel of Types.descr * label
exception UnboundId of id * bool
exception UnboundExtId of Types.CompUnit.t * id
exception ShouldHave2 of Types.descr * string * Types.descr
exception Error of string
exception Warning of string * Types.t

val empty_env: t

val register_types : Types.CompUnit.t -> t -> unit
  (* Register types of the environment for the pretty-printer *)

val enter_cu : U.t -> Types.CompUnit.t -> t -> t 
val find_cu  : U.t -> t -> Types.CompUnit.t

val find_value: id -> t -> Types.t
val enter_type: id -> Types.t -> t -> t
val iter_values: t -> (id -> Types.t -> unit) -> unit

val typ: t -> Ast.ppat -> Types.Node.t
val pat: t -> Ast.ppat -> Patterns.node

val dump_types: Format.formatter -> t -> unit
val dump_ns: Format.formatter -> t -> unit
val set_ns_table_for_printer: t -> unit


val type_schema: t -> U.t -> string -> t

val type_ns : t -> U.t -> Ns.t -> t

val type_keep_ns : t -> bool -> t

val type_expr: t -> Ast.pexpr -> Typed.texpr * Types.descr

val type_defs: t -> (Location.loc * U.t * Ast.ppat) list -> t

val type_let_decl: t -> Ast.ppat -> Ast.pexpr -> 
  t * Typed.let_decl * (id * Types.t) list

val type_let_funs: t -> Ast.pexpr list -> 
  t * Typed.texpr list * (id * Types.t) list
  (* Assume that all the expressions are Abstractions *)


val get_schema_validator: string -> Ns.qname -> Value.t -> Value.t
val get_schema_names: t -> U.t list  (** registered schema names *)

(* Operators *)

type type_fun = Types.t -> bool -> Types.t

val register_op: string -> int -> (type_fun list -> type_fun) -> unit
val flatten: type_fun -> type_fun



module IType : sig
  type node

  val get_type: node -> Types.t

  val itype: Types.t -> node
  val delayed: unit -> node
  val link: node -> node -> unit

  val ior: node -> node -> node
  val iand: node -> node -> node

  val times: node -> node -> node
  val record: bool -> (node * node option) Ident.label_map -> node
  val xml: node -> node -> node
  val optional: node -> node

  val get_ct: node -> Types.t * node

(* Regular expression *)
  type regexp
  val rexp: regexp -> node
  val simplify_regexp: regexp -> regexp
  val eps: regexp
  val emp: regexp
  val seq: regexp -> regexp -> regexp
  val alt: regexp -> regexp -> regexp
  val star: regexp -> regexp
  val mix: regexp -> regexp
  val elem: node -> regexp
end

(* Forward definitions *)
val from_comp_unit: (Types.CompUnit.t -> t) ref
val has_comp_unit: (U.t -> bool) ref
val has_ocaml_unit: (U.t -> bool) ref
val has_static_external: (string -> bool) ref
val load_schema: 
  (U.t -> string -> (Types.t * (Value.t -> Value.t)) Ident.Env.t) ref
