(in-package :cmucl-interface)

(defmacro foreign-define (name value)
  `(eval-when (:load-toplevel :compile-toplevel)
     (defconstant ,name ,value)
     (export ',name)))

(defun cmu-typename (typename)
  (if (atom typename)
    (case typename
      ((double float int short char unsigned-int unsigned-short
	unsigned-char long unsigned-long void)
       typename)
      (signed-char 'char)
      (string 'c-string)
      (pointer 'unsigned-int)
      (otherwise (error "Unknown type name ~A.~%" typename)))
    (case (first typename)
      (pointer
       (let ((sec (second typename)))
	 (cond
	  ((member sec '(char unsigned-char))
	   'c-string) ;; strings for char pointers
	  ((or (member sec '(struct function void union))
	       (and (consp sec) (member (first sec) '(struct pointer))))
	   'unsigned-int) ;; ints for several kinds of pointers
	  (t ;; otherwise assume an array
	   `(* ,(cmu-typename sec)))))))))

;; ARRAY-DATA-ADDRESS is taken from the CMUCL user's manual
;; Alien objects chapter: accessing lisp arrays (8.7.5)
(declaim (inline array-data-address))
(defun array-data-address (array)
  "Return the physical address of where the actual data of an array is
stored.

ARRAY must be a specialized array type in CMU Lisp.  This means ARRAY
must be an array of one of the following types:

              double-float
              single-float
              (unsigned-byte 32)
              (unsigned-byte 16)
              (unsigned-byte  8)
              (signed-byte 32)
              (signed-byte 16)
              (signed-byte  8)
"
  (declare (type (or (array (signed-byte 8))
		     (array (signed-byte 16))
		     (array (signed-byte 32))
		     ;;#+signed-array (array (signed-byte 8))
		     ;;#+signed-array (array (signed-byte 16))
		     ;;#+signed-array (array (signed-byte 32))
		     (array (unsigned-byte 8))
		     (array (unsigned-byte 16))
		     (array (unsigned-byte 32))
		     (array single-float)
		     (array double-float))
		 array)
	   (optimize (speed 3) (safety 0))
	   (ext:optimize-interface (safety 3)))
  ;; with-array-data will get us to the actual data.  However, because
  ;; the array could have been displaced, we need to know where the
  ;; data starts.
  (lisp::with-array-data ((data array)
			  (start)
			  (end))
    (declare (ignore end))
    ;; DATA is a specialized simple-array.  Memory is laid out like this:
    ;;
    ;;   byte offset    Value
    ;;        0         type code (should be 70 for double-float vector)
    ;;        4         4 * number of elements in vector
    ;;        8         1st element of vector
    ;;      ...         ...
    ;;
    (let ((addr (+ 8 (logandc1 7 (kernel:get-lisp-obj-address data))))
	  (type-size
	   (let ((type (array-element-type data)))
	     (cond ((or (equal type '(signed-byte 8))
			(equal type '(unsigned-byte 8)))
		    1)
		   ((or (equal type '(signed-byte 16))
			(equal type '(unsigned-byte 16)))
		    2)
		   ((or (equal type '(signed-byte 32))
			(equal type '(unsigned-byte 32)))
		    4)
		   ((equal type 'single-float)
		    4)
		   ((equal type 'double-float)
		    8)
		   (t
		    (error "Unknown specialized array element type ~A"
			   type))))))
      (declare (type (unsigned-byte 32) addr)
	       (optimize (speed 3) (safety 0) (ext:inhibit-warnings 3)))
      (system:int-sap (the (unsigned-byte 32)
			(+ addr (* type-size start)))))))


(defmacro foreign-function (name arguments rettype c-name)
  ;; symlist : list of gensyms used as arguments
  ;; alien-type : list of alien types converted from arguments
  ;; array-args : list of t (if alien-type is array) or nil (otherwise)
  ;; arg-mapping : mapping between array args old and new gensyms, and types
  ;;    (list argument-sym new-sym type) e.g. (#:g1 #:g2 double)
  ;; lengths : list of gensyms for the lengths of arrays
  (let* ((symlist (mapcar #'(lambda (x) (declare (ignore x)) (gensym))
	 		  arguments))
	 (alien-types (mapcar #'cmu-typename arguments))
	 (array-args (mapcar #'(lambda (x)
				 (and (consp x) (eql (first x) '*)))
			     alien-types))
	 (has-arrays (remove-if #'null array-args))
	 ;;(extern-function (intern (concatenate 'string
	 ;;"_EXTERN_" (symbol-name name))))
	 (extern-function (gensym))
	 )
    `(eval-when (:load-toplevel :compile-toplevel)
    ;;`(progn
       (def-alien-variable (,c-name ,extern-function)
	   (function ,(cmu-typename rettype) ,@alien-types))
       (declaim (inline ,name))
       ,(if has-arrays
	  ;; hard version
	  `(defun ,name ,symlist
	     (declare (optimize (speed 3)
				;;(safety 0)
				(ext:inhibit-warnings 3)
				))
	     (unwind-protect
		 (progn
		   (ext:gc-off) ;; make sure arrays don't get moved by GC
		   (alien-funcall
		    ,extern-function
		    ,@(mapcar #'(lambda (x y)
				  (if x `(array-data-address ,y) y))
			      array-args symlist)))
	       (ext:gc-on)))
	  ;; no arrays; easy version
	  `(defun ,name ,symlist
	     (alien-funcall ,extern-function ,@symlist)))
       (export ',name))))
