#!/usr/bin/env python
# -*- Mode: python -*-
#
# Copyright (C) 2003 Mark Ferrell <xrxgrok@yahoo.com>
#
# -----------------------------------------------------------------------
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
#   1. Redistributions of source code must retain the above copyright notice,
#      this list of conditions and the following disclaimer.
#
#   2. Redistributions in binary form must reproduce the above copyright
#      notice, this list of conditions and the following disclaimer in the
#      documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
# EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
# ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
# -----------------------------------------------------------------------

synonyms = ['di', 'dif']
summary  = """Show differences between changesets"""
usage = """Usage: cscvs diff [[-P[branch.]cset1] [-P[branch.]cset2]] [format_options]

	If one -P argument is provided, and it resolves to only a single
	changeset, then the provided diff will be between that changeset and
	the changeset immediately prior.

	If one -P argument is provided, and it resolves to multiple changesets,
	then the provided diff will be between the first and last changesets in
	that range.

	If two -P arguments are provided, then the diff will be between the
	first changeset in each. This usage is provided to permit diff'ing
	between changesets on different branches.

	Syntax for specifying a changeset or range is as follows:

	-P[branch.][changeset]
			Specify branch and changeset(s) to list.
			Branch specification not compatible with branch-
			based changeset syntaxes below.
	   cset1:cset2	Between set1 and set2, including set1 and set2.
	   cset1::cset2	Between set1 and set2, excluding set1 and set2.
	   cset:	set and following changesets on the same branch.
	   set::	After set on the same branch.
	   :cset	set and previous changesets on the same branch.
	   ::cset	Before set on the same branch.
	   cset		Just set
	   branch	All changesets on the branch.
	   branch.	The last changeset on the branch.
	
format_options
  -c      Context diffs (default)
  -u      Unidiff format.
  -s      Short patch - one liner per file.
  -t      Top two diffs - last change made to the file.
  -N      Display new files
"""

def diff(config):
	import sys, os, getopt, string
	from RangeArgParser import RangeArgParser
	import StorageLayer
	SCM = StorageLayer.convertingImport('SCM')
	CVS = StorageLayer.convertingImport('CVS')

	catalog = SCM.Catalog.Catalog(config)
	csetSpec1 = None
	csetSpec2 = None
	cset1 = None
	cset2 = None

	csetSpecArgs = [arg[2:] for arg in config.args if arg[:2] == '-P']
	config.args  =     [arg for arg in config.args if arg[:2] != '-P']

	if len(csetSpecArgs) > 2:
		raise CVS.Usage
	if len(csetSpecArgs) >= 1:
		csetSpec1 = RangeArgParser(catalog)
		if not csetSpec1.parseArg(csetSpecArgs[0]): raise CVS.Usage
	if len(csetSpecArgs) == 2:
		csetSpec2 = RangeArgParser(catalog)
		if not csetSpec2.parseArg(csetSpecArgs[1]): raise CVS.Usage
		cset1 = csetSpec1.getChangesetIterator().next()
		cset2 = csetSpec2.getChangesetIterator().next()
	if len(csetSpecArgs) == 1:
		csets = csetSpec1.getChangesets()
		cset1 = csets[0]
		if len(csets) > 1: cset2 = csets[-1]

	diff_new = False
	if '-N' in config.args:
		diff_new = True
		del config.args[config.args.index('-N')]

	args = string.join(config.args)

	if cset1 and cset2:
		for filename in cset1.getFilesWithDifferingRevisionsIterator(cset2):
			oldRev = cset1.getLatestRevisionForFile(filename)
			newRev = cset2.getLatestRevisionForFile(filename)
			if oldRev is None or oldRev.type in (StorageLayer.RevisionDAO.REMOVE,
			                                     StorageLayer.RevisionDAO.PLACEHOLDER):
				if not diff_new: continue
				rev1 = '-D"1970/01/01 00:00:00"'
			else:
				rev1 = '-r%s' % oldRev.revision
			if newRev.type == StorageLayer.RevisionDAO.REMOVE:
				if not diff_new: continue
				rev2 = '-D"1970/01/01 00:00:00"'
			else:
				rev2 = '-r%s' % newRev.revision
			sys.stdout.write("cvs rdiff %s %s %s %s\n" % (args, rev1, rev2, os.path.join(config.prefix, newRev.filename)))
			os.system("cvs rdiff %s %s %s %s" % (args, rev1, rev2, os.path.join(config.prefix, newRev.filename)))

	elif cset1:
		for rev in cset1.getAllNewRevisionsIterator():
			orev = rev.predecessor
			if orev is None: rev1 = '-D"1970/01/01 00:00:00"'
			else:            rev1 = '-r%s' % orev.revision
			os.system("cvs rdiff %s %s -r%s %s" % (args, rev1, rev.revision, os.path.join(config.prefix, rev.filename)))
	else:
		raise RuntimeError, 'Must specify at least one "-P" option'

	catalog.close()


if __name__ == '__main__':
	import sys,os
	sys.path.append('modules')
	import CVS

	config = CVS.Config(os.path.curdir())
	config.cmd = 'diff'
	config.args = sys.args[1:]
	config.cat_path = os.path.join(config.topdir, "CVS/Catalog.sqlite")
	try:	diff(config)
	except CVS.Usage:
		raise RuntimeError(usage)

# tag: Mark Ferrell Tue Jun 17 17:27:14 CDT 2003 (cmds/diff.py)
#
