/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * uaddr.h: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: uaddr.h,v 1.6 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifndef __UADDR_H__
#define __UADDR_H__

/********************************************************************************************/
/* don't change the 2 following defines without discarding uaddr_ready and uaddr_partial DB */
/********************************************************************************************/
#define UADDR_NICK_SIZE 64
#define UADDR_DL 96

#define MAX_UADDR_PING 6	
#define DELAY_BETWEEN_PING 30
#define MAX_UNRESOLVED_IP 1000

/*********************************************/
/* uaddr_ready DB content. The key is a nick */
/*********************************************/
typedef struct
{
	int valid;								/* 1=slot is valid, 0= slot is empty */
	char nick[UADDR_NICK_SIZE];		/* nick */
	char dl_addr[UADDR_DL];				/* remote user addr */
	time_t register_time;				/* date when this entry has been created */
	unsigned int cyclic_dirty:2;		/* cyclic dirty flag */
												/*            0 and 1 mean the entry is clean. After a download try which has failed, the counter */
												/*            is increased. When the value is 2 or 3, this entry is temporarily invalid. The counter */
												/*            is cyclic, after 4 failed connection attempts, the entry becomes again valid */
												/*            the goal of this counter is to quickly detect user who has changed their IP */
} UADDR_READY;

/*****************************************************************************************/
/* uaddr_partial DB content. The key is a dl_addr (string of upto UADDR_DL-1 characters) */
/*****************************************************************************************/
typedef struct
{
	int valid;								/* 1=slot is valid, 0= slot is empty */
	char dl_addr[UADDR_DL];				/* remote user addr */
	unsigned char ping_counter;		/* number of time the ping command has been sent */
												/*            this counter is increased after each try to the nick associated to the dl_addr */
												/*            when nick is filled, it is set to 0. If the counter reaches MAX_UADDR_PING */
												/*            the entry is deleted */
} UADDR_PARTIAL;

/******************************************************************************/
/* to reduce duplicated code, the DCTC being the clock master is also the one */
/* performing UADDR action                                                    */
/******************************************************************************/
void create_uaddr_thread(void);

/********************************/
/* perform UADDR initialisation */
/********************************/
void init_uaddr(void);

/*************/
/* end UADDR */
/*************/
void exit_uaddr(void);

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* output: 0= ok    */
/*         1= error */
/******************************/
/* works on UADDR ready table */
/******************************/
int add_uaddr_entry(char *nickname, const char *dl_addr);

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* addr_dl is always uniq in the database */
/******************************************/
/* output: 0= ok    */
/*         1= error */
/********************************/
/* works on UADDR partial table */
/********************************/
int add_uaddr_entry_addr_dl_only(char *addr_dl);

/**********************************************************/
/* the following function adds an entry to the UADDR file */
/**********************************************************/
/* addr_dl is always uniq in the database */
/******************************************/
/* output: 0= ok    */
/*         1= error */
/****************************************************/
/* the function is smarter than the previous one    */
/* it also checks if the address does not exist in  */
/* the uaddr_ready DB and if it does not exist with */
/* a different port.                                */
/****************************************************/
int add_uaddr_entry_addr_dl_only_xtra(char *dl_addr);

/*************************************************************/
/* the following functions remove an entry to the UADDR file */
/*************************************************************/
void delete_uaddr_entry_by_name(char *nickname);		/* UADDR ready table */
void delete_uaddr_entry_by_addr(char *dl_addr);			/* UADDR partial table */

/*****************************************************/
/* check if the given address exists in the database */
/*****************************************************/
/* output: 1=yes, 0=no */
/***********************/
int check_uaddr_entry_by_addr(char *dl_addr);

/**********************************************************/
/* check if the given nickname has a known remote address */
/**********************************************************/
/* output: 1=yes, 0=no */
/******************************/
/* works on UADDR ready table */
/******************************/
int check_uaddr_entry_by_name(char *nickname);

/************************************************/
/* get the remote address of the given nickname */
/*********************************************************************************/
/* output: NULL (not found) or a GString (host:port) to free when no more useful */
/*********************************************************************************/
/* works on UADDR ready table */
/******************************/
GString *get_uaddr_dl_addr_by_name(char *nickname);


/*****************************************************************************/
/* this function scans the UADDR files and looks if there is something to do */
/*****************************************************************************/
/* this function must be called regularly */
/******************************************/
void uaddr_action();

/*******************************************/
/* output UADDR content into CMD_KB format */
/*************************************************/
/* works on UADDR ready and UADDR partial tables */
/*************************************************/
void list_uaddr_content(void);

/******************************************************************************************************************************/
/* the following function increases the cyclic error counter in the ping_counter of the UADDR entry having the given nickname */
/******************************************************************************************************************************/
/* output: -1= error                 */
/*         new value of ping counter */
/*************************************/
/* works on UADDR ready table */
/******************************/
int uaddr_increase_error_flag(char *nickname);

/***************************************************************************************************************************/
/* the following function resets the cyclic error counter in the ping_counter of the UADDR entry having the given nickname */
/***************************************************************************************************************************/
/* output: -1= error                 */
/*         new value of ping counter */
/*************************************/
/* works on UADDR ready table */
/******************************/
int uaddr_reset_error_flag(char *nickname);

void unode_alter_socket_tos(void);
#endif
