#!/usr/bin/perl

package Emcast::Handler;

use strict;

our $VERSION   = 0.0.2;


####################

use constant EMCAST_VERSION => 1;

use constant EMCAST_INIT    => 0;
use constant EMCAST_JOIN    => 2;
use constant EMCAST_LEAVE   => 3;
use constant EMCAST_SEND    => 4;
use constant EMCAST_RECV    => 5;
use constant EMCAST_GETOPT  => 6;
use constant EMCAST_SETOPT  => 7;

sub new
{
    my $invocant = shift;
    my $class = ref($invocant) || $invocant;
    my $self = {
	        'in'           	  => \*STDIN,
	        'out'          	  => \*STDOUT,
		'loopback'     	  => 0,
		'handle_loopback' => 0,
	        @_,
	    };
    return bless $self, $class;
}


sub loop
{
    while (loop_once() == 0) {};
    exit 0;
}


sub loop_once
{
    my $self = shift;
    my $buf;
    my $len;

    sysread ($self->in(), $buf, 2) == 2 or return -1;
    my $id = unpack ("n", $buf);

    print STDERR "loop once $id\n";

    if ($id == EMCAST_INIT)  # init
    {
	my $version;
	my $fifo_len;
	my $fifonam;

	sysread ($self->in(), $buf, 2) == 2 or return -1;
	$version = unpack ("n", $buf);

	($fifonam, $fifo_len) = $self->read_bytes ();
	return -1 if $fifo_len == -1;

	my $init = $self->init;
	my ($res, $our_version);
	if (defined($init))
	{
	    ($res, $our_version) = $init->($version);
	}
	else
	{
	    if ($version > EMCAST_VERSION)
	    {
		$our_version = EMCAST_VERSION;
	    }
	    else
	    {
		$our_version = $version;
	    }
	    $res = 0;
	}

	$self->write_byte ($res) or return -1;  # response
	$self->write_short (1) or return -1;    # version
	return 1 if $res != 0;

	my $fifo;
	open ($fifo, ">" . $fifonam) or return 1;
	$self->fifo($fifo);
    }

    elsif ($id == EMCAST_JOIN)
    {
	my ($buf, $len) = $self->read_bytes();
	return -1 if $len == -1;

	my $join = $self->join();
	my $res = 0;
	$res = $join->($buf) if (defined $join);

	$self->write_byte ($res) or return -1;
    }

    elsif ($id == EMCAST_LEAVE)
    {
	my $leave = $self->leave ();
	my $res = 0;
	$res = $leave->() if (defined $leave);

	$self->write_byte ($res) or return -1;
    }

    elsif ($id == EMCAST_SEND)
    {
	my ($buf, $len) = $self->read_bytes();
	return -1 if $len == -1;

	my $send = $self->send();
	my $res = 0;
	$res = $send->($buf, $len) if (defined $send);

	$self->write_byte ($res) or return -1;

	if ($self->handle_loopback() && $self->loopback())
	{
	    $self->recv ($buf, $len, '', 0);
	}
    }
    elsif ($id == EMCAST_GETOPT)
    {
	my ($optname, $optname_len) = $self->read_bytes();
	return -1 if $optname_len == -1;
	my $res;
	my $optval;
	my $optval_len;

	if ($self->handle_loopback() && $optname eq 'loopback')
	{
	    $optval = pack("N", $self->loopback());
	    $optval_len = 4;
	}
	else
	{
	    my $getopt = $self->getopt();
	    if (defined $getopt)
	    {
		($res, $optval, $optval_len) = $getopt->($optname);
		$optval_len = 0 if $res;
	    }
	    else
	    {
		$optval_len = 0;
		$res = 0;
	    }
	}

	my $buf = pack("Cna" . $optval_len, $res, $optval_len, $optval);
	my $len = 3 + $optval_len;
	syswrite($self->out(), $buf, $len) == $len or return -1;
    }

    elsif ($id == EMCAST_SETOPT)
    {
	sysread ($self->in(), $buf, 2) == 2 or return -1;
	my $optname_len = unpack ("n", $buf);
	sysread ($self->in(), $buf, 2) == 2 or return -1;
	my $optval_len  = unpack ("n", $buf);
	my $optname;
	sysread ($self->in(), $optname, $optname_len) == $optname_len 
	    or return -1;
	my $optval;
	sysread ($self->in(), $optval, $optval_len) == $optval_len 
	    or return -1;
	
	my $res;
	if ($self->handle_loopback() && $optname eq 'loopback')
	{
	    if ($optval_len == 4)
	    {
		my $loopback = unpack("N", $optval);
		$loopback = 1 if $loopback;
		$self->loopback($loopback);
		$res = 0;
	    }
	    else
	    {
		$res = 2;
	    }
	}
	else
	{
	    my $setopt = $self->setopt();

	    if (defined $setopt)
	    {
		$res = $setopt->($optname, $optval, $optval_len);
	    }
	    else
	    {
		$res = 1; # bad option
	    }
	}

	$self->write_byte ($res) or return -1;
    }
    else
    {
	return 1;
    }

    return 0;
}


sub write_byte
{
    my $self = shift;
    my $buf;
    my $len;

    $buf = pack ("C", $_[0]);
    $len = syswrite ($self->out(), $buf, 1);
    return $len == 1;
}


sub write_short
{
    my $self = shift;
    my $buf;
    my $len;

    $buf = pack ("n", $_[0]);
    $len = syswrite ($self->out(), $buf, 2);
    return $len == 2;
}


sub read_bytes
{
    my $self = shift;
    my $buf = '';
    my $len;
    my $buf_len;

    $len = sysread ($self->in(), $buf, 2);
    return (undef, -1) if $len != 2;
    $buf_len = unpack ("n", $buf);
    $len = sysread ($self->in(), $buf, $buf_len);
    return (undef, -1) if $len != $buf_len;
    return ($buf, $buf_len);
}


sub recv
{
    my ($self, $buf, $len, $from, $fromlen) = @_;

    my $b = pack("nnna" . $len . "a" . $fromlen, 
		 EMCAST_RECV, $len, $fromlen, $buf, $from);
    my $l = 6 + $len + $fromlen;
    syswrite ($self->fifo(), $b, $l) == $l or return -1;

    return $len;
}




####################

our %autoloaded = ( 'in'              => undef,
		    'out'             => undef,
		    'fifo'            => undef,
		    'loopback'        => undef,
		    'handle_loopback' => undef,
		    'init'            => undef,
		    'join'            => undef,
		    'leave'           => undef,
		    'send'            => undef,
		    'getopt'          => undef,
		    'setopt'          => undef,
		    );

our $AUTOLOAD;

sub AUTOLOAD 
{
    my $self = @_;
    my $class = ref $self;
    my $meth;

    ($meth = $AUTOLOAD) =~ s/^.*:://;

    return @_ unless (exists $autoloaded{$meth});
    
    eval <<EOSub;
sub $meth {
    my \$self = shift;
        
    if (\@_) 
    {
        my \$old = \$self->{"$meth"};
        \$self->{"$meth"} = shift;
        return \$old;
    }
    else 
    {
        return \$self->{"$meth"};
    }
}
EOSub
    
    goto &$meth;
}

1;
