/*
 * Receives data from the EMWIN DM-296 modem line and digests it.
 *
 * This is a hack by A. Maitland Bottoms to combine the emwinmcr.c
 * and the emwinmcs.c written by:
 *          Antonio Querubin, Capt
 *  	    199'th Weather Flight
 *          Hawaii Air National Guard
 *	    tony@hiang.af.mil
 *	    AH6BW
 *
 * into a program that is not network aware in any way :-(
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include <time.h>
#include <sys/stat.h>

#ifdef WIN32
#include <direct.h>
#include <winsock.h>
#else
#include <unistd.h>
#include <sys/param.h>
#ifndef __EMX__
#include <syslog.h>
#endif
#endif

#define VERSION "1.0"
FILE *openemwinstream2(char *devicename,char *settings);

#undef DEBUG 1

struct QBT {		/* http://iwin.nws.noaa.gov/emwin/winpro.htm */
  unsigned char preamble[6];
  unsigned char header[80];
  unsigned char data[1024];
  unsigned char trailer[6];
} QBT;
struct QBT qbtpacket;

int debuglevel = 0;

void toggledebug(int sig) {	/* SIGUSR1 signal handler. */
  char logbuf[80];
  debuglevel ^= 1;
  sprintf(logbuf,"Debug level = %d",debuglevel);
  syslog(0,logbuf);
}

int getqbtpacket(FILE *emwinstream, struct QBT *qbtpacket) {
  char qbtchar, qbtbuf[sizeof(QBT)];
  int nullcount = 0;
  char logbuf[80];

  memset(qbtpacket,'\0',sizeof(*qbtpacket));

  while(1) {

    switch(qbtchar=fgetc(emwinstream)) {
    case '\0' : /* Look for the null preamble. */
	nullcount++;
	break;
    case '/':  /* Possible start of a line header. */
	if (nullcount >= 6) {
	  if (debuglevel) {
            sprintf(logbuf,"Null count = %d",nullcount);
            syslog(0,logbuf);
	  }
	  ungetc(qbtchar, emwinstream);
	  if (fgets(qbtbuf,sizeof(qbtbuf),emwinstream) != NULL) {
	    if (debuglevel) {
              sprintf(logbuf,"%d chars in header",strlen(qbtbuf));
              syslog(0,logbuf);
	    }
	    if (strlen(qbtbuf) == sizeof(qbtpacket->header)) {
	      strncpy(qbtpacket->header,qbtbuf,sizeof(qbtpacket->header));
	      if (fread(qbtpacket->data, sizeof(qbtpacket->data),
		  1, emwinstream) == 1) return(0);
	    }
	  }
	}
        if (debuglevel) {
          syslog(0,"Resynchronizing");
        }
    default:
	nullcount = 0;	/* Restart the null train search. */
	break;
    }
  }
} /* qbtpacket */

/* Listens on the EMWIN multicast address/port and reconstructs the
incoming file(s) or outputs the EMWIN QBT block to stdout.

Copyright (C) 1999 Antonio Querubin

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

--
Please send patches and/or bug reports to:  tony@lava.net

Contributions/bug fixes by:
  Julian Cowley
  Maitland Bottoms (AA4HS)

*/


/* Normally found in sys/param.h but LCC-Win32 doesn't have it. */
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

#define TRUE 1
#define FALSE 0

#if defined(WIN32) || defined(__EMX__)
/* This assumes a WeatherNode directory structure */
#define BASEDIR "\\weather\\wxdata\\"
#define TRACKDIR BASEDIR "rxtrack"
#define TMPDIR BASEDIR "rxtmp\\tmp"
#define OUTDIR BASEDIR "newfiles"
#define NULLDEVICE "NUL"
#define COPYCMD "copy %s %s"
#define PATHSEPARATOR "\\"
#else
#ifdef linux
#define TMPDIR "/var/tmp"
#else
#define TMPDIR P_tmpdir
#endif
#define OUTDIR "."
#define BASEDIR "/weather/wxdata/"
#define TRACKDIR BASEDIR "rxtrack"
#define SERIALPORT "/dev/ttyS1"
#define SERIALSETTINGS "9600,N,8,1"
#define NULLDEVICE "/dev/null"
#define COPYCMD "/bin/cp -f %s %s"
#define PATHSEPARATOR "/"
#endif

#ifndef UNZIPPATH
#define UNZIPPATH "unzip"
#endif
#define UNZIPPARAMS " -qoj %s -d %s"

#ifndef BEEPMODE
#define BEEPMODE 0
#endif

#ifndef OUTMODE
#define OUTMODE 2
#endif

#ifndef DEBUG
#define DEBUG 0
#endif

#define MAXBLOCKS 500

/* These are defined externally since they might be referenced by
   other functions. */
unsigned long pktcount =0 ;
unsigned long badpktcount = 0;
unsigned char *serialport = SERIALPORT;
unsigned char *serialsettings = SERIALSETTINGS;
unsigned char *outdir = OUTDIR;
unsigned char *basedir = BASEDIR;
unsigned char *trackdir = TRACKDIR;
unsigned char *tmpdir = TMPDIR;
unsigned char *unzippath = UNZIPPATH;
unsigned char *pathseparator = PATHSEPARATOR;
int outmode = OUTMODE;
int beepmode = BEEPMODE;
int debug = DEBUG;
int recompat=0;

struct PKTATTR {
  unsigned char filename[13];	/* Null-terminated. */
  unsigned int blocknum, totalblocks;
  unsigned long checksum;
  unsigned char datestamp[24];	/* Null-terminated. */
  unsigned char *data;
} pktattr;

#ifdef __EMX__
/* Under OS/2 the standard chdir() and getcwd() behave differently
   so we redefine these to their UNIX equivalents. */
#define chdir _chdir2
#define getcwd _getcwd2
#endif

#ifdef WIN32
/* perror() is useless with Windows sockets. */
#define psocketerror(S) \
  fprintf(stderr,"%s:  WSA error #%d\n", S, WSAGetLastError());
#else
#define psocketerror(S) perror(S);
#endif


/* Prints statistical data. */
void dumpstats(void) {
  fprintf(stderr,"\nPackets received = %lu, # of bad packets = %lu ",
       pktcount, badpktcount);
  if (pktcount) fprintf(stderr," (%f%%)", (float) badpktcount/pktcount*100);
  fprintf(stderr,"\n");
} /* dumpstats() */

/* Ctrl-C/SIGINT/SIGTERM handler. */
void timetodie(int sig) {
  dumpstats();
#ifdef WIN32
  /* Tell WinSock we're leaving */
  WSACleanup();
#endif
  /* Restore default handler and run it. */
  signal(sig,SIG_DFL);
  raise(sig);
  /* Shouldn't get this far but just in case... */
  exit(EXIT_SUCCESS);
} /* timetodie() */

/* SIGHUP handler. */
void resetstats(int sig) {
  dumpstats();
  pktcount = 0;
  badpktcount = 0;
  signal(sig,SIG_IGN); /* We want to continue. */
  signal(sig,resetstats);
} /* resetstats() */

/* Parses and checks an EMWIN QBT block.
   Return values: -1 if any check fails, 0 otherwise.
   The attr structure is filled with the parsed block values if all
   checks are successful.  Otherwise the content of the structure is
   undefined. */
int checkblock(const struct QBT *blockp, struct PKTATTR *attr) {
  unsigned char tempbuf[sizeof(blockp->header) + 1];
  int i;
  unsigned char filename[13];
  unsigned long ourchecksum;
  time_t filetime;
  struct tm timest;
  unsigned char apm[3];

  /* The header isn't null-terminated so we have to add a NULL before
     parsing with sscanf. */
  memcpy(tempbuf,blockp->header,sizeof(blockp->header));
  tempbuf[sizeof(tempbuf)-1] = '\0';
  if (sscanf(tempbuf,"/PF%12c/PN %u /PT %u /CS %lu /FD%23c",
             filename, &(attr->blocknum), &(attr->totalblocks),
             &(attr->checksum), attr->datestamp) != 5) {
    fprintf(stderr,"Bad header.\n");
    return (-1);
  }
  filename[sizeof(filename)-1] = '\0';
  attr->datestamp[sizeof(attr->datestamp)-1] = '\0';

  /* Do a sanity check of the header. */

  /* Strip any trailing white-space from the filename. */
  if (sscanf(filename,"%s",attr->filename) != 1) {
    fprintf(stderr,"Missing filename.\n");
    return (-1);
  }
  for (i=0; i<strlen(attr->filename); i++) {
    if (!(isalnum(attr->filename[i]) || attr->filename[i] == '.')) {
      fprintf(stderr,"Invalid filename.\n");
      return (-1);
    }
  }

  /* Range check the block numbers. */
  if (attr->blocknum < 1 || attr->blocknum > attr->totalblocks ||
      attr->totalblocks > MAXBLOCKS) {
    fprintf(stderr,"Bad block number(s).\n");
    return (-1);
  }

  /* Validate the datestamp by trying to convert it to a time value. */
  memset(&timest,'\0',sizeof(timest));
  if (sscanf(attr->datestamp,"%u/%u/%u %u:%u:%u %3s",
             &timest.tm_mon,&timest.tm_mday,&timest.tm_year,
             &timest.tm_hour,&timest.tm_min,&timest.tm_sec,apm) != 7) {
    fprintf(stderr,"Bad date string.\n");
    return (-1);
  }
  if (strcmp(apm,"AM") && strcmp(apm,"PM") && strcmp(apm,"GMT")) {
    fprintf(stderr,"Bad date/timezone string.\n");
    return (-1);
  }
  /* Setup the time structure. */
  timest.tm_mon--;
  /* Adjust the hour for AM/PM. */
  if (timest.tm_hour == 12 && apm[0] == 'A') timest.tm_hour = 0;
  else if (timest.tm_hour < 12 && apm[0] == 'P') timest.tm_hour += 12;
  timest.tm_year -= 1900;
  if ((filetime=mktime(&timest)) == -1) {
    fprintf(stderr,"Bad date/time.\n");
    return (-1);
  }

  fprintf(stderr,"%s, # %u of %u, %s",
          attr->filename, attr->blocknum, attr->totalblocks,
          ctime(&filetime));

  /* Checksum the data portion. */
  for (i=0, ourchecksum=0;
       i<sizeof(blockp->data);
       ourchecksum += (unsigned int) blockp->data[i++]);
  if (ourchecksum != attr->checksum) {
    fprintf(stderr,"Checksum = %lu, should be %lu.\n",
            ourchecksum, attr->checksum);
    return (-1);
  }

  attr->data = (unsigned char *)blockp->data;
  return (0);
} /* checkblock() */

/* Save the EMWIN block.
   Return values:
     -1=failure
      0=block saved
      1=file completed by this block
   savedfilepath is set to point at the full path to the saved output
   file. */
int saveblock(const struct PKTATTR *pktp, unsigned char *savedfilepath) {
  FILE *trackfp, *datafp = NULL;
  int i, n, newfile, datasize;
  unsigned char trackrec[sizeof(pktp->datestamp)+MAXBLOCKS],
                trackpath[MAXPATHLEN], datapath[MAXPATHLEN];

  /* Construct the full file paths. */
  sprintf(datapath, "%s" PATHSEPARATOR "%s", tmpdir, pktp->filename);
#if defined(WIN32) || defined(__EMX__)
  sprintf(trackpath, "%s" PATHSEPARATOR "%s", TRACKDIR, pktp->filename);
#else
  if (recompat) {
    sprintf(trackpath, "%s" PATHSEPARATOR "%s", trackdir, pktp->filename);
  } else {
  /* In UNIX use the same tmpdir to store the tracking file but
     add a '.track' suffix to the filename. */
  sprintf(trackpath, "%s.track", datapath);
  }
#endif

  newfile = FALSE;
  /* Index value into the track record for this block. */
  n = sizeof(pktp->datestamp) + pktp->blocknum - 1;

  /* Open the block tracking file first. */
  if ((trackfp=fopen(trackpath,"r+")) == NULL) newfile = TRUE;
  else {
    fgets(trackrec,sizeof(trackrec),trackfp);
    /* Drop the line terminator if it exists. */
    if (trackrec[strlen(trackrec)-1] == '\n')
      trackrec[strlen(trackrec)-1] = '\0';
    /* If the datestamp doesn't match then create a new file. */
    if (strncmp(pktp->datestamp,trackrec,sizeof(pktp->datestamp)-1)) {
      newfile = TRUE;
      fclose(trackfp);
    }
    /* Make sure the block numbers and track record length make sense. */
    else if (n >= strlen(trackrec) ||
             sizeof(pktp->datestamp)+pktp->totalblocks !=
                    strlen(trackrec)) {
      fclose(trackfp);
      fprintf(stderr,"Track record length error.\n");
      return(-1);
    }
    /* Do nothing if we already have the block. */
    else if (trackrec[n] == 'Y') {
      fclose(trackfp);
      return(0);
    }
    /* If the data file can't be opened, assume it must be created. */
    else if ((datafp=fopen(datapath,"r+b")) == NULL) {
      newfile = TRUE;
      fclose(trackfp);
    }
  }

  if (newfile) {
    if ((datafp=fopen(datapath,"w+b")) == NULL) {
      perror("saveblock, fopen data file");
      return(-1);
    }
    /* Construct a new track record.  It consists of the file
       datestamp, a comma, and a series of . or Y characters where Y
       indicates the block has already been received. */
    memcpy(trackrec,pktp->datestamp,sizeof(pktp->datestamp));
    trackrec[sizeof(pktp->datestamp)-1] = ',';
    memset(trackrec+sizeof(pktp->datestamp), '.', pktp->totalblocks);
    trackrec[sizeof(pktp->datestamp)+pktp->totalblocks] = '\0';
    if ((trackfp=fopen(trackpath,"w+")) == NULL) {
      perror("saveblock, fopen track file");
      return(-1);
    }
    fputs(trackrec,trackfp);
  }

  /* Did we already save this block? */
  if (trackrec[n] != 'Y') {
    /* Write the block. */
    if (fseek(datafp, sizeof(qbtpacket.data)*(pktp->blocknum - 1),
        SEEK_SET)) {
      perror("saveblock, fseek");
      return(-1);
    }

    if ((pktp->blocknum == pktp->totalblocks) && 
        ((strstr(pktp->filename,".TXT") != NULL) || 
         (strstr(pktp->filename,".HTM") != NULL))) {
      /* If this is the last block of a text file, truncate the trailing
         null characters by changing the number of bytes to write. */
      for(datasize=sizeof(qbtpacket.data); datasize > 0; datasize--)
        if (pktp->data[datasize-1] != '\0') break;
    }
    else datasize=sizeof(qbtpacket.data);
    if (fwrite(pktp->data, datasize, 1, datafp) != 1) {
      perror("saveblock, fwrite");
      return(-1);
    }

    /* Update the track record. */
    trackrec[n] = 'Y';
    rewind(trackfp);
    fputs(trackrec,trackfp);
  }
  fclose(datafp);
  fclose(trackfp);

  /* Check the track record to see if we have all the parts. */
  for (i=sizeof(pktp->datestamp);
       i<sizeof(pktp->datestamp) + pktp->totalblocks; i++)
  if (trackrec[i] != 'Y') return(0);

  strcpy(savedfilepath,datapath);
  return(1);
} /* saveblock() */

void showusage(void) {
  fprintf(stderr,
    "Usage:  emwinmcr [-o output-directory]"
#if !defined(WIN32) && !defined(__EMX__)
    " [-t temp-directory]"
#endif
    "\n"
    "                 [-m output-mode] [-b] [-d] [-h]\n"
    "Options:\n"
    "-p serialport        source of incoming data\n"
    "                     (default = /dev/ttyS1).\n"
    "-s serialsettings    configuration of serial port\n"
    "                     (default = 9600,N,8,1).\n"
    "-o output-directory  Directory to place completed files\n"
    "                     (default = %s).\n"
#if !defined(WIN32) && !defined(__EMX__)
    "-w                   Use RealEmwin compatable directory structure\n"
    "-t temp-directory    Directory to place temporary files\n"
    "                     (default = %s).\n"
#endif
    "-m output-mode       0=receive only.\n"
    "                     1=send packets to stdout only.\n"
    "                     2=save completed files (default).\n"
#ifndef WIN32
    "                     3=same as 2 but runs in daemon mode.\n"
#endif
    "-b                   Beep on bad packets.\n"
    "-d                   Toggles diagnostic mode (default = %s).\n"
    "-u                   Path to unzip command (default = %s).\n"
    "-h                   Display options.\n",
#if !defined(WIN32) && !defined(__EMX__)
    outdir, tmpdir, debug ? "on":"off", unzippath);
#else
    outdir, debug ? "on":"off", unzippath);
#endif
} /* showusage() */

int main(const int argc, const char *argv[]) {
  FILE *emwinstream;
  int i, savestatus, pktisbad = FALSE, unzipstatus, copystatus;
  time_t t;
  unsigned char timebuf[26],
                tempbuf[sizeof(COPYCMD) + 2*MAXPATHLEN],
                completedfile[MAXPATHLEN];
  unsigned char *zisextp;
#ifndef WIN32
  pid_t	pid;
#endif

  /* Parse the command line arguments. */
  for (i=1; i < argc; i++) {
    if (argv[i][0] == '-') switch (argv[i][1]) {
      case 'o':  /* Output directory for completed files. */
        if (i+1 < argc) outdir = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  output directory.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
      case 'p':  /* Port for incoming data. */
        if (i+1 < argc) serialport = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  serial port.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
      case 's':  /* Serial settings. */
        if (i+1 < argc) serialsettings = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  serial settings.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
#ifndef WIN32
      case 'w':  /* compat enable. */
        recompat = 1;
        break;
      case 't':  /* Temp directory for incomplete files and block
                 tracking files. */
        if (i+1 < argc) tmpdir = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  tmp directory.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
#endif
      case 'm':  /* Output mode. */
        if (i+1 < argc) {
          if (sscanf(argv[i+1],"%u",&outmode) != 1) {
            fprintf(stderr,"Invalid mode.\n");
            exit(EXIT_FAILURE);
          }
#ifdef WIN32
          if (outmode > 2) {
#else
          if (outmode > 3) {
#endif
            fprintf(stderr,"Invalid mode.\n");
            exit(EXIT_FAILURE);
          }
        }
        else {
          fprintf(stderr,"Missing mode.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
      case 'b':  /* Beep enable. */
        beepmode = TRUE;
        break;
      case 'd':  /* Toggle the diagnostic output mode from the compile
                    default. */
        debug ^= 1;
        break;
      case 'u':  /* Path to unzip utility for ZIS files. */
        if (i+1 < argc) unzippath = (unsigned char *)argv[i+1];
        else {
          fprintf(stderr,"Missing option value:  unzip path.\n");
          exit(EXIT_FAILURE);
        }
        i++;
        break;
      case 'h':
  	showusage();
        exit(EXIT_SUCCESS);
        break;
      default:
        fprintf(stderr,"Unknown option %s\n", argv[i]);
        showusage();
        exit(EXIT_FAILURE);
    } /* end of switch (argv[i][1]) */
    else {
      showusage();
      exit(EXIT_FAILURE);
    }
  }

  if (recompat) { /* This may be of interest when using samba */
     basedir = strcat(outdir,"wxdata");
     trackdir = strcat(basedir,"rxtrack");
     tmpdir = strcat(basedir,"rxtmp/tmp");
     outdir = strcat(basedir,"newfiles");
  }
  /* Preliminary error checks before we redirect diagnostic output to 
     /dev/null. */
  if (outmode > 1) {
#ifndef __APPLE__
/* getcwd() coredumps on Apple OS X so we'll bypass this check for now. */
    /* Make sure the directories exist and get their full pathnames. */
    if (chdir(outdir) < 0) {
      perror("Can't access outdir");
      exit(EXIT_FAILURE);
    }
    outdir = getcwd(NULL,MAXPATHLEN);
    if (chdir(tmpdir) < 0) {
      perror("Can't access tmpdir");
      exit(EXIT_FAILURE);
    }
    tmpdir = getcwd(NULL,MAXPATHLEN);
#endif
#ifdef __EMX__
/* getcwd() on the EMX compiler insists on converting \ to / 
   so we convert them back. */
    {unsigned char *p;
     while((p=strchr(tmpdir,'/')) != NULL) *p = '\\';
     while((p=strchr(outdir,'/')) != NULL) *p = '\\';
    }
#endif
#if !defined(WIN32) && defined(__EMX__)
    /* Set the file creation mask so that downloaded files aren't
       executable nor writeable. */
    umask(umask(0) | 0133);
#endif
  }

  emwinstream=openemwinstream2(serialport,serialsettings);

  /* Now we can redirect diagnostic output. */
  if (!debug) {
    if (freopen(NULLDEVICE,"a",stderr) == NULL) {
      perror("Can't redirect stderr");
      exit(EXIT_FAILURE);
    }
  }

  fprintf(stderr,"emwinser, version " VERSION
                 ", compiled " __DATE__ " " __TIME__ "\n");

  /* Setup the signal handlers.  Some of these don't exist or may have
     problems under Windows, especially NT. */
#ifdef SIGINT
  signal(SIGINT,timetodie);
#endif
#ifdef SIGTERM
  signal(SIGTERM,timetodie);
#endif
#ifdef SIGHUP
  signal(SIGHUP,resetstats);
#endif
#ifdef SIGUSR1
  signal(SIGUSR1, toggledebug);	/* Set trap for SIGUSR1. */
#endif

  switch (outmode) {
  case 1:
    fprintf(stderr,"Sending packets to stdout.\n");
    break;
  case 2:
    fprintf(stderr,"Completed files will be moved to %s\n"
            "Filename will be written to stdout.\n",outdir);
    break;
#ifndef WIN32
  case 3: /* Daemon invokation */

    if((pid=fork()) < 0) {
      perror("Can't fork daemon process");
      exit(EXIT_FAILURE);
    }
    else if (pid != 0) exit(EXIT_SUCCESS); /* Parent goes bye-bye */

    /* Child continues */
#ifndef __EMX__
    /* become session leader */
    if (setsid() < 0) {
      perror("main, setsid");
      exit(EXIT_FAILURE);
    }
    openlog("emwinmcr",LOG_PID,LOG_USER);
    syslog(LOG_INFO,"EMWIN multicast receiver daemon starting");
#endif
    break;
#endif
  default: ;
  }

  while(TRUE) {

    if (pktisbad) {
      badpktcount++;
      if (beepmode) fprintf(stderr,"\a");
      pktisbad = FALSE;
    }

    getqbtpacket(emwinstream, &qbtpacket);
    if (debuglevel) {
	strncpy(tempbuf,qbtpacket.header,sizeof(qbtpacket.header));
	/* Truncate the trailing white-space at the end of
	the line so that it doesn't appear in the syslog. */
	for (i=sizeof(qbtpacket.header)-1; isspace(tempbuf[i]); i--)
          tempbuf[i] = '\0';
	syslog(0,tempbuf);
    }
    pktcount++;

    if (outmode < 3) { /* Timestamp the diagnostic output. */
      t = time(NULL);
      strcpy(timebuf,ctime(&t));
      /* Drop the \n at the end of the line. */
      timebuf[strlen(ctime(&t))-1] = '\0';
      fprintf(stderr,"%s:  ",timebuf);
    }

    if(checkblock(&qbtpacket,&pktattr) == -1) pktisbad = TRUE;
    else switch (outmode) {
      case 1:
        fwrite(&qbtpacket, sizeof(qbtpacket), 1, stdout);
        break;
      case 2:
      case 3:
        if ((savestatus = saveblock(&pktattr,completedfile)) == 1) {

          unzipstatus = -1;
          copystatus = -1;
 
          zisextp = strstr(pktattr.filename,".ZIS");
          if (zisextp != NULL) {
            /* Special handling for compressed text files.  If the unzip
               operation fails, ignore the error and continue on as if it
               were a normal file. */
            sprintf(tempbuf, "%s" UNZIPPARAMS, 
                    unzippath, completedfile, outdir);
            unzipstatus = system(tempbuf);             
          }
          
          if (zisextp == NULL || unzipstatus) {
            /* Copy the new file to the destination directory. */
            sprintf(tempbuf, COPYCMD, completedfile, outdir);
            if ((copystatus=system(tempbuf))) perror("Copy failed");
          }

          /* Delete the temporary file and output the file name if unzip
             or copy succeeded. */
          if (unzipstatus == 0 || copystatus == 0) {
            if (remove(completedfile)) perror("Can't remove tempfile");
            if (outmode < 3) {
              if (zisextp != NULL && unzipstatus == 0) {
                /* For some reason, the uncompressed filename is in
                   lower-case. */
                for (i=0; pktattr.filename[i] != '\0'; i++) 
                  pktattr.filename[i] = tolower(pktattr.filename[i]);
                strcpy(zisextp,".txt");
              } 
              printf("%s\n",pktattr.filename);
              fflush(stdout);
            }
          }

        }
        else if (savestatus == -1) pktisbad = TRUE;
        break;
      default: ; 
    } /* end of switch (outmode) */

    fflush(stderr);	/* For http streaming of the diagnostic output. */
  }
} /* main() */
