#include <glib.h>
#include "edvtypes.h"
#include "edvmimetypes.h"


edv_mimetype_struct *EDVMimeTypeNew(
	edv_mimetype_class mt_class,	/* One of EDV_MIMETYPE_CLASS_* */
	const gchar *value,
	const gchar *type,              /* MIME Type name */
	const gchar *description	/* Short verbose description */
);
edv_mimetype_struct *EDVMimeTypeCopy(const edv_mimetype_struct *m);
void EDVMimeTypeDelete(edv_mimetype_struct *m);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new MIME Type.
 */
edv_mimetype_struct *EDVMimeTypeNew(
	edv_mimetype_class mt_class,	/* One of EDV_MIMETYPE_CLASS_* */
	const gchar *value,
	const gchar *type,		/* MIME Type name */
	const gchar *description	/* Short verbose description */
)
{
	edv_mimetype_struct *m = EDV_MIMETYPE(
	    g_malloc0(sizeof(edv_mimetype_struct))
	);
	if(m == NULL)
	    return(m);

	m->mt_class = mt_class;

	m->value = STRDUP(value);
	m->type = STRDUP(type);
	m->description = STRDUP(description);

	m->handler = EDV_MIMETYPE_HANDLER_COMMAND;

	m->command = NULL;
	m->command_name = NULL;
	m->total_commands = 0;

	m->access_time = 0l;
	m->modify_time = 0l;
	m->change_time = 0l;

	return(m);
}

/*
 *	Coppies the MIME Type.
 */
edv_mimetype_struct *EDVMimeTypeCopy(const edv_mimetype_struct *m)
{
	gint i, n;
	edv_mimetype_struct *tar;
	const edv_mimetype_struct *src = m;

	if(src == NULL)
	    return(NULL);

	tar = EDVMimeTypeNew(
	    src->mt_class,
	    src->value,
	    src->type,
	    src->description
	);
	if(tar == NULL)
	    return(NULL);

	n = EDV_MIMETYPE_TOTAL_ICON_STATES;
	for(i = 0; i < n; i++)
	{
	    tar->small_icon_file[i] = STRDUP(src->small_icon_file[i]);
	    tar->medium_icon_file[i] = STRDUP(src->medium_icon_file[i]);
	    tar->large_icon_file[i] = STRDUP(src->large_icon_file[i]);
	}

	tar->total_commands = src->total_commands;
	tar->command = (gchar **)g_malloc(
	    src->total_commands * sizeof(gchar *)
	);
	tar->command_name = (gchar **)g_malloc(
	    src->total_commands * sizeof(gchar *)
	);
	for(i = 0; i < src->total_commands; i++)
	{
	    tar->command[i] = STRDUP(src->command[i]);
	    tar->command_name[i] = STRDUP(src->command_name[i]);
	}

	return(tar);
}

/*
 *	Deletes the MIME Type.
 */
void EDVMimeTypeDelete(edv_mimetype_struct *m)
{
	gint i, n;

	if(m == NULL)
	    return;

	/* Delete icon file paths */
	n = EDV_MIMETYPE_TOTAL_ICON_STATES;
	for(i = 0; i < n; i++)
	{
	    g_free(m->small_icon_file[i]);
	    g_free(m->medium_icon_file[i]);
	    g_free(m->large_icon_file[i]);
	}

	/* Delete commands */
	n = m->total_commands;
	for(i = 0; i < n; i++)
	{
	    g_free(m->command[i]);
	    g_free(m->command_name[i]);
	}
	g_free(m->command);
	g_free(m->command_name);

	/* Delete value, type, and description */
	g_free(m->value);
	g_free(m->type);
	g_free(m->description);

	g_free(m);
}
