/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: player.cc,v 1.30.2.1 2003/10/06 19:26:39 dheck Exp $
 */
#include "player.hh"
#include "display.hh"
#include "game.hh"
#include "sound.hh"
#include "objects.hh"
#include "actors.hh"
#include "px/tools.hh"
#include <algorithm>
#include <iostream>
#include <cassert>

using namespace player;
using namespace std;
using namespace enigma;

namespace
{
    class PlayerInfo {
    public:
        PlayerInfo();

        string          name;
        Inventory       inventory;
        vector<Actor*>  actors;
        bool            out_of_lives;
        double          dead_dtime; // number of seconds the player is
                                    // already dead
    };

    typedef vector<PlayerInfo> PlayerList;

    struct RespawnInfo {
        RespawnInfo (Actor *a, double t)
        : actor(a), time_left(t)
        {}

        Actor *actor;           // The actor to respawn
        double time_left;       // Time left before respawning
    };


    struct LevelLocalData {
        // Functions
        LevelLocalData() : move_counter(0) {}

        void respawn_dead_actors(double dtime);
        void resurrect_actor (Actor *a);
        bool remove_extralife (Actor *a);

        // Variables
        vector<RespawnInfo> respawn_list;
        int                 move_counter; // counts movements of stones

    };
}



PlayerInfo::PlayerInfo ()
: out_of_lives(false),
  dead_dtime(0)
{}



//----------------------------------------
// Local variables
//----------------------------------------

namespace
{
    LevelLocalData leveldat;
    PlayerList players;         // this currently has always size 2
    unsigned   icurrent_player               = 0;
    bool       current_player_inhibit_pickup = false;
    bool       autoadd_yinyang               = false; // true -> always add it-yinyang to Inventory
}

void
LevelLocalData::respawn_dead_actors(double dtime)
{
    for (unsigned i=0; i<respawn_list.size(); ) {
        RespawnInfo &info = respawn_list[i];

        info.time_left -= dtime;
        if (info.time_left < 0) {
            info.actor->respawn();
            respawn_list.erase(respawn_list.begin()+i);
            continue;           // don't increment i
        }
        ++i;
    }
}

void
LevelLocalData::resurrect_actor (Actor *a)
{
    const double RESPAWN_TIME = 1.5;

    SendMessage(a, "resurrect");
    remove_extralife(a);
    respawn_list.push_back(RespawnInfo(a, RESPAWN_TIME));
}

bool
LevelLocalData::remove_extralife (Actor *a)
{
    Inventory *inv = player::GetInventory(a->int_attrib("player"));
    int        idx = inv->find("it-extralife");

    if (idx == -1) // no extralife found
        return false;

    delete inv->yield_item(idx);
    return true;
}


//----------------------------------------
// Inventory impl.
//----------------------------------------
unsigned const Inventory::max_items = 12;

Inventory::Inventory() : items()
{}

Inventory::~Inventory()
{
    clear();
}

void
Inventory::clear()
{
    px::delete_sequence(items.begin(), items.end());
    items.clear();
}

Item *
Inventory::get_item(int idx)
{
    return idx >= size() ? 0 : items[idx];
}

Item *
Inventory::yield_item(int idx)
{
    if (0 <= idx && idx < size())
    {
        Item *it = items[idx];
        items.erase(items.begin()+ idx);
        redraw();
        return it;
    }
    return 0;
}

Item *
Inventory::yield_first()
{
    return yield_item(0);
}

Item *
Inventory::remove_item(Item *wanted) {
    vector<Item*>::iterator e = items.end();
    for (vector<Item*>::iterator i = items.begin(); i != e; ++i) {
        if (*i == wanted) {
            items.erase(i);
            return wanted;
        }
    }
    return 0;
}

void
Inventory::add_item(Item *i)
{
    items.insert(items.begin(), i);
    redraw();
}

void
Inventory::redraw()
{
    display::GetStatusBar()->update_inventory(this);
}

void
Inventory::rotate(int dir)
{
    if (!items.empty())
    {
	if (dir==1)
	    std::rotate(items.begin(), items.begin()+1, items.end());
	else
	    std::rotate(items.begin(), items.end()-1, items.end());

        redraw();
    }
}

void
Inventory::activate_first()
{
    if (!items.empty())
    {
        Item *it = items[0];
        Actor *ac = 0;
        GridPos p;
        if (!players[icurrent_player].actors.empty()) {
            ac = players[icurrent_player].actors[0];
            p = GridPos(ac->get_pos());
        }

        switch (it->activate(ac, p)) {
        case world::ITEM_DROP:
            // only drop if no item underneath and actor allows it
            if (world::GetItem(p) == 0 && ac->can_drop_items()) {
                remove_item(it);
                redraw();
                world::SetItem(p, it);
                it->on_drop(ac);
            }
            break;
        case world::ITEM_DROP_AS_STONE:
            if (world::GetStone(p) == 0 && ac->can_drop_items()) {
                remove_item(it);
                redraw();
                world::SetItemAsStone(p, it);
            }
            break;
        case world::ITEM_KILL:
            remove_item(it);
            redraw();
            break;
        case world::ITEM_KEEP:
            break;
        }
    }
}

int
Inventory::find(const string& kind, int start_idx) {
    int size_ = size();
    for (int i = start_idx; i<size_; ++i) {
        if (get_item(i)->is_kind(kind))
            return i;
    }
    return -1;
}


//----------------------------------------
// Global functions
//----------------------------------------

void
player::NewGame(int nplayers, bool add_yinyang)
{
    assert(nplayers > 0);

    autoadd_yinyang = add_yinyang;
    players.clear();
    players.resize(nplayers);

    for (int i=0; i<nplayers; ++i) {
        Inventory *inv = GetInventory(i);

        inv->add_item(world::MakeItem("it-extralife"));
        inv->add_item(world::MakeItem("it-extralife"));
    }
}

void player::InitMoveCounter() { leveldat.move_counter = 0; }
int player::IncMoveCounter(int increment) {
    leveldat.move_counter += increment; return leveldat.move_counter;
}
int player::GetMoveCounter() { return leveldat.move_counter; }

void
player::NewWorld()
{
    // Clear up the inventories of all players: keep only extra lifes.
    for (unsigned iplayer=0; iplayer<players.size(); ++iplayer)
    {
        Inventory *inv = GetInventory(iplayer);
        int nextralifes=0;
        for (int i=0; i<inv->size(); ++i)
            if (inv->get_item(i)->is_kind("it-extralife"))
                nextralifes += 1;
        inv->clear();
        for (int i=0; i<nextralifes; ++i)
            inv->add_item(world::MakeItem("it-extralife"));

        if (autoadd_yinyang)
            inv->add_item(world::MakeItem("it-yinyang"));

        players[iplayer].actors.clear();
    }
    SetCurrentPlayer(0);
    leveldat = LevelLocalData();
}

void
player::LevelFinished()
{
    for (unsigned i=0; i<players.size(); ++i) {
        for (unsigned j=0; j<players[i].actors.size(); ++j) {
            Actor *a = players[i].actors[j];
            SendMessage(a, "disappear");
        }
    }
}

Inventory *
player::GetInventory(int iplayer)
{
    return &players[iplayer].inventory;
}
Inventory *
player::GetInventory(Actor *a)
{
    if (const Value *v = a->get_attrib("player"))
        return GetInventory(to_int(*v));
    return 0;
}

Item *
player::wielded_item (Actor *a) {
    if (player::Inventory *inv = player::GetInventory(a))
        return inv->get_item(0);
    return 0;
}

bool
player::wielded_item_is(Actor *a, const string &kind)
{
    if (player::Inventory *inv = GetInventory(a))
        if (Item *it = inv->get_item(0))
            return it->is_kind(kind);
    return false;
}


int
player::CurrentPlayer()
{
    return icurrent_player;
}

bool
player::IsCurrentPlayer(Actor *a) {
    return static_cast<int>(icurrent_player) == a->int_attrib("player");
}

void
player::SetCurrentPlayer(unsigned iplayer)
{
    if (iplayer >= players.size())
        fprintf(stderr, "SetCurrentPlayer: no such player %d\n", iplayer);
    else {
        icurrent_player = iplayer;
        display::GetStatusBar()->set_inventory(GetInventory(iplayer));
        if (!players[iplayer].actors.empty())
            if (Actor *a = players[iplayer].actors[0])
                display::SetReferencePoint (a->get_pos());
    }
}

unsigned 
player::NumberOfRealPlayers() 
{
    unsigned real_players = 0;

    for (unsigned i=0; i<players.size(); ++i) {
        if (!players[i].actors.empty()) {
            ++real_players;
        }
    }

    return real_players;
}

/*
 * Sets respawn positions for black or white actors
 */
void
player::SetRespawnPositions(GridPos pos, bool black) 
{
    px::V2 center = pos.center();

    for (unsigned i=0; i<players.size(); ++i) {
        vector<Actor *> &al = players[i].actors;

        for (unsigned j=0; j<al.size(); ++j) {
            const Value *val = al[j]->get_attrib(black ? "blackball" : "whiteball");
            if (val) {
                al[j]->set_respawnpos(center);
            }
        }
    }
}

/*
 * Remove respawn positions for black or white actors
 */
void
player::RemoveRespawnPositions(bool black) 
{ 
    for (unsigned i=0; i<players.size(); ++i) {
        vector<Actor *> &al = players[i].actors;

        for (unsigned j=0; j<al.size(); ++j) {
            const Value *val = al[j]->get_attrib(black ? "blackball" : "whiteball");
            if (val) {
                al[j]->remove_respawnpos();
            }
        }
    }
}

unsigned 
player::CountActorsOfKind(const char *kind) 
{
    unsigned count = 0;
    for (unsigned i=0; i<players.size(); ++i) {
        vector<Actor *> &al = players[i].actors;
        for (unsigned j=0; j<al.size(); ++j) {
            if (!kind || al[j]->is_kind(kind))
                ++count;
        }
    }
    return count;
}

void
player::Suicide()
{
    for (unsigned i=0; i<players.size(); ++i) {
        vector<Actor *> &al = players[i].actors;
        for (unsigned j=0; j<al.size(); ++j) {
            SendMessage(al[j], "shatter");
        }
    }
}

void
player::AddActor (int iplayer, Actor *a)
{
    assert(iplayer >= 0 && (unsigned)iplayer < players.size());

    // @@@ FIXME:  out_of_lives is set correctly, but I don't know
    // how to 'skip' an actor..
//     if (!players[iplayer].out_of_lives) {
        ReleaseActor(a);
        players[iplayer].actors.push_back(a);

        if ((unsigned)iplayer == icurrent_player)
            display::SetReferencePoint (a->get_pos());
        // display::FollowSprite(a->get_spriteid());
//     }
}

void
player::SwapPlayers()
{
    if (NumberOfRealPlayers() >= 2) {
        SetCurrentPlayer(1-icurrent_player);
    }
//     else {
//         fprintf(stderr, "No two players to swap!\n");
//     }
}

namespace {
    bool has_extralive(unsigned pl) {
        return players[pl].inventory.find("it-extralife") != -1;
    }

    bool resurrect_actor (unsigned pl, Actor *a) {
        assert(enigma::ConserveLevel); // no resurrection otherwise!

        bool has_life = has_extralive(pl);
        if (has_life) {
            leveldat.resurrect_actor(a); // = resurrect with delay
        }
        else {
            players[pl].out_of_lives = true;
        }

        return has_life;
    }

    void CheckDeadActors() {
        bool           toggle_player    = false;
        const unsigned NO_PLAYER        = UINT_MAX;
        unsigned       toggle_to_player = NO_PLAYER;
        bool           new_game         = false; // complete restart (new lives)
        bool           reset_level      = false; // reset items etc.
        unsigned       psize            = players.size();

        if (enigma::ConserveLevel) {
            bool all_players_dead = true;

            for (unsigned pl = 0; pl<psize; ++pl) {
                vector<Actor*>& actors = players[pl].actors;

                if (!actors.empty()) {
                    bool                     all_actors_dead = true;
                    vector<Actor*>::iterator a_end           = actors.end();

                    for (vector<Actor*>::iterator ai = actors.begin(); ai != a_end; ++ai) {
                        Actor *a = *ai;

                        if (!a->is_dead() || resurrect_actor(pl, a)) {
                            all_players_dead = all_actors_dead = false;
                        }
                    }

                    if (all_actors_dead) {
                        if (pl == icurrent_player)
                            toggle_player = true;
                    }
                    else {
                        if (toggle_to_player == NO_PLAYER)
                            toggle_to_player = pl;
                    }
                }
            }

        }
        else {                  // ConserveLevel == false
            for (unsigned pl = 0; pl<psize; ++pl) {
                vector<Actor*>& actors = players[pl].actors;

                if (!actors.empty()) {
                    vector<Actor*>::iterator a_end          = actors.end();
                    bool                     has_dead_actor = false;

                    for (vector<Actor*>::iterator ai = actors.begin(); ai != a_end; ++ai) {
                        Actor *a = *ai;

                        if (a->is_dead()) {
                            has_dead_actor = true;
                            if (has_extralive(pl)) {
                                leveldat.remove_extralife(a);
                                reset_level = true;
                            }
                            else {
                                players[pl].out_of_lives = true;
                                if (pl == icurrent_player)
                                    toggle_player = true;
                            }
                            break;
                        }
                    }

                    if (!has_dead_actor) { // does not have dead player
                        if (toggle_to_player == NO_PLAYER)
                            toggle_to_player = pl;
                    }
                }
            }
        }

        if (toggle_player) {
            if (toggle_to_player == NO_PLAYER)
                new_game = true;
            else
                SetCurrentPlayer(toggle_to_player);
        }

        if (reset_level)
            enigma::RestartLevel(); // should restart w/o scrolling
        else if (new_game)
            enigma::RestartGame();
    }


}


void
player::Tick(double dtime)
{
    // Update position of active actor for stereo sound and screen position
    if (!players[icurrent_player].actors.empty()) {
        Actor *ac = players[icurrent_player].actors[0];
        const world::ActorInfo &ai = *ac->get_actorinfo();
        sound::SetListenerPosition (ai.pos);
        display::SetReferencePoint (ai.pos);
    }

    // Respawn actors that have been dead for a certain amount of time
    leveldat.respawn_dead_actors(dtime);

    // Update the respawn list or restart the game when all actors are
    // dead and no extra lifes are left.
    CheckDeadActors();
}

void
player::InhibitPickup(bool flag)
{
    current_player_inhibit_pickup = flag;
}

namespace
{
    // returns pointer to inventory if actor may pickup sth
    //         0 otherwise

    Inventory *MayPickup(Actor *a) {
        int iplayer=-1;
        a->int_attrib("player", &iplayer);
        if (iplayer < 0 || (unsigned)iplayer >= players.size()) {
            //        cerr << "PickupItem: illegal 'player' entry\n";
            return 0;
        }

        if ((unsigned)iplayer==icurrent_player && current_player_inhibit_pickup) {
            // do not pick up items if current player doesn't want to
            return 0;
        }

        if (a->is_flying()) { // do not pick up items while flying
            return 0;
        }

        Inventory *inv = &players[iplayer].inventory;
        if (inv->is_full()) { // do not pickup if inventory is full
            return 0;
        }
        return inv;
    }
}

void
player::PickupItem(Actor *a, GridPos p)
{
    Inventory *inv = MayPickup(a);
    if (inv) {
        Item *item = world::YieldItem(p);
        if (item) {
            item->on_pickup(a);
            inv->add_item(item);
            sound::PlaySound("pickup");
        }
    }
}

void
player::PickupStoneAsItem(Actor *a, enigma::GridPos p) {
    Inventory *inv = MayPickup(a);
    if (inv) {
        world::Stone *stone = world::YieldStone(p);
        if (stone) {
            Item *item = world::ConvertToItem(stone); // disposes stone
            if (item) {
                inv->add_item(item);
                sound::PlaySound("pickup");
            }
        }
    }
}


void
player::ActivateItem()
{
    Inventory &inv = players[icurrent_player].inventory;
    inv.activate_first();
}

void
player::RotateInventory(int dir)
{
    sound::PlaySound("invrotate");
    Inventory &inv = players[icurrent_player].inventory;
    inv.rotate(dir);
}

