#include <gtk/gtk.h>
#include "entity.h"
#include "gtk-common.h"
#include "string.h"

/* In this file we have the great fun of parsing out the style * from an
 * attribute. * * Everything is sorta like gtk's style rcs. * * * COLOR
 * STYLE: * style="fg[norm{al}]=c, *        fg[{pre}li{ght}]=c, *
 * fg[sel{ected}]=c, *        fg[{in}sen{sitive}]=c, *
 * fg[on{action}]=c" * * Only the stuff NOT in {} is necessary when setting a 
 * color.  Of * course fg can be replaced with base, bg, dark, light, mid,
 * font, * text, or image. * I don't think people should use light, dark,
 * or mid... * 'c' can be any text color or a #hheexx-i-decimal color, and in
 * * the case of image[*]=file.pix file.pix needs to be in the path. * *
 * style="font[size]=#,font[type]=type,font[foundry]=foundry, *
 * font=fontstring" * * font=fontstring is platform specific.  For all others
 * we attempt to * find the closest font. */

/* Internal types of lvalues. */
#define EST_FG		1
#define EST_BG		2
#define EST_BASE	3
#define EST_DARK	4
#define EST_LIGHT	5
#define EST_MID		6
#define EST_FONT	7
#define EST_IMAGE	8


GHashTable *original_styles_ht;

GtkStyle *
rendgtk_rc_get_style (GtkWidget * widget)
{
    GtkStyle *style;
    int type;

    type = GTK_OBJECT_TYPE (GTK_OBJECT (widget));

    style = g_hash_table_lookup (original_styles_ht, gtk_type_name (type));
    if (!style) {
	style = gtk_rc_get_style (widget);

	if (!style)
	    style = gtk_style_copy (GTK_WIDGET (widget)->style);

	g_hash_table_insert (original_styles_ht, gtk_type_name (type), style);

	EDEBUG (("gtk-style", "added new style\n"));
    }

    return style;
}

static int
rendgtk_widget_style_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkStyle *style;
    GtkWidget *widget;
    GtkWidget *label;

    EDEBUG (("gtk-style", "in rendgtk_widget_style_attr_set\n"));

    widget = enode_get_kv (node, "top-widget");

    if (!widget)
	return TRUE;

    /* Build a new style. */
    style = rendgtk_style_parser (value, rendgtk_rc_get_style (widget));

    /* apply the new style. */
    gtk_widget_set_style (GTK_WIDGET (widget), style);


    label = enode_get_kv (node, "bottom-widget-label");
    if (label) {
	/* I guess styles are ref counted.... FIXME */
	style = rendgtk_style_parser (value, rendgtk_rc_get_style (label));

	/* apply the new style. */
	gtk_widget_set_style (GTK_WIDGET (label), style);
    }


    return TRUE;
}


/* Find the gtk index from the style ndx. */
int
rendgtk_style_nxd (GtkStateType * out_ndx, char *ndx)
{

    if ((strstr (ndx, "norm") && (((*out_ndx) = GTK_STATE_NORMAL) || 1))
	|| (strstr (ndx, "on") && ((*out_ndx) = GTK_STATE_ACTIVE))
	|| (strstr (ndx, "sel") && ((*out_ndx) = GTK_STATE_SELECTED))
	|| (strstr (ndx, "li") && ((*out_ndx) = GTK_STATE_PRELIGHT))
	|| (strstr (ndx, "sen") && ((*out_ndx) = GTK_STATE_INSENSITIVE))
	) {
	return TRUE;		/* Found an index. */
    }

    return FALSE;		/* Didn't find an index. */
}

static void
rendgtk_style_set_color (GtkStyle * style,
			 int lval_type, char *ndx, GdkColor * col)
{
    GtkStateType out_ndx;

    if (!rendgtk_style_nxd (&out_ndx, ndx))
	return;			/* Wasn't a valid index type. */


    /* Set the correct color. */
    if (EST_FG == lval_type)
	style->fg[out_ndx] = (*col);
    else if (EST_BG == lval_type)
	style->bg[out_ndx] = (*col);
    else if (EST_BASE == lval_type)
	style->base[out_ndx] = (*col);
    else if (EST_FONT == lval_type)
	style->text[out_ndx] = (*col);
    else if (EST_DARK == lval_type)
	style->dark[out_ndx] = (*col);
    else if (EST_LIGHT == lval_type)
	style->light[out_ndx] = (*col);
    else if (EST_MID == lval_type)
	style->mid[out_ndx] = (*col);
}

/* Attempt to make sense of the rule and apply in to the style. */
static void
rendgtk_widget_style_fill (GtkStyle * style, char *rule)
{
    char *cur,
    *ndx;
    char *lval,
    *rval;
    GdkColor col;
    int has_rval;		/* True if there is an rval. */
    int lval_type;

    lval = cur = rule;

    /* Need to find the '=' first.  It should always exist. */
    for (cur = rule; (*cur) != '\0' && (*cur) != '='; cur++) {
	/* NO OP. */
    }

    /* See if we even found a '='. */
    if ('\0' == (*cur)) {
	return;			/* Just ignore this rule. */
    }


  /*** Have an '=' operator, so far so good. ***/


    (*cur) = '\0';		/* Bust the string in two right around the =. 
				 */
    rval = (++cur);		/* Save off the right value. */



    /* Bust out the index value if there is one. */
    for (ndx = lval; (*ndx) != '[' && (*ndx) != '\0'; ndx++) {
	/* NO OP. */
    }


    /* See if there is a rval. */
    if ('\0' == rval[0])
	has_rval = FALSE;
    else
	has_rval = TRUE;


  /*** See what style type we are setting. ***/
    if ('[' == (*ndx)) {	/* looks like one that has an index. */
	(*ndx) = '\0';		/* bust the lval into type/index. */
	ndx++;			/* Point the index to the actual start of the 
				 * index. */


	/* See if we know about this lval type.  Play a trick to * set the
	 * type. This is all around nasty code. */
	if ((strstr (lval, "fg") && (lval_type = EST_FG))
	    || (strstr (lval, "bg") && (lval_type = EST_BG))
	    || (strstr (lval, "base") && (lval_type = EST_BASE))
	    || (strstr (lval, "dark") && (lval_type = EST_DARK))
	    || (strstr (lval, "mid") && (lval_type = EST_MID))
	    || (strstr (lval, "light") && (lval_type = EST_LIGHT))
	    || (strstr (lval, "font") && (lval_type = EST_FONT))	/* SAME. 
									 */
	    ||(strstr (lval, "text") && (lval_type = EST_FONT))	/* SAME. */
	    ||(strstr (lval, "image") && (lval_type = EST_IMAGE))
	    );			/* NO OP. */
	else
	    return;		/* Don't know this type. */

      /*** Search for index types that we know about. ***/

	/* Can look for color. */
	if (lval_type != EST_IMAGE && has_rval && gdk_color_parse (rval, &col)) {
	    rendgtk_style_set_color (style, lval_type, ndx, &col);
	} else if (EST_FONT == lval_type) {	/* Look for a font type. */
	} else {		/* Must be an image. */

	}
    } else {			/* Regular old equality. */

	/* See if we know about this lval type. */

	/* See if the rval is a known type. */
    }

}


/* Glorified POS style="" attribute parser. */
GtkStyle *
rendgtk_style_parser (EBuf * value, GtkStyle * style)
{
    char *cur,
    *rule;
    int i;


    if (ebuf_empty (value)) {
	return style;
    }

    if (style)
	style = gtk_style_copy (style);
    else
	return NULL;

    value = ebuf_new_with_ebuf (value);	/* Don't want to mess up original
					 * value. */

    rule = cur = value->str;

    /* First task is to bust up the value on ','s. */
    for (i = 0; i <= value->len; i++) {
	/* Find one style directive at a time. */
	if ('\0' == (*cur) || ',' == (*cur)) {
	    (*cur) = '\0';	/* Ensure a \0. */

	    /* Update the style with this rule. */
	    rendgtk_widget_style_fill (style, rule);

	    rule = cur + 1;	/* Start the next rule after the \0. * This
				 * maybe one past but the loop breaks out. */
	}
	cur++;			/* Continue down the string. */
    }

    ebuf_free (value);		/* Free our copy of the value. */

    return style;
}



void
rendgtk_type_widget_style_attr_register (Element * element)
{
    ElementAttr *e_attr;

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "style";
    e_attr->description = "The look and color of this widget.";
    e_attr->value_desc = "string";
    e_attr->possible_values = "Comma seperated style rules.";
    e_attr->set_attr_func = rendgtk_widget_style_attr_set;
    element_register_attrib (element, e_attr);

    original_styles_ht = g_hash_table_new (g_str_hash, g_str_equal);
}

