/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#define SELECTION_ATTR	(A_BOLD)

typedef void (*clist_item_delete_cb)(void *list_item);
typedef void (*clist_delete_cb)(void *clist);
typedef int (*clist_select_item_cb)(void *clist, void *list_item);
typedef int (*clist_unselect_item_cb)(void *clist, void *list_item);
typedef void (*clist_activate_item_cb)(void *clist, void *list_item);
typedef void (*clist_focus_item_cb)(void *clist, void *list_item);
typedef int (*clist_idle_func)(void *clist);
typedef int (*clist_enum_func)(void *user_data, handle_array_t **handles);
typedef int (*clist_filter_func)(object_handle_t handle, void *user_data);
typedef int (*clist_format_func)(object_handle_t handle, void *user_data, GPtrArray *text);

typedef enum {
	CLIST_TEXT_JUSTIFY_LEFT,
	CLIST_TEXT_JUSTIFY_CENTER,
	CLIST_TEXT_JUSTIFY_RIGHT
} clist_text_justification;

/**
 *	struct clist_column - describes a column in a clist
 *	@startx: the offset in the clist at which this column starts
 *	@width: the maximum width of the column
 *	@justification: where to place the strings (right, center, or left justified)
 *
 *	This structure allows for maintaining track of information pertaining to a column
 *	in a clist.
 */
struct clist_column {
	int startx;
	int width;
	clist_text_justification justification;
};

/**
 *	struct clist_item - describes one row in a selection list
 *	@user_data: whatever the user wants to associate with this row
 *	@delete_cb: the handler to call when a list_item is to be deleted
 *	@is_selected: TRUE if the item is currently selected
 *	@is_selectable: TRUE if the item can be selected
 *	@row_text: the array of text strings to display at the given offset in the row
 *
 *	This structure maintains information about a selection list item.
 */
struct clist_item {
	void *user_data;
	clist_item_delete_cb delete_cb;
	int is_selected;
	int is_selectable;
	char **row_text;
};

/**
 *	struct clist - contains information of a choice/selection list
 *	@win: the sub-window id
 *	@choices: the linked list of available choices (rows)
 *	@selections: the linked list of selected rows (in the order selected)
 *	@top_item: the address of the top item displayed in the selection window
 *	@focus_item: the address of the item with the current focus
 *	@idle_queue: list of clist_idle_func to run after processing clist events
 *	@delete_cb: the handler to call when a clist is to be deleted
 *	@select_cb: the handler to call when a list_item is selected
 *	@unselect_cb: the handler to call when a list_item is selected
 *	@activate_cb: the handler to call when a list_item is activated
 *	@focus_cb: the handler to call when a list_item receives focus
 *	@mode: selection mode (CLIST_MODE_MULTI or CLIST_MODE_SINGLE)
 *	@background: the clist background char/attribute pair
 *	@focus_color: the color-pair to give the focus_item
 *	@selected_color: the color-pair to give a selected item
 *	@cols: the number of columns in the clist
 *	@column: the array of clist_column structures
 *	@user_data: whatever the caller wants to associate with the clist
 *
 *	This structure maintains information about a selection list that can
 *	be displayed in a sub-window. If the list is longer than the viewport
 *	then it can scrolled. The choice list allows multiple or single selections
 *	and the registration of handlers when selecting and unselecting rows
 *	as well as when activating a row (press ENTER on a list item).
 */
struct clist {
	WINDOW *win;
	GList *choices;
	GList *selections;
	GList *top_item;
	GList *focus_item;
	GList *idle_queue;
	clist_delete_cb delete_cb;
	clist_select_item_cb select_cb;
	clist_unselect_item_cb unselect_cb;
	clist_activate_item_cb activate_cb;
	clist_focus_item_cb focus_cb;
	chtype background;
	chtype focus_color;
	chtype selected_color;
	int mode;
	int cols;
	struct clist_column *column;
	void *user_data;
};

#define CLIST_MODE_SINGLE	0x0
#define CLIST_MODE_MULTI	0x1

struct clist *create_clist(WINDOW *parent_win, int cols, int height, int width, int starty, int startx,
				void *user_data);
void delete_clist(struct clist *clist);
void delete_clist_item(struct clist *clist, struct clist_item *item);
void select_item(struct clist *clist, struct clist_item *item);
void unselect_item(struct clist *clist, struct clist_item *item);
inline int set_selection_mode(struct clist *clist, int newmode);
inline void set_clist_color_attrs(struct clist *clist, chtype background, short focus_color,
				short selected_color);
void set_clist_column_count(struct clist *clist, int cols);
void set_clist_column_info(struct clist *clist, int col, int width, int startx, 
				clist_text_justification justification);
inline void set_clist_delete_cb(struct clist *clist, clist_delete_cb delete_cb);
inline clist_select_item_cb get_clist_select_item_cb(struct clist *clist);
inline clist_unselect_item_cb get_clist_unselect_item_cb(struct clist *clist);
inline void set_clist_select_item_cb(struct clist *clist, clist_select_item_cb select_cb);
inline void set_clist_unselect_item_cb(struct clist *clist, clist_unselect_item_cb unselect_cb);
inline void set_clist_activate_item_cb(struct clist *clist, clist_activate_item_cb activate_cb);
inline void set_clist_focus_item_cb(struct clist *clist, clist_focus_item_cb focus_cb);
inline void set_clist_top_item(struct clist *clist, GList *item);
inline GList *find_item_from_handle(struct clist *clist, engine_handle_t handle);
inline void set_clist_focus_item(struct clist *clist, GList *item);
inline GList *get_clist_top_item(struct clist *clist);
inline GList *get_clist_focus_item(struct clist *clist);
inline int get_clist_column_width(struct clist *clist, int col);
inline int get_clist_column_start(struct clist *clist, int col);
inline int get_clist_column_end(struct clist *clist, int col);
inline int calc_clist_column_width(struct clist *clist, float width_in_percent);
void print_clist_column_title(struct clist *clist, int col, char *title);
struct clist_item *append_item(struct clist *clist, GPtrArray *text, void *user_data,
				clist_item_delete_cb delete_cb);
inline void set_item_selectable(struct clist_item *item, int is_selectable);
inline int is_clist_empty(struct clist *clist);
void draw_clist(struct clist *clist);
void clear_clist(struct clist *clist);
int is_item_visible(struct clist *clist, GList *item);
int process_clist_events(struct clist *clist, int key);
void sched_clist_idle_func(struct clist *clist, clist_idle_func func);
int clist_allow_all(object_handle_t handle, void *user_data);
int disallow_select_events(struct clist *clist, struct clist_item *item);
inline void clist_populate_handle(struct clist *clist, object_handle_t handle,
				clist_format_func format_func,
				clist_item_delete_cb delete_cb, void *user_data);
void clist_populate(struct clist *clist, clist_enum_func enum_func, clist_filter_func filter_func,
			clist_format_func format_func, clist_item_delete_cb delete_cb, void *user_data);

