# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
from xml.dom.minidom import parseString

import gtk
import gobject
import os

from gazpacho import choice, propertyclass, util, widget
from gazpacho.loader import tags
from gazpacho.placeholder import Placeholder
from gazpacho.uieditor import UIEditor
from gazpacho.widgetregistry import widget_registry
from gazpacho.custompropertyeditor import CustomPropertyEditor

from xml.sax.saxutils import escape

# The root library is the name of the library Gazpacho will import to create
# the widgets. e.g import gtk
root_library = 'gtk'

# When loading the widget class from the catalog files we strip off this prefix
# for every widget
widget_prefix = 'Gtk'

def glade_gtk_widget_condition(widget_class):
    button_gtype = gobject.type_from_name('GtkButton')
    menuitem_gtype = gobject.type_from_name('GtkMenuItem')
    
    if gobject.type_is_a(widget_class.type, button_gtype):
        return True
    elif gobject.type_is_a(widget_class.type, menuitem_gtype):
        return True
    else:
        return False # XXX TODO check for no window widgets

def glade_gtk_widget_get_tooltip(context, gtk_widget):
    data = gtk.tooltips_data_get(gtk_widget)
    if data is not None:
        return data[2]

    return None

def glade_gtk_widget_set_tooltip(context, gtk_widget, tooltip):
    gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
    tooltips = gwidget.project.tooltips
    if tooltip:
        tooltips.set_tip(gtk_widget, tooltip, None)
    else:
        tooltips.set_tip(gtk_widget, None, None)

class EventsAdaptor(object):
    """We need to store the events value in an attribute of the Gazpacho
    widget because the events property of the gtk widget is used by
    Gazpacho for internal things"""
    def get(self, context, gtk_widget):
        gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
        return gwidget._events
            
    def set(self, context, gtk_widget, value):
        gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
        gwidget._events = value

    def save(self, context, gtk_widget):
        gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
        events_prop = gwidget.get_glade_property('events')
        return choice.flags_to_string(events_prop._value, events_prop.klass.spec)
    
def glade_gtk_box_get_size(context, box):
    return len(box.get_children())

def glade_gtk_box_set_size(context, box, size):
    old_size = len(box.get_children())
    new_size = size

    if new_size == old_size:
        return
    elif new_size > old_size:
        # The box has grown. Add placeholders
        while new_size > old_size:
            box.add(context.create_placeholder())
            old_size += 1
    else:
        # The box has shrunk. Remove the widgets that are not on those slots
        child = box.get_children()[-1]
        while old_size > new_size and child:
            gwidget = widget.get_widget_from_gtk_widget(child)
            if gwidget: # It may be None, e.g a placeholder
                gwidget.project.remove_widget(child)

            box.remove(child)
            child = box.get_children()[-1]
            old_size -= 1
            
    box.set_data('glade_nb_placeholders', new_size)

def glade_gtk_text_view_get_text(context, text_view):
    buffer = text_view.get_buffer()
    (start, end) = buffer.get_bounds()
    return buffer.get_text(start, end, False)

def glade_gtk_text_view_set_text(context, text_view, text):
    text_view.get_buffer().set_text(text or '')
    
def glade_gtk_notebook_get_n_pages(context, notebook):
    return notebook.get_n_pages()

def glade_gtk_notebook_set_n_pages(context, notebook, pages):
    gwidget = widget.get_widget_from_gtk_widget(notebook)
    old_size = notebook.get_n_pages()
    new_size = pages

    if new_size == old_size:
        return

    if new_size > old_size:
        # The notebook has grown. Add pages
        while new_size > old_size:
            notebook.append_page(context.create_placeholder(), None)
            old_size += 1
            
    else:
        # The notebook has shrunk. Remove pages
        
        # Thing to remember is that GtkNotebook starts the
        # page numbers from 0, not 1 (C-style). So we need to do
        # old_size-1, where we're referring to "nth" widget.
        while old_size > new_size:
            child_widget = notebook.get_nth_page(old_size - 1)
            child_gwidget = widget.get_widget_from_gtk_widget(child_widget)

            # If we got it, and it's not a placeholder, remove it from project
            if child_gwidget:
                child_gwidget.project.remove_widget(child_widget)

            notebook.remove_page(old_size - 1)
            old_size -= 1

def glade_gtk_table_set_n_common(context, table, value, for_rows):
    new_size = value
    p = for_rows and 'n-rows' or 'n-columns'
    old_size = table.get_property(p)
    if new_size == old_size:
        return

    if new_size < 1:
        return

    gwidget = widget.get_widget_from_gtk_widget(table)
    if new_size > old_size:
        if for_rows:
            table.resize(new_size, table.get_property('n-columns'))
            for i in range(table.get_property('n-columns')):
                for j in range(old_size, table.get_property('n-rows')):
                    table.attach(context.create_placeholder(), i, i+1, j, j+1)
        else:
            table.resize(table.get_property('n-rows'), new_size)
            for i in range(old_size, table.get_property('n-columns')):
                for j in range(table.get_property('n-rows')):
                    table.attach(context.create_placeholder(), i, i+1, j, j+1)
    else:
        # Remove from the bottom up
        l = table.get_children()
        l.reverse()
        for child in l:
            p = for_rows and 'top-attach' or 'left-attach'
            start = table.child_get_property(child, p)
            p = for_rows and 'bottom-attach' or 'right-attach'
            end = table.child_get_property(child, p)

            # We need to completely remove it
            if start >= new_size:
                table.remove(child)
                continue
            # If the widget spans beyond the new border, we should resize it to
            # fit on the new table
            if end > new_size:
                p = for_rows and 'bottom-attach' or 'right-attach'
                table.child_set_property(child, p, new_size)
        table.resize(for_rows and new_size or table.get_property('n-rows'),
                     for_rows and table.get_property('n-columns') or new_size)

        p = for_rows and 'n-columns' or 'n-rows'
        table.set_data('glade_nb_placeholders',
                       new_size * table.get_property(p))
                       
    
def glade_gtk_table_set_n_rows(context, table, rows):
    glade_gtk_table_set_n_common(table, rows, True)

def glade_gtk_table_set_n_columns(context, table, columns):
    glade_gtk_table_set_n_common(table, columns, False)

def glade_gtk_button_set_stock(context, ):
    print 'gtk button set stock'

def glade_gtk_statusbar_get_has_resize_grip(context, statusbar):
    return statusbar.get_has_resize_grip()

def glade_gtk_statusbar_set_has_resize_grip(context, statusbar, value):
    statusbar.set_has_resize_grip(value)

# Pre Create functions

def glade_gtk_tree_view_pre_create(context, tree_view, interactive=True):
    model = gtk.ListStore(str) # dummy model
    tree_view.set_model(model)

    # While we don't support column editing on the treeview
    # this does not make sense
##     renderer = gtk.CellRendererText()
##     column = gtk.TreeViewColumn('Column 1', renderer, text=0)
##     tree_view.append_column(column)

##     column = gtk.TreeViewColumn('Column 2', renderer, text=0)
##     tree_view.append_column(column)

def glade_gtk_combo_box_pre_create(context, combo, interactive=True):
    model = gtk.ListStore(gobject.TYPE_STRING)
    combo.set_model(model)

def glade_gtk_window_post_create(context, window, interactive=True):
    window.set_default_size(440, 250)

### Managing the custom editors for GtkImage
def _image_icon_editor_add_widget(box, name, gtk_widget, expand):
    box.pack_start(gtk_widget, expand=expand)
    box.set_data(name, gtk_widget)

def glade_gtk_image_icon_editor_create(context):

    store = gtk.ListStore(str, str)
    completion = gtk.EntryCompletion()
    completion.set_model(store)

    cell = gtk.CellRendererPixbuf()
    completion.pack_start(cell, True)
    completion.add_attribute(cell, 'stock-id', 0)

    cell = gtk.CellRendererText()
    completion.pack_start(cell, True)
    completion.add_attribute(cell, 'text', 1)

    completion.set_text_column(0)
    completion.set_minimum_key_length(1)
    
    _image_icon_create_stock_popup(completion)
    
    entry = gtk.Entry()
    entry.set_completion(completion)
    
    fc_button = gtk.Button(_('...'))

    hbox = gtk.HBox()
    _image_icon_editor_add_widget(hbox, 'image-icon-entry', entry, True)
    _image_icon_editor_add_widget(hbox, 'image-icon-file-chooser', fc_button,
                                  False)

    return hbox

def _image_icon_create_stock_popup(completion):
    model = completion.get_model()
    
    for stock_id in gtk.stock_list_ids():
        if stock_id == 'gtk-missing-image': continue
        stock_item = gtk.stock_lookup(stock_id)
        if not stock_item:
            continue
        model.append([stock_id, stock_item[1]])

def _image_icon_entry_changed(entry, proxy):
    if entry.get_text() in gtk.stock_list_ids():
        proxy.set_property('stock', entry.get_text())

def _image_icon_file_clicked(button, (proxy, image, entry)):
    dialog = gtk.FileChooserDialog('Chooser', None,
                                   gtk.FILE_CHOOSER_ACTION_OPEN,
                                   buttons=(gtk.STOCK_CANCEL,
                                            gtk.RESPONSE_CANCEL,
                                            gtk.STOCK_OPEN,
                                            gtk.RESPONSE_OK))

    filter = gtk.FileFilter()
    filter.set_name("Images")
    filter.add_mime_type("image/png")
    filter.add_mime_type("image/jpeg")
    filter.add_mime_type("image/gif")
    filter.add_mime_type("image/x-xpixmap")
    filter.add_pattern("*.png")
    filter.add_pattern("*.jpg")
    filter.add_pattern("*.gif")
    filter.add_pattern("*.xpm")

    dialog.add_filter(filter)
    
    response = dialog.run()
    if response == gtk.RESPONSE_OK and dialog.get_filename():
        entry.handler_block(entry.get_data('handler-id-changed'))
        entry.set_text(os.path.basename(dialog.get_filename()))
        entry.handler_unblock(entry.get_data('handler-id-changed'))
        proxy.set_property('file', dialog.get_filename())
        image.set_data('image-file-name', entry.get_text())

    dialog.destroy()
    return
 
def _image_icon_reconnect(gtk_widget, signal, callback, userdata):
    handler_id = gtk_widget.get_data('handler-id-' + signal)
    if handler_id:
        gtk_widget.disconnect(handler_id)

    handler_id = gtk_widget.connect(signal, callback, userdata)
    gtk_widget.set_data('handler-id-' + signal, handler_id)

def glade_gtk_image_icon_editor_update(context, box, image, proxy):
    entry = box.get_data('image-icon-entry')
    toggle = box.get_data('image-icon-stock-chooser')
    fc_button = box.get_data('image-icon-file-chooser')

    stock_id = image.get_property('stock')
    if stock_id and stock_id != 'gtk-missing-image':
        text = stock_id
    elif image.get_data('image-file-name'):
        text = image.get_data('image-file-name')
    else:
        text = ''

    entry.set_text(text)

    _image_icon_reconnect(entry, 'changed', _image_icon_entry_changed, proxy)
    _image_icon_reconnect(fc_button, 'clicked', _image_icon_file_clicked,
                          (proxy, image, entry))

def glade_gtk_image_icon_size_editor_create(context):
    store = gtk.ListStore(str, int)

    combo = gtk.ComboBox(store)
    
    cell = gtk.CellRendererText()
    combo.pack_start(cell, True)
    combo.add_attribute(cell, 'text', 0)

    return combo

def _image_icon_size_setup_combo(editor, image):
    model = editor.get_model()
    model.clear()
    
    stock_id = image.get_property('stock')
    if not stock_id or stock_id == 'gtk-missing-image':
        editor.set_sensitive(False)
        return

    icon_set = gtk.icon_factory_lookup_default(stock_id)
    editor.set_sensitive(True)
    
    icon_size = image.get_property('icon-size')
    for size in icon_set.get_sizes():
        iter = model.append([gtk.icon_size_get_name(size), size])

        if size == icon_size:
            editor.handler_block(editor.get_data('handler-id-changed'))
            editor.set_active_iter(iter)
            editor.handler_unblock(editor.get_data('handler-id-changed'))

def _image_icon_size_notify(image, param_spec, (editor, proxy)):
    _image_icon_size_setup_combo(editor, image)

def _image_icon_size_changed(editor, proxy):
    iter = editor.get_active_iter()
    if iter:
        model = editor.get_model()
        size = model.get_value(iter, 1)

        proxy.set_value(size)

def glade_gtk_image_icon_size_editor_update(context, editor, image, proxy):
    handler_id = image.get_data('image-notify-id')
    if handler_id:
        image.disconnect(handler_id)

    _image_icon_reconnect(editor, 'changed', _image_icon_size_changed, proxy)

    _image_icon_size_setup_combo(editor, image)
    
    handler_id = image.connect('notify', _image_icon_size_notify,
                               (editor, proxy))
    image.set_data('image-notify-id', handler_id)
    return


def glade_gtk_image_get_file(context, image):
    return image.get_data('image-file-name')

def glade_gtk_message_dialog_post_create(context, dialog, interactive=True):
    dialog.set_default_size(400, 115)

# Replace child functions
def glade_gtk_notebook_replace_child(context, current, new, container):
    page_num = container.page_num(current)
    if page_num == -1:
        return

    page = container.get_nth_page(page_num)
    label = container.get_tab_label(current)

    container.remove_page(page_num)
    container.insert_page(new, label, page_num)

    new.show()
    container.set_current_page(page_num)

# Fill Empty functions

def _glade_gtk_paned_set_position(paned):
    """This function get called until the paned is realized.
    Then it puts the bar in the middle.
    """
    if paned.flags() & gtk.REALIZED:
        alloc = paned.get_allocation()
        if isinstance(paned, gtk.VPaned):
            pos = alloc.height / 2
        else:
            pos = alloc.width / 2
        paned.set_position(pos)
        return False
    else:
        return True
    
def glade_gtk_paned_fill_empty(context, paned):
    paned.add1(context.create_placeholder())
    paned.add2(context.create_placeholder())
    # we want to set the position in the middle but
    # as the paned has not yet a parent it is not
    # realized we can't know its size
    gobject.idle_add(_glade_gtk_paned_set_position, paned)


# these widgets does not need a viewport when added to a scrolledwindow
scrollable_widgets = (gtk.TreeView, gtk.TextView, gtk.Viewport)

class ScrolledWindowAdaptor:

    def fill_empty(self, context, scrolledwindow):
        self._add_viewport(scrolledwindow, context.get_project())

    def _add_viewport(self, scrolledwindow, project):
        """ScrolledWindow should be empty before calling this method"""
        klass = widget_registry.get_by_name('GtkViewport')
        viewport = widget.Widget(klass, project)
        viewport.create_gtk_widget(interactive=False)
        scrolledwindow.add(viewport.gtk_widget)
        project.add_widget(viewport.gtk_widget)
        viewport.gtk_widget.show_all()
        return viewport
        
    def replace_child(self, context, current, new, container):
        global scrollable_widgets
        
        if current is None:
            raise ValueError("Can not replace None widget")

        # we don't care about current since it is probably different from
        # our child because we always have a viewport when adding a widget
        # that is not scrollable
        child = container.get_child()
        container.remove(child)
        project = context.get_project()
        if child is not current:
            project.remove_widget(child)
        
        # luckily viewports or scrolledwindow does not have any packing
        # properties

        if isinstance(new, scrollable_widgets):
            container.add(new)
        elif isinstance(new, Placeholder):
            self._add_viewport(container, project)
        else:
            raise ValueError("You can only put scrollable widgets or "
                             "placeholders inside a ScrollableWindow: "
                             "%s" % new)

class ContainerAdaptor:
    def replace_child(self, context, current, new, container):
        if current is None:
            container.add(new)
            return
        
        if current.parent != container:
            return

        props = {}
        for pspec in gtk.container_class_list_child_properties(type(container)):
            props[pspec.name] = container.child_get_property(current, pspec.name)

        container.remove(current)
        container.add(new)
            
        for name, value in props.items():
            container.child_set_property(new, name, value)

    def fill_empty(self, context, gtk_widget):
        gtk_widget.add(context.create_placeholder())

class ViewportAdaptor(ContainerAdaptor):

    def replace_child(self, context, current, new, container):
        """If the new child is a scrollable widget and our parent is a
        scrolled window, we remove ourselves"""
        global scrollable_widgets
        parent = container.get_parent()
        if (isinstance(parent, gtk.ScrolledWindow)
            and isinstance(new, scrollable_widgets)):
            project = context.get_project()
            project.remove_widget(container)
            parent.remove(container)
            parent.add(new)
        else:
            ContainerAdaptor.replace_child(self, context, current, new,
                                           container)

class BaseAttach:
    """Base class for LeftAttach, RightAttach, TopAttach and BottomAttach
    adaptors"""
    def _get_attach(self, child):
        """Returns the four attach packing properties in a tuple"""
        right = self.table.child_get_property(child, 'right-attach')
        left = self.table.child_get_property(child, 'left-attach')
        top = self.table.child_get_property(child, 'top-attach')
        bottom = self.table.child_get_property(child, 'bottom-attach')
        return (left, right, top, bottom)

    def _cell_empty(self, x, y):
        """Returns true if the cell at x, y is empty. Exclude child from the
        list of widgets to check"""
        empty = True
        for w in self.table.get_children():
            left, right, top, bottom = self._get_attach(w)
            if (left <= x and (x + 1) <= right 
                and top <= y and (y + 1) <= bottom):
                empty = False
                break

        return empty

    def _create_placeholder(self, context, x, y):
        """Puts a placeholder at cell (x, y)"""
        placeholder = context.create_placeholder()
        self.table.attach(placeholder, x, x+1, y, y+1)

    def _fill_with_placeholders(self, context, y_range, x_range):
        """Walk through the table creating placeholders in empty cells.
        Only iterate between x_range and y_range.
        Child is excluded in the computation to see if a cell is empty
        """
        for y in range(self.n_rows):
            for x in range(self.n_columns):
                if self._cell_empty(x, y):
                    self._create_placeholder(context, x, y)

    def _initialize(self, child, prop_name):
        """Common things all these adaptors need to do at the beginning"""
        self.table = child.get_parent()
        (self.left_attach,
         self.right_attach,
         self.top_attach,
         self.bottom_attach) = self._get_attach(child)

        self.n_columns = self.table.get_property('n-columns')
        self.n_rows = self.table.get_property('n-rows')

        self.prop_name = prop_name
        self.child = child
        self.gchild = widget.get_widget_from_gtk_widget(self.child)
        self.gprop = self.gchild.get_glade_property(self.prop_name)

    def _value_is_between(self, value, minimum, maximum):
        if value < minimum:
            self.gprop._value = minimum
            return False

        if value > maximum:
            self.gprop._value = maximum
            return False

        return True

    def _internal_set(self, context, value, minimum, maximum,
                      y_range, x_range):
        """Check if value is between minium and maximum and then remove or
        add placeholders depending if we are growing or shrinking.
        If we are shrinking check the cells in y_range, x_range to add
        placeholders
        """
        if not self._value_is_between(value, minimum, maximum):
            return

        placeholder = context.create_placeholder()

        # are we growing?
        if self._is_growing(value):
            # check if we need to remove some placeholder
            for ph in filter(lambda w: isinstance(w, type(placeholder)),
                             self.table.get_children()):
                lph, rph, tph, bph = self._get_attach(ph)
                if self._cover_placeholder(value, lph, rph, tph, bph):
                    self.table.remove(ph)

            self.table.child_set_property(self.child, self.prop_name, value)

        # are we shrinking? maybe we need to create placeholders
        elif self._is_shrinking(value):
            self.table.child_set_property(self.child, self.prop_name, value)
            self._fill_with_placeholders(context, y_range, x_range)


    # virtual methods that should be implemented by subclasses:
    def _is_growing(self, value):
        """Returns true if the child widget is growing"""

    def _is_shrinking(self, value):
        """Returns true if the child widget is shrinking"""

    def _cover_placeholder(self, value, left, right, top, bottom):
        """Return True if there is a placeholder in these coordinates"""
        
class TableAdaptor:

    class LeftAttachAdaptor(BaseAttach):
        def _is_growing(self, value):
            return value < self.left_attach

        def _is_shrinking(self, value):
            return value > self.left_attach

        def _cover_placeholder(self, value, left, right, top, bottom):
            if value < right and self.left_attach > left:
                if top >= self.top_attach and bottom <= self.bottom_attach:
                    return True
            return False

        def set(self, context, child, value):
            self._initialize(child, 'left-attach')
            self._internal_set(context, value, 0, self.right_attach - 1,
                               range(self.n_rows),
                               range(self.left_attach, value))

    class RightAttachAdaptor(BaseAttach):
        def _is_growing(self, value):
            return value > self.right_attach

        def _is_shrinking(self, value):
            return value < self.right_attach

        def _cover_placeholder(self, value, left, right, top, bottom):
            if value > left and self.right_attach < right:
                if top >= self.top_attach and bottom <= self.bottom_attach:
                    return True
            return False
            
        def set(self, context, child, value):
            self._initialize(child, 'right-attach')
            self._internal_set(context, value,
                               self.left_attach + 1, self.n_columns,
                               range(value, self.right_attach),
                               range(self.n_rows))

    class BottomAttachAdaptor(BaseAttach):
        def _is_growing(self, value):
            return value > self.bottom_attach

        def _is_shrinking(self, value):
            return value < self.bottom_attach

        def _cover_placeholder(self, value, left, right, top, bottom):
            if value > top and self.bottom_attach < bottom:
                if left >= self.left_attach and right <= self.right_attach:
                    return True                
            return False

        def set(self, context, child, value):
            self._initialize(child, 'bottom-attach')
            self._internal_set(context, value,
                               self.top_attach + 1, self.n_rows,
                               range(value, self.bottom_attach),
                               range(self.n_columns))

    class TopAttachAdaptor(BaseAttach):
        def _is_growing(self, value):
            return value < self.top_attach

        def _is_shrinking(self, value):
            return value > self.top_attach

        def _cover_placeholder(self, value, left, right, top, bottom):
            if value < bottom and self.top_attach > top:
                if left >= self.left_attach and right <= self.right_attach:
                    return True
            return False
        
        def set(self, context, child, value):
            self._initialize(child, 'top-attach')
            self._internal_set(context, value, 0, self.bottom_attach - 1,
                               range(self.n_columns),
                               range(self.top_attach, value))

    class NRowsAdaptor:
        def set(self, context, table, rows):
            glade_gtk_table_set_n_common(context, table, rows, True)

    class NColumnsAdaptor:
        def set(self, context, table, cols):
            glade_gtk_table_set_n_common(context, table, cols, False)
        
    def pre_create(self, context, table, interactive=True):
        """ a GtkTable starts with a default size of 1x1, and setter/getter of
        rows/columns expect the GtkTable to hold this number of placeholders,
        so we should add it. """
        table.attach(context.create_placeholder(), 0, 1, 0, 1)

    def post_create(self, context, table, interactive=True):
        if not interactive:
            return
        gwidget = widget.get_widget_from_gtk_widget(table)
        property_rows = gwidget.get_glade_property('n-rows')
        property_cols = gwidget.get_glade_property('n-columns')
        dialog = gtk.Dialog(_('Create a table'), None,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT | \
                            gtk.DIALOG_NO_SEPARATOR,
                            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        dialog.set_position(gtk.WIN_POS_MOUSE)
        dialog.set_default_response(gtk.RESPONSE_ACCEPT)
        
        label_rows = gtk.Label(_('Number of rows')+':')
        label_rows.set_alignment(0.0, 0.5)
        label_cols = gtk.Label(_('Number of columns')+':')
        label_cols.set_alignment(0.0, 0.5)
        
        spin_button_rows = gtk.SpinButton()
        spin_button_rows.set_increments(1, 5)
        spin_button_rows.set_range(1.0, 10000.0)
        spin_button_rows.set_numeric(False)
        spin_button_rows.set_value(3)
        spin_button_rows.set_property('activates-default', True)
        
        spin_button_cols = gtk.SpinButton()
        spin_button_cols.set_increments(1, 5)
        spin_button_cols.set_range(1.0, 10000.0)
        spin_button_cols.set_numeric(False)
        spin_button_cols.set_value(3)
        spin_button_cols.set_property('activates-default', True)
        
        table = gtk.Table(2, 2, True)
        table.set_col_spacings(4)
        table.set_border_width(12)
        
        table.attach(label_rows, 0, 1, 0, 1)
        table.attach(spin_button_rows, 1, 2, 0, 1)
        
        table.attach(label_cols, 0, 1, 1, 2)
        table.attach(spin_button_cols, 1, 2, 1, 2)
        
        dialog.vbox.pack_start(table)
        table.show_all()
        
        # even if the user destroys the dialog box, we retrieve the number and
        # we accept it.  I.e., this function never fails
        dialog.run()
        
        property_rows.set(spin_button_rows.get_value_as_int())
        property_cols.set(spin_button_cols.get_value_as_int())
        
        dialog.destroy()
        

class UIAdaptor:
    """This represents a fake property to edit the ui definitions of
    menubars and toolbars.

    It does not store anything because this information is stored in
    the uimanager itself. It's just a way to put a button in the Gazpacho
    interface to call the UIEditor.
    """
    def __init__(self):
        self._ui_editor = UIEditor()

    def create_editor(self, context):
        button = gtk.Button(_('Edit...'))
        button.set_data('connection-id', -1)
        return button

    def update_editor(self, context, button, gtk_widget, proxy):
        connection_id = button.get_data('connection-id')
        if connection_id != -1:
            button.disconnect(connection_id)

        connection_id = button.connect('clicked', self._ui_editor_edit,
                                       gtk_widget, proxy, context)
        button.set_data('connection-id', connection_id)

    def _ui_editor_edit(self, gtk_widget, menubar, proxy, context):
        app_window = context.get_application_window()
        self._ui_editor.set_transient_for(app_window)
    
        self._ui_editor.set_widget(menubar, proxy)
        self._ui_editor.present()

class BarNameAdaptor(object):
    def set(self, context, gtk_widget, new_name):
        """If the user changes the name of the menubar the UI definition
        needs to be updated"""
        old_name = gtk_widget.get_property('name')
        if old_name == new_name:
            return # this avoid recursion infinite loop
        gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
        gwidget.name = new_name

        gwidget.project.uim.update_widget_name(gwidget)
            
class CommonBarsAdaptor(object):

    def post_create(self, context, gtk_widget, ui_string):
        # create some default actions
        gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
        project = gwidget.project
        project.uim.create_default_actions()

        project.uim.add_ui(gwidget, ui_string)
        new_gtk_widget = project.uim.get_gtk_widget(gwidget)

        # we need to replace gtk_widget with new_widget
        gwidget.setup_widget(new_gtk_widget)
        gwidget.apply_properties()
    
    def save(self, context, document, gwidget):
        """This saver is needed to avoid saving the children of toolbars
        and menubars
        """
        # write the xml node (but no children in it)
        node = gwidget.write_basic_information(document)
        node.setAttribute('constructor', 'initial-state')
        gwidget.write_properties(document, node)
        gwidget.write_signals(document, node)
        return node

    def load(self, context, gtk_widget, widget_tree, blacklist):
        """This loader is special because of these features:
        - It does not load the children of the menubar/toolbar
        - Load the uimanager and put its content (action groups) into the
        project
        """
        old_name = gtk_widget.name
        
        project = context.get_project()
        if not project.uim.is_loading():
            # we have to load the ui manager first
            project.uim.load(widget_tree)

        klass = widget_registry.get_by_type(gtk_widget.__gtype__)

        # we need to save the properties of this gtk_widget because otherwise
        # when we got it from the uimanager it's gonna be another widget with
        # different properties
        props = {}
        for prop in gobject.list_properties(gtk_widget):
            if prop.flags != gobject.PARAM_READWRITE:
                continue
            if propertyclass.get_type_from_spec(prop) is gobject.TYPE_OBJECT:
                continue
            props[prop.name] = gtk_widget.get_property(prop.name)
        
        gwidget = widget.load_gwidget(gtk_widget, klass, project, blacklist)
        gwidget._name = gwidget.gtk_widget.name

        # change the gtk_widget for the one we get from the uimanager
        project.uim.load_widget(gwidget, old_name)

        # now gtk_widget is different and we have to restore the properties
        for prop_name, prop_value in props.items():
            gwidget.gtk_widget.set_property(prop_name, prop_value)
            
        widget.load_properties(gwidget)
        widget.load_signals(gwidget, widget_tree)

        return gwidget

class MenuBarAdaptor(CommonBarsAdaptor):

    def post_create(self, context, menubar, interactive=True):
        gwidget = widget.get_widget_from_gtk_widget(menubar)
        # A None in this list means a separator
        names = [gwidget.name, _('FileMenu'), _('New'), _('Open'), _('Save'),
                 _('SaveAs'), _('Quit'), _('EditMenu'), _('Copy'), _('Cut'),
                 _('Paste')]
        tmp = []
        for name in names:
            tmp.extend([name, name])
        
        ui_string = """<menubar action="%s" name="%s">
  <menu action="%s" name="%s">
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <separator/>
    <menuitem action="%s" name="%s"/>
  </menu>
  <menu action="%s" name="%s">
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
  </menu>
</menubar>""" % tuple(tmp)
    
        super(MenuBarAdaptor, self).post_create(context, menubar, ui_string)

class ToolbarAdaptor(CommonBarsAdaptor):
    def post_create(self, context, toolbar, interactive=True):
        gwidget = widget.get_widget_from_gtk_widget(toolbar)

        names = [gwidget.name, _('New'), _('Open'), _('Save'),
                 _('Copy'), _('Cut'), _('Paste')]
        tmp = []
        for name in names:
            tmp.extend([name, name])
    
        ui_string = """<toolbar action="%s" name="%s">
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <separator/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
</toolbar>""" % tuple(tmp)

        super(ToolbarAdaptor, self).post_create(context, toolbar, ui_string)

class DialogAdaptor:
    def get_internal_child(self, context, dialog, name=None):
        """Return the internal child with that name or if name is None return
        a tuple of all the possible names for internal children
        """
        if name is None:
            return ('vbox', 'action_area')
        else:
            return getattr(dialog, name, None)
    
    def child_property_applies(self, context, ancestor, gtk_widget,
                               property_id):
        if property_id == 'response-id' and \
           isinstance(gtk_widget.parent, gtk.HButtonBox) and \
           isinstance(gtk_widget.parent.parent, gtk.VBox) and \
           gtk_widget.parent.parent.parent == ancestor:
            return True
        elif gtk_widget.parent == ancestor:
            return True

        return False
    
    def fill_empty(self, context, dialog):
        dialog.vbox.pack_start(context.create_placeholder())
        
    def post_create(self, context, dialog, interactive=True):
        gwidget = widget.get_widget_from_gtk_widget(dialog)
        if not gwidget: return
        
        # create the GladeWidgets for internal children
        self._setup_internal_children(gwidget)
                
        dialog.action_area.pack_start(context.create_placeholder())
        dialog.action_area.pack_start(context.create_placeholder())
        
        # set a reasonable default size for a dialog
        dialog.set_default_size(320, 260)

    def _setup_internal_children(self, gwidget):
        child_class = widget_registry.get_by_name('GtkVBox')
        vbox_widget = widget.Widget(child_class, gwidget.project)
        vbox_widget.setup_internal_widget(gwidget.gtk_widget.vbox, 'vbox',
                                          gwidget.name or '')

        child_class = widget_registry.get_by_name('GtkHButtonBox')
        action_area_widget = widget.Widget(child_class, gwidget.project)
        action_area_widget.setup_internal_widget(gwidget.gtk_widget.action_area,
                                                 'action_area',
                                                 gwidget.name or '')
        
# Adaptors for RadioButton
class GroupAdaptor:
    def __init__(self):
        self.radiobutton = None
        self.proxy = None
        self.updating = False
        
    def create_editor(self, context):
        liststore = gtk.ListStore(str, object)
        combo = gtk.ComboBox(liststore)
        cell = gtk.CellRendererText()
        combo.pack_start(cell, True)
        combo.add_attribute(cell, 'text', 0)
        combo.connect('changed', self._on_combo_changed)
        return combo

    def update_editor(self, context, combo, gtk_widget, proxy):
        self.radiobutton = gtk_widget
        self.proxy = proxy

        self.updating = True

        # get a list of all the radiobuttons in this project
        project = context.get_project()
        rbs = [w for w in project.widgets \
               if isinstance(w, gtk.RadioButton)]
        model = combo.get_model()
        model.clear()
        for rb in rbs:
            model.append((rb.name, rb))

        # gtk.RadioButton.get_group always return a list with at least one
        # element: the radiobutton itself.
        group = gtk_widget.get_group()[-1]
        gi = rbs.index(group)

        combo.set_active(gi)

        self.updating = False
        
    
    def _on_combo_changed(self, combo):
        if self.radiobutton is None or self.proxy is None:
            return

        if self.updating:
            return

        model = combo.get_model()
        active_iter = combo.get_active_iter()
        rb = model.get_value(active_iter, 1)
        if rb == self.radiobutton:
            self.proxy.set_value(None)
        else:
            self.proxy.set_value(rb)

    def save(self, context, gtk_widget):
        """Return a string representation of the object"""
        group = gtk_widget.get_group()[-1]
        return escape(group.name)

# Adaptors for Button
class ImageSelector(CustomPropertyEditor):
    """Dialog to choose from a stock icon, an image from the filesystem
    or no image at all.
    """
    def __init__(self):
        CustomPropertyEditor.__init__(self)
        self.loading = False

    def _create_widgets(self):
        msg = _('Use a builtin icon')
        self.stock_radio_button = gtk.RadioButton(None, msg, True)
        self.stock_radio_button.connect('toggled',
                                        self._on_radio_button__toggled)
        self.vbox.pack_start(self.stock_radio_button, False)

        # put the stock icons in a treeview
        liststore = gtk.ListStore(str, str)
        stocks = gtk.stock_list_ids()
        stocks.sort()
        for stock in stocks:
            info = gtk.stock_lookup(stock)
            if info is None:
                label = "* %s" % stock
            else:
                label = info[1]
            liststore.append((stock, label))
        self.stock_list = gtk.TreeView(liststore)
        self.stock_list.set_headers_visible(False)
        column = gtk.TreeViewColumn()
        cell1 = gtk.CellRendererPixbuf()
        column.pack_start(cell1)
        column.add_attribute(cell1, 'stock-id', 0)
        cell2 = gtk.CellRendererText()
        column.pack_start(cell2)
        column.add_attribute(cell2, 'text', 1)
        self.stock_list.append_column(column)
        self.stock_list.get_selection().connect('changed', self._on_stock_list_selection_changed)

        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.stock_list)
        sw.set_size_request(300, 300)
        self.stock_alignment = gtk.Alignment()
        self.stock_alignment.set_padding(0, 0, 12, 0)
        self.stock_alignment.add(sw)
        self.vbox.pack_start(self.stock_alignment, True, True)
        
        msg = _('Use a custom image')
        self.custom_radio_button = gtk.RadioButton(self.stock_radio_button,
                                                   msg, True)
        self.custom_radio_button.connect('toggled',
                                         self._on_radio_button__toggled)
        self.vbox.pack_start(self.custom_radio_button, False)
        self.custom_hbox = gtk.HBox()
        self.filename_entry = gtk.Entry()
        self.filename_entry.set_editable(False)
        self.filename_entry.connect('changed', self._on_filename_entry_changed)
        self.custom_hbox.pack_start(self.filename_entry, True, True)
        file_button = gtk.Button(_("Choose a file..."))
        file_button.connect('clicked', self._on_file_button_clicked)
        self.custom_hbox.pack_start(file_button, False, False)
        self.custom_alignment = gtk.Alignment()
        self.custom_alignment.set_padding(0, 0, 12, 0)
        self.custom_alignment.add(self.custom_hbox)
        self.vbox.pack_start(self.custom_alignment, True, False)

        msg = _('Do not use an image')
        self.no_image_radio_button = gtk.RadioButton(self.stock_radio_button,
                                                     msg, True)
        self.no_image_radio_button.connect('toggled',
                                           self._on_radio_button__toggled)
        
        self.vbox.pack_start(self.no_image_radio_button, False)

        self.vbox.pack_start(gtk.HSeparator(), False)
        
        # encourage the user to use a stock item
        label = gtk.Label()
        advice = """<small>Try to use always a predefined icon because
it has several advantages:
        <b>* Theme friendly</b>
        <b>* Translations of the labels</b>
        <b>* Consistency (better usability)</b></small>"""
        label.set_alignment(0.0, 0.0)
        label.set_markup(advice)
        self.vbox.pack_start(label, False, False)

    def _on_radio_button__toggled(self, button):
        if (self.stock_radio_button.get_active() and
            button is not self.stock_radio_button):
            self.stock_alignment.set_sensitive(True)
            self.custom_alignment.set_sensitive(False)
            if not self.loading:
                self.set_stock_image(self.stock_list.get_selection())

        elif (self.custom_radio_button.get_active() and
              button is not self.custom_radio_button):

            self.stock_alignment.set_sensitive(False)
            self.custom_alignment.set_sensitive(True)
            if not self.loading:
                filename = self.filename_entry.get_text()
                if filename:
                    self.set_custom_image(filename)

        elif (self.no_image_radio_button.get_active() and
              button is not self.no_image_radio_button):
            
            self.stock_alignment.set_sensitive(False)
            self.custom_alignment.set_sensitive(False)
            if not self.loading:
                self._unset_image_on_button(self.widget)

    def _on_stock_list_selection_changed(self, selection):
        if not self.loading:
            self.set_stock_image(selection)

    def _select_stock_item(self, stock_id):
        model = self.stock_list.get_model()
        model_iter = model.get_iter_first()
        while model_iter:
            if model.get_value(model_iter, 0) == stock_id:
                self.stock_list.get_selection().select_iter(model_iter)
                break
            model_iter = model.iter_next(model_iter)
            
    def set_stock_image(self, selection):
        model, selected_iter = selection.get_selected()
        if selected_iter is not None:
            stock_id = model.get_value(selected_iter, 0)

            self._unset_image_on_button(self.widget)

            prop = self.gwidget.get_glade_property('use-stock')
            prop.set(True)

            prop = self.gwidget.get_glade_property('label')
            prop.set(stock_id)

            prop = self.gwidget.get_glade_property('use-underline')
            prop.set(True)

            # we need to refresh the property editor since we changed
            # the label property
            project = self.gwidget.project
            project._app.refresh_editor()
            
    def _on_file_button_clicked(self, button):
         dialog = gtk.FileChooserDialog(_('Open image ...'), self,
                                        gtk.FILE_CHOOSER_ACTION_OPEN,
                                        (gtk.STOCK_CANCEL,
                                         gtk.RESPONSE_CANCEL,
                                         gtk.STOCK_OPEN,
                                         gtk.RESPONSE_OK))
         file_filter = gtk.FileFilter()
         file_filter.set_name("Images")
         file_filter.add_mime_type("image/png")
         file_filter.add_mime_type("image/jpeg")
         file_filter.add_mime_type("image/gif")
         file_filter.add_mime_type("image/x-xpixmap")
         file_filter.add_pattern("*.png")
         file_filter.add_pattern("*.jpg")
         file_filter.add_pattern("*.gif")
         file_filter.add_pattern("*.xpm")

         dialog.add_filter(file_filter)
    
         response = dialog.run()
         filename = dialog.get_filename()
         if response == gtk.RESPONSE_OK and filename:
             self.filename_entry.set_text(filename)

         dialog.destroy()

    def _on_filename_entry_changed(self, entry):
        if self.loading:
            return
        
        filename = entry.get_text()
        if filename:
            self.set_custom_image(filename)
            
    def set_custom_image(self, filename):
        image = gtk.Image()
        image.set_from_file(filename)
        image.set_data('image-file-name', filename)
        self._set_image_on_button(self.widget, image)
        
    def _set_image_on_button(self, button, image):
        self._unset_image_on_button(button)

        name = button.name
        
        text = button.get_label()
        underline = button.get_use_underline()
        # we need to create some extra widgets and also their gazpacho
        # wrappers
        alignment = gtk.Alignment(0.5, 0.5)
        alignment.set_name("%s_alignment" % name)
        hbox = gtk.HBox(spacing=2)
        hbox.set_name("%s_hbox" % name)
        image.set_alignment(0.5, 0.5)
        image.set_name("%s_image" % name)
        label = gtk.Label(text)
        label.set_name("%s_label" % name)
        label.set_use_underline(underline)
        label.set_alignment(0.5, 0.5)
        hbox.pack_start(image, False, False)
        hbox.pack_start(label, False, False)
        alignment.add(hbox)
        button.remove(self.widget.get_child())
        button.add(alignment)
        button.show_all()

        # now add the wrappers
        names = ["%s_%s" % (name, w) for w in ["alignment", "hbox", "image",
                                               "label"]]
        gwidget = widget.get_widget_from_gtk_widget(button)
        project = gwidget.project
        widget.create_widget_from_gtk_widget(alignment, project, names)
        project.add_widget(alignment)

        # set the selection to the button again
        project.selection_set(button, True)

    def _unset_image_on_button(self, button):
        child = button.get_child()
        text = ''
        if isinstance(child, gtk.Alignment):
            # we were using a custom image
            gwidget = widget.get_widget_from_gtk_widget(button)
            project = gwidget.project
            # we have a custom image so we remove all this children
            hbox = child.get_child()
            label = hbox.get_children()[1]
            text = label.get_text()
            button.remove(child)
            button.set_label(text)
            
            # we only need to remove the Alignment since the remove is a
            # recursive operation
            project.remove_widget(child)

        prop = self.gwidget.get_glade_property('use-stock')
        prop.set(False)

        prop = self.gwidget.get_glade_property('use-underline')
        prop.set(False)

        prop = self.gwidget.get_glade_property('label')
        prop.set(text)

        # we need to refresh the property editor since we changed
        # the label property
        project = self.gwidget.project
        project._app.refresh_editor()

    def set_widget(self, gtk_widget, proxy):
        super(ImageSelector, self).set_widget(gtk_widget, proxy)
        self.set_title(_("Editing the image of button %s") % self.gwidget.name)

        self.loading = True
        
        # see if we have a custom image
        child = gtk_widget.get_child()
        if isinstance(child, gtk.Alignment):
            hbox = child.get_child()
            image = hbox.get_children()[0]
            storage_type = image.get_storage_type()
            
            if storage_type == gtk.IMAGE_STOCK:
                stock_id = image.get_stock()[0]
                self._select_stock_item(stock_id)
                self.stock_radio_button.set_active(True)
            else:
                # we have a custom image
                filename = image.get_data('image-file-name')
                self.filename_entry.set_text(filename)
                self.custom_radio_button.set_active(True)

        # or maybe we don't have any image
        else:
            self.no_image_radio_button.set_active(True)

        self.loading = False

class ButtonImageAdaptor:
    def __init__(self):
        self.image_chooser = ImageSelector()
        
    def create_editor(self, context):
        button = gtk.Button(_('Edit...'))
        button.set_data('connection-id', -1)
        return button

    def update_editor(self, context, button, gtk_widget, proxy):
        connection_id = button.get_data('connection-id')
        if connection_id != -1:
            button.disconnect(connection_id)

        connection_id = button.connect('clicked', self._image_editor_edit,
                                       gtk_widget, proxy, context)
        button.set_data('connection-id', connection_id)

    def _image_editor_edit(self, gtk_widget, menubar, proxy, context):
        app_window = context.get_application_window()
        self.image_chooser.set_transient_for(app_window)
    
        self.image_chooser.set_widget(menubar, proxy)
        self.image_chooser.present()

def get_real_label_widget(button):
    """Return the widget that holds the text of the button which is not
    the button itself when we have a custom image.
    """
    if isinstance(button, gtk.Button):
        return button
    else:
        child = button.get_child()
        hbox = child.get_child()
        label = hbox.get_children()[1]
        return label

class ButtonLabelAdaptor:
    """This adaptor takes into account if there is a custom image on the
    button to set and get the label
    """
    def get(self, context, button):
        gtk_widget = get_real_label_widget(button)
        return gtk_widget.get_label()
            
    def set(self, context, button, value):
        gtk_widget = get_real_label_widget(button)
        gtk_widget.set_label(value)
        if gtk_widget != button:
            # we have a custom image
            gwidget = widget.get_widget_from_gtk_widget(gtk_widget)
            prop = gwidget.get_glade_property('label')
            prop.set(value)
    
class ButtonUnderlineAdaptor:
    """This adaptor takes into account if there is a custom image on the
    button to set and get the underline setting
    """
    def get(self, context, button):
        gtk_widget = get_real_label_widget(button)
        return gtk_widget.get_use_underline()
            
    def set(self, context, button, value):
        gtk_widget = get_real_label_widget(button)
        gtk_widget.set_use_underline(value)
    
