/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "parse.h"
#include "matrix.h"
#include "measure.h"
#include "model.h"
#include "morph.h"
#include "select.h"
#include "space.h"
#include "spatial.h"
#include "surface.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"
#include "opengl.h"
#include "zone.h"

#define DEBUG 0

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];
extern GtkWidget *apd_label;

/* globals */
gdouble tmat[9];
gdouble tvec[3];
gdouble edit_anim_n=0;
GtkWidget *elem_entry, *grid_dim, *grid_sep;
GtkWidget *symop_combo, *axis_entry, *obj_entry, *periodicity_spin;
GtkWidget *transmat[12];

enum{AT_ANY, AT_SELECTED, AT_LATTICE, AT_LATMAT};

/*********************************************/
/* update widget values with internal values */
/*********************************************/
void update_transmat(void)
{
gint i, j;
gchar *text;

/* set transformation matrix widget entries */
for (j=0 ; j<3 ; j++)
  {
  for (i=0 ; i<3 ; i++)
    {
    text = g_strdup_printf("%f", tmat[3*j+i]);
/* NB: display transpose */
    gtk_entry_set_text(GTK_ENTRY(transmat[3*j+i]), text);
    g_free(text);
    }
  }

/* set translation widget entries */
for (i=0 ; i<3 ; i++)
  {
  text = g_strdup_printf("%f", tvec[i]);
  gtk_entry_set_text(GTK_ENTRY(transmat[9+i]), text);
  g_free(text);
  }
}

/********************************************/
/* restore original transformation settings */
/********************************************/
void reset_transmat(void)
{
VEC3SET(tvec, 0.0, 0.0, 0.0);
get_rot_matrix(tmat, IDENTITY, NULL, 0);
update_transmat();
}

/*************************************/
/* construct a transformation matrix */
/*************************************/
void construct_transmat(void)
{
gint n, flag=0, type=-1;
guint obj;
const gchar *text;
gdouble angle, mat[16];
GSList *list;
struct vec_pak *vec1, *vec2;
struct spatial_pak *spatial=NULL;
struct model_pak *data;

/* model to act on */
data = sysenv.active_model;
g_return_if_fail(data != NULL);

text = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(symop_combo)->entry));

if (g_ascii_strncasecmp(text, "z alignment", 11) == 0)
  type = ALIGNMENT;
if (g_ascii_strncasecmp(text, "reflection", 10) == 0)
  type = REFLECTION;
if (g_ascii_strncasecmp(text, "rotation", 8) == 0)
  type = PAXIS;
if (g_ascii_strncasecmp(text, "lattice", 7) == 0)
  type = LATMAT;
if (g_ascii_strncasecmp(text, "identity", 8) == 0)
  type = IDENTITY;

g_assert(type >= 0);

/* special cases */
switch(type)
  {
  case IDENTITY:
    reset_transmat();
    return;

  case LATMAT:
    memcpy(tmat, data->latmat, 9*sizeof(gdouble));
    VEC3SET(tvec, 0.0, 0.0, 0.0);
    update_transmat();
    return;
  }

/* rotation angle */
text = gtk_entry_get_text(GTK_ENTRY(axis_entry));
angle = str_to_float(text);

/* check for special reference objects */
text = gtk_entry_get_text(GTK_ENTRY(obj_entry));
if (g_ascii_strncasecmp("x", text, 1) == 0)
  flag = 1;
if (g_ascii_strncasecmp("y", text, 1) == 0)
  flag = 2;
if (g_ascii_strncasecmp("z", text, 1) == 0)
  flag = 3;
if (g_ascii_strncasecmp("a", text, 1) == 0)
  flag = 4;
if (g_ascii_strncasecmp("b", text, 1) == 0)
  flag = 5;
if (g_ascii_strncasecmp("c", text, 1) == 0)
  flag = 6;

/* construct appropriate spatial */
if (flag)
  {
  vec1 = g_malloc(sizeof(struct vec_pak));
  vec2 = g_malloc(sizeof(struct vec_pak));

  VEC3SET(vec1->x, 0.0, 0.0, 0.0);
  switch(flag)
    {
    case 1:
      VEC3SET(vec2->x, 1.0, 0.0, 0.0);
      if (data->periodic)
        vecmat(data->ilatmat, vec2->x);
      break;
    case 2:
      VEC3SET(vec2->x, 0.0, 1.0, 0.0);
      if (data->periodic)
        vecmat(data->ilatmat, vec2->x);
      break;
    case 3:
      VEC3SET(vec2->x, 0.0, 0.0, 1.0);
      if (data->periodic)
        vecmat(data->ilatmat, vec2->x);
      break;

    case 4:
      VEC3SET(vec2->x, 1.0, 0.0, 0.0);
      break;
    case 5:
      VEC3SET(vec2->x, 0.0, 1.0, 0.0);
      break;
    case 6:
      VEC3SET(vec2->x, 0.0, 0.0, 1.0);
      break;

    default:
      g_assert_not_reached();
    }
  spatial = g_malloc(sizeof(struct spatial_pak));
  spatial->type = SPATIAL_VECTOR;
  spatial->data = NULL;
  spatial->data = g_slist_append(spatial->data, vec1);
  spatial->data = g_slist_append(spatial->data, vec2);
  }
else
  {
/* no special reference object - assume a numerical reference */
  obj = (guint) str_to_float(text);
  n=0;
  list = data->spatial;
  while (list)
    {
    if (n == obj)
      {
      spatial = (struct spatial_pak *) list->data;
      break;
      }
    n++;
    list = g_slist_next(list);
    }
  }
 
/* check */
if (!spatial)
  {
  show_text(ERROR, "Undefined reference spatial object.\n");
  return;
  }

/* construct */
make_symmat(mat, data->latmat, (gpointer *) spatial, type, angle);
ARR3SET(&tmat[0], &mat[0]);
ARR3SET(&tmat[3], &mat[4]);
ARR3SET(&tmat[6], &mat[8]);
tvec[0] = mat[3];
tvec[1] = mat[7];
tvec[2] = mat[11];
update_transmat();

/* free if specially created */
if (flag)
  {
  free_slist(spatial->data);
  g_free(spatial);
  }
}

/********************************************/
/* put text entry values into actual matrix */
/********************************************/
void change_transmat(GtkWidget *w, gint i)
{
const gchar *text;

text = gtk_entry_get_text(GTK_ENTRY(transmat[i]));
if (i < 9)
  tmat[i] = str_to_float(text);
else
  tvec[i-9] = str_to_float(text);
}

/************************************************/
/* apply the current transformation/translation */
/************************************************/
void apply_transmat(GtkWidget *w, gint mode)
{
GSList *item, *list=NULL;
struct model_pak *data;
struct core_pak *core;
struct shel_pak *shel;

data = sysenv.active_model;
if (!data)
  return;

/* application mode */
switch(mode)
  {
  case AT_ANY:
    list = data->cores;
    break;

  case AT_SELECTED:
    list = data->selection;
    break;

  case AT_LATTICE:
    new_lattice(tmat, data);
    return;

  case AT_LATMAT:
    matmat(tmat, data->latmat);
    memcpy(data->ilatmat, data->latmat, 9*sizeof(gdouble));
    invmat(data->ilatmat);
    init_objs(REDO_COORDS, data);
    redraw_canvas(SINGLE);
    return;

  default:
    g_assert_not_reached();
  }

for (item=list ; item ; item=g_slist_next(item))
  {
  core = (struct core_pak *) item->data;

/* transform */
  vecmat(data->latmat, core->x);
  vecmat(tmat, core->x);
  ARR3ADD(core->x, tvec);
  vecmat(data->ilatmat, core->x);

/* assoc. shell? */
  if (core->shell)
    {
    shel = (struct shel_pak *) core->shell;

/* transform */
    vecmat(data->latmat, shel->x);
    vecmat(tmat, shel->x);
    ARR3ADD(shel->x, tvec);
    vecmat(data->ilatmat, shel->x);
    }
  }

/* update */
init_objs(REDO_COORDS, data);

/* continuous bond update */
if (data->cbu)
  {
  calc_bonds(data);
  calc_mols(data);
  }
redraw_canvas(SINGLE);
}

/***************************************************/
/* construct animation using multiple applications */
/***************************************************/
void apply_n_transmat(GtkWidget *w, gint mode)
{
gint i;
gdouble mat[9];
struct model_pak *model;
struct transform_pak *transform;

model = sysenv.active_model;
if (!model)
  return;

/* close animation dialog (since num_frames will be changed) */
dialog_destroy_single(ANIM, model);

/* starting point is the current orientation */
memcpy(mat, model->rotmat, 9*sizeof(gdouble));

for (i=0 ; i<edit_anim_n ; i++)
  {
/* create transformations */
  transform = g_malloc(sizeof(struct transform_pak));
  model->transform_list = g_slist_append(model->transform_list, transform);
  transform->id = ROTATION;

transpose(mat);
  matmat(tmat, mat);
transpose(mat);

  memcpy(transform->matrix, mat, 9*sizeof(gdouble));
  VEC3SET(transform->vector, model->offset[0], model->offset[1], 0.0);
  transform->scalar = model->scale;
  model->num_frames++; 
  }

if (model->num_frames)
  model->animation = TRUE;

redraw_canvas(SINGLE);
}

/*********************************************/
/* find and record the maximum region number */
/*********************************************/
gint region_max(struct model_pak *mdata)
{
gint max;
GSList *list;
struct core_pak *core;

max = 0;
for (list=mdata->cores ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  if (core->region > max)
    max = core->region;
  }
return(max);
}

/*************************************************************/
/* region changing routine - for dealing with polar surfaces */
/*************************************************************/
/* can now go both ways via the direction flag (UP or DOWN) */
#define DEBUG_REGION_SWITCH_ATOM 0
gint region_move_atom(struct core_pak *core, gint direction,
                                    struct model_pak *data)
{
gint flag, primary, secondary, mov[2];
gdouble vec[3], tmp[3], d[3];
GSList *list;
struct core_pak *comp;

#if DEBUG_REGION_SWITCH_ATOM
printf("       model: %s\n", data->basename);
printf(" periodicity: %d\n", data->periodic);
printf("         hkl: %f %f %f\n", data->surface.miller[0],
          data->surface.miller[1], data->surface.miller[2]);
printf("        dhkl: %f\n", data->surface.dspacing);
printf("region sizes: %f %f\n", data->surface.region[0], data->surface.region[1]);
printf("      moving: ");
if (direction == UP)
  printf("UP\n");
else
  printf("DOWN\n");
#endif

/* checks */
g_return_val_if_fail(data != NULL, 1);
g_return_val_if_fail(data->periodic == 2, 1);
if (data->surface.region[0] < 1)
  {
  show_text(ERROR, "region 1 is empty.\n");
  return(1);
  }

/* setup region switching labels */
if (direction == UP)
  {
  primary = REGION1A;
  secondary = REGION2A;
  }
else
  {
  primary = REGION2A;
  secondary = REGION1A;
  }

/* get fractional depth translation vector */
ARR3SET(vec, data->surface.depth_vec);
vecmat(data->ilatmat, vec);

/* calculate offset to region boundary */
ARR3SET(tmp, vec);
if (direction == DOWN)
  {
  VEC3MUL(tmp, data->surface.region[0]);
  VEC3MUL(tmp, -1.0);
  }
else
  {
  if (data->surface.region[1] == 0)
    {
    VEC3MUL(tmp, data->surface.region[0]);
    }
  else
    {
    VEC3MUL(tmp, data->surface.region[1]);
    }
  }

/* if region 2 is empty, just move core to the bottom */
if (data->surface.region[1] == 0.0)
  {
  ARR3ADD(core->x, tmp);
  if (core->shell)
    {
    ARR3ADD((core->shell)->x, tmp);
    }
  atom_colour_scheme(data->colour_scheme, core, data);
  return(0);
  }

/* get coordinates of target atom */
ARR3ADD(tmp, core->x);

#if DEBUG_REGION_SWITCH_ATOM
P3VEC("    translation: ", vec);
P3VEC("  target coords: ", tmp);
#endif

/* find the target */
flag=0;
for (list=data->cores ; list ; list=g_slist_next(list))
  {
  comp = (struct core_pak *) list->data;

/* only atoms of the same type need apply */
  if (core->atom_code != comp->atom_code)
    continue;

/* get difference vector */
  ARR3SET(d, comp->x);
  ARR3SUB(d, tmp);

/* pbc constraint */
  while(d[0] < -FRACTION_TOLERANCE)
    d[0] += 1.0;
  while(d[0] > 0.5)
    d[0] -= 1.0;
  while(d[1] < -FRACTION_TOLERANCE)
    d[1] += 1.0;
  while(d[1] > 0.5)
    d[1] -= 1.0;

/* test difference vector's magnitude */
  if (VEC3MAGSQ(d) < FRACTION_TOLERANCE)
    {
/* change its labelling */
#if DEBUG_REGION_SWITCH_ATOM
printf("Matched core: %p\n", comp);
#endif
    comp->region = secondary;
    if (comp->shell)
      {
      (comp->shell)->region = secondary;
      }
    atom_colour_scheme(data->colour_scheme, comp, data);
    flag++;
    break;
    }
  }

if (!flag)
  {
  show_text(ERROR, "Failed to find a boundary image.\n");
  return(1);
  }

/* now move selected atom to bottom of region 2 */
ARR3SET(tmp, vec);
VEC3MUL(tmp, (data->surface.region[0] + data->surface.region[1]));
if (direction == UP)
  VEC3MUL(tmp, -1.0);
ARR3SUB(core->x, tmp);
core->region = primary;
/* pbc constrain */
fractional_clamp(core->x, mov, 2);

if (core->shell)
  {
  ARR3SUB((core->shell)->x, tmp);
  ARR2ADD((core->shell)->x, mov);
  (core->shell)->region = primary;
  }

atom_colour_scheme(data->colour_scheme, core, data);

return(0);
}

/********************************************/
/* move a selection using the above routine */
/********************************************/
#define DEBUG_REGION_MOVE 0
void region_move(GtkWidget *w, gint direction)
{
gchar txt[60];
GSList *list;
struct model_pak *data;
struct core_pak *core;

data = sysenv.active_model;

/* checks */
if (!data)
  return;
if (data->periodic != 2)
  return;
if (!data->selection)
  {
  show_text(WARNING, "Empty selection.\n");
  return;
  }

#if DEBUG_REGION_MOVE 
P3MAT("depth vec:", data->surface.depth_vec);
#endif

/* quit if no depth info ie a loaded 2D file */
if (VEC3MAG(data->surface.depth_vec) < FRACTION_TOLERANCE)
  {
  show_text(ERROR,
  "This is only possible for surfaces generated in the current session.\n");
  return;
  }

/* move all atoms in selection */
for (list=data->selection ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;
  region_move_atom(core, direction, data); 
  }

/* update */
init_objs(REDO_COORDS, data);
calc_bonds(data);
calc_mols(data);

#if DEBUG_REGION_MOVE 
printf("old dipole: %f\n", data->gulp.sdipole);
#endif

/* recompute */
calc_emp(data);

#if DEBUG_REGION_MOVE 
printf("new dipole: %f\n", data->gulp.sdipole);
#endif

/* inform the user */
sprintf(txt, "New surface dipole: %f\n", data->gulp.sdipole);
show_text(STANDARD, txt);

redraw_canvas(SINGLE);
}

/************************************/
/* change a selections region label */
/************************************/
void region_set(GtkWidget *w, GtkWidget *spin)
{
gint region;
GSList *list;
struct model_pak *mdata;
struct core_pak *core;

mdata = sysenv.active_model;

/* checks */
if (!mdata)
  return;
if (GTK_IS_SPIN_BUTTON(spin))
  region = SPIN_IVAL(GTK_SPIN_BUTTON(spin));
else
  return;

/* update maximum region number */
if (region > mdata->region_max)
  mdata->region_max = region;

for (list=mdata->selection ; list ; list=g_slist_next(list))
  {
  core = (struct core_pak *) list->data;

  core->region = region;

  if (core->shell)
    (core->shell)->region = region;

/* update region colouring (if necessary) */
  if (mdata->colour_scheme == REGION)
    atom_colour_scheme(REGION, core, mdata);
  }
redraw_canvas(SINGLE);
}

/****************************/
/* connectivity only update */
/****************************/
void update_bonds(void)
{
struct model_pak *data;

data = sysenv.active_model;
if (!data)
  return;

init_objs(REDO_COORDS, data);
calc_bonds(data);
calc_mols(data);
redraw_canvas(SINGLE);
}

/*******************************/
/* creates a new (blank) model */
/*******************************/
void create_new_model(void)
{
struct model_pak *data;

/* make a slot for the new model */
data = model_new();
sysenv.active_model = data;

/* setup model parameters */
data->id = CREATOR;
strcpy(data->filename, "new_model");
g_free(data->basename);
data->basename = g_strdup("new_model");

/* initialize */
prep_model(data);
data->mode = ATOM_ADD;

/* rmax has to be after init_objs() as INIT_COORDS will set it to 1.0 */
data->rmax = 5.0*RMAX_FUDGE;

/* update/redraw */
tree_model_add(data);
tree_select_model(data);
redraw_canvas(SINGLE);
}

/***********************/
/* add atom event hook */
/***********************/
void add_atom(gdouble *r, struct model_pak *data)
{
gchar *orig, *elem;
struct core_pak *core;
gpointer zone;

/* get the atom type from the label */
orig = (gchar *) gtk_entry_get_text(GTK_ENTRY(apd_label));
elem = g_strdup(orig);

/* add the atom to the model */
core = new_core(elem, data);

/* set atom position in the plane running through the origin */
r[2] = 0.0;
ARR3SET(core->rx, r);
vecmat(data->irotmat, r);
vecmat(data->ilatmat, r);
ARR3ADD(r, data->centroid);
ARR3SET(core->x, r);

data->cores = g_slist_append(data->cores, core);

/* make sure this atom is in the apd box, so we can keep adding this type */
select_clear();
data->selection = g_slist_prepend(data->selection, core);
core->status |= SELECT;

/* NEW - update zones */
/* FIXME - min/max may also need to be adjusted */
zone = zone_ptr(core, data);
zone_append_core(zone, core);

/* updates */
atom_properties_update(core, data);
g_slist_free(data->unique_atom_list);
data->unique_atom_list = find_unique(ELEMENT, data);
init_atom_charge(core, data);
calc_emp(data);

/* done */
g_free(elem);
}

/*********************/
/* move atoms around */
/*********************/
void move_atom(gdouble *r, gint mode)
{
struct model_pak *data;
static struct core_pak *core=NULL;

/* START is called by button_press_event */
/* UPDATE is called by mouse_motion_event */
/* STOP is called when a switch_mode event occurs - ie cleanup */

data = sysenv.active_model;

switch(mode)
  {
  case START:
/* try to find an atom at the mouse pointer */
    core = seek_coord3d(r, data);     
    break;

  case UPDATE:
    if (core)
      {
/* preserve the atom's z coord */
      r[2] = core->rx[2];

/* set the new coords */
      ARR3SET(core->rx, r);
      vecmat(data->irotmat, r);
      vecmat(data->ilatmat, r);
      ARR3ADD(r, data->centroid);
      ARR3SET(core->x, r);

/* continuous bond update */
      if (data->cbu)
        redo_atom_bonds(core, data);
      }
    break;

  case STOP:
    core = NULL;
    break;
  }
}

/*******************************************************/
/* permanently remove all deleted cores from all lists */
/*******************************************************/
#define DEBUG_DELETE_COMMIT 0
void delete_commit(struct model_pak *data)
{
gint flag1=FALSE, flag2=FALSE;
gpointer m;
GSList *list1, *list2;
struct core_pak *core;
struct shel_pak *shell;
struct bond_pak *bond;

g_assert(data != NULL);

/* delete flaged cores in list */
list1 = data->cores;
while (list1)
  {
  core = (struct core_pak *) list1->data;
  list1 = g_slist_next(list1);

  if (core->status & DELETED)
    {
#if DEBUG_DELETE_COMMIT
printf("Deleting %s core [%p] ...\n", core->label, core);
#endif
    flag1 = TRUE;

/* flag assoc. shell */
    if (core->shell)
      {
      shell = core->shell;
      shell->status |= DELETED;
      }

/* update connectivity */
    wipe_atom_bonds(core, data);
    list2 = data->ubonds;
    while (list2)
      {
      bond = (struct bond_pak *) list2->data;
      list2 = g_slist_next(list2);

      if (bond->atom1 == core || bond->atom2 == core)
        {
        data->ubonds = g_slist_remove(data->ubonds, bond);
        g_free(bond);
        }
      }

/* update selection */
    data->selection = g_slist_remove(data->selection, core);

/* delete any labels that reference the deleted core */
    list2 = data->measure_list;
    while (list2)
      {
      m = list2->data;
      list2 = g_slist_next(list2);
      if (measure_has_core(core, m))
        measure_free(m, data);
      }
/* update main list */
    data->cores = g_slist_remove(data->cores, core);
    g_free(core);
    }
  }

/* delete flaged shells in list */
list1 = data->shels;
while (list1)
  {
  shell = (struct shel_pak *) list1->data;
  list1 = g_slist_next(list1);

  if (shell->status & DELETED)
    {
    flag2 = TRUE;
/* update main list */
    data->shels = g_slist_remove(data->shels, shell);
    g_free(shell);
    }
  }

/* refresh totals */
data->num_atoms = g_slist_length(data->cores);
data->num_shells = g_slist_length(data->shels);

/* refresh spatial partitioning */
/* it's probably best to refresh the partitioning here, rather than */
/* incrementally as it's speedups for large models we're targeting */
zone_init(data);

/* cope with deleted bonds - expensive, so only do if required */
if (flag1)
  {
  calc_bonds(data);
  calc_mols(data);
  }

/* refresh net charge calc */
calc_emp(data);

/* refresh unique atom list */
g_slist_free(data->unique_atom_list);
data->unique_atom_list = find_unique(ELEMENT, data);

/* refresh geometric info dialog */
meas_graft_model(data);

/* reset functions with static pointers to cores */
data->state = 0;
}

/**********************************/
/* flag core and associated shell */
/**********************************/
void delete_core(struct core_pak *core)
{
core->status |= DELETED;
if (core->shell)
  (core->shell)->status |= DELETED;
}

/*********************************/
/* delete atom by pixel position */
/*********************************/
void delete_atom_at(gdouble *r, struct model_pak *data)
{
struct core_pak *core;

/* attempt to match point with an atom */
core = seek_coord3d(r, data);     
if (core)
  {
  delete_core(core);
  delete_commit(data);
  }
}

/*******************************/
/* model switching update hook */
/*******************************/
void update_creator()
{
}

/***********************************/
/* change periodicity of the model */
/***********************************/
void cb_modify_periodicity(void)
{
gint n;
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

n = SPIN_IVAL(GTK_SPIN_BUTTON(periodicity_spin));

/* remove all symmetry and convert to cartesian */
space_make_p1(model);

/* init new lattice */
model->periodic = n;

/* uncomment below to use the supplied transformation matrix */
/* instead of the model's own lattice matrix */
/*
for (n=0 ; n<model->periodic ; n++)
  {
  model->latmat[n+0] = tmat[n+0];
  model->latmat[n+3] = tmat[n+3];
  model->latmat[n+6] = tmat[n+6];
  }
*/
model->construct_pbc = TRUE;
prep_model(model);

/* updates */
tree_model_add(model);
dialog_destroy_single(GENSURF, model);
gtksh_relation_update(model);
redraw_canvas(SINGLE);
}

/****************************/
/* object modification page */
/****************************/
/* TODO - molecule fragments (library?) */
void objects_page(GtkWidget *box)
{
GtkWidget *frame, *vbox;

g_return_if_fail(box != NULL);

/* Frame */
frame = gtk_frame_new ("Atoms");
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), PANEL_SPACING);

/* hbox for the modes */
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width (GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_button_x("Add atoms",
               gtk_switch_mode, GINT_TO_POINTER(ATOM_ADD),
               vbox);
gtksh_button_x("Delete atoms",
               gtk_switch_mode, GINT_TO_POINTER(ATOM_DELETE),
               vbox);
gtksh_button_x("Delete molecules",
               gtk_switch_mode, GINT_TO_POINTER(MOL_DELETE),
               vbox);

/* Frame */
frame = gtk_frame_new ("Spatials");
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), PANEL_SPACING);

/* hbox for the modes */
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width (GTK_CONTAINER(vbox), PANEL_SPACING);

/* mode buttons */
gtksh_button_x("Add ribbons", 
               gtk_switch_mode, GINT_TO_POINTER(DEFINE_RIBBON),
               vbox);
gtksh_button_x("Add vectors", 
               gtk_switch_mode, GINT_TO_POINTER(DEFINE_VECTOR),
               vbox);
gtksh_button_x("Add planes",  
               gtk_switch_mode, GINT_TO_POINTER(DEFINE_PLANE),
               vbox);
gtksh_button_x("Delete vectors", 
               gtk_switch_mode, (gpointer) DELETE_VECTOR,
               vbox);
/* Frame */
/*
frame = gtk_frame_new ("Misc");
gtk_box_pack_start(GTK_BOX(box),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), PANEL_SPACING);

vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width (GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_button_x("Add picture", image_import_dialog, NULL, vbox);
*/

/* done */
gtk_widget_show_all(box);
}

/************************/
/* transformations page */
/************************/
void trans_page(GtkWidget *box)
{
gint i, j;
GList *list;
GtkWidget *frame, *table, *hbox, *vbox, *label;

g_assert(box != NULL);

/* transformation construction */
frame = gtk_frame_new("Construction");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);


/* TODO - use spinners instead of gtk entries for these numbers */
/* axes order */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, FALSE, 0);

label = gtk_label_new("Rotation angle (degrees):");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
axis_entry = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), axis_entry, FALSE, FALSE, 0);
/*
gtk_entry_set_text(GTK_ENTRY(axis_entry), "90");
*/

/* spatial object */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, FALSE, 0);

label = gtk_label_new("Reference spatial object: ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
obj_entry = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), obj_entry, FALSE, FALSE, 0);
/*
gtk_entry_set_text(GTK_ENTRY(obj_entry), "0");
*/


/* construction buttons */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

label = gtk_label_new("Construct");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

list = NULL;
list = g_list_prepend(list, "identity matrix");
list = g_list_prepend(list, "lattice matrix");
list = g_list_prepend(list, "reflection matrix");
list = g_list_prepend(list, "rotation matrix");
list = g_list_prepend(list, "z alignment matrix");
list = g_list_reverse(list);

symop_combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(symop_combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(symop_combo), list);
gtk_box_pack_start(GTK_BOX(hbox), symop_combo, FALSE, FALSE, PANEL_SPACING);

gtksh_button_x(NULL, construct_transmat, NULL, hbox);

/* coordinate transformation */
frame = gtk_frame_new("Transformation matrix");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* table */
table = gtk_table_new(5, 4, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

label = gtk_label_new("a*");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);
label = gtk_label_new("b*");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
label = gtk_label_new("c*");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);

label = gtk_label_new("a");
gtk_table_attach_defaults(GTK_TABLE(table),label,1,2,0,1);
label = gtk_label_new("b");
gtk_table_attach_defaults(GTK_TABLE(table),label,2,3,0,1);
label = gtk_label_new("c");
gtk_table_attach_defaults(GTK_TABLE(table),label,3,4,0,1);
label = gtk_label_new("t");
gtk_table_attach_defaults(GTK_TABLE(table),label,4,5,0,1);

/* column loop */
for (j=0 ; j<3 ; j++)
  {
/* row loop */
  for (i=0 ; i<3 ; i++)
    {
    transmat[3*j+i] = gtk_entry_new();
    gtk_table_attach_defaults(GTK_TABLE(table),transmat[3*j+i],i+1,i+2,j+1,j+2);
    gtk_widget_set_size_request(transmat[3*j+i], 7*sysenv.gtk_fontsize, -1);
    }
  }
/* translation */
for (j=0 ; j<3 ; j++)
  {
  transmat[9+j] = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),transmat[9+j],4,5,j+1,j+2);
  gtk_widget_set_size_request(transmat[9+j], 7*sysenv.gtk_fontsize, -1);
  }

/* control buttons */
frame = gtk_frame_new("Coordinate transformations");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_button_x("Apply to all atoms", 
               apply_transmat, GINT_TO_POINTER(AT_ANY),
               vbox);
gtksh_button_x("Apply to selected atoms",
               apply_transmat, GINT_TO_POINTER(AT_SELECTED),
               vbox);

/* animation construction */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Generate animation frames ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
gtksh_direct_spin(NULL, &edit_anim_n, 0.0, 1000.0, 1.0, NULL, NULL, hbox);
gtksh_button_x(NULL, apply_n_transmat, GINT_TO_POINTER(AT_ANY), hbox);


/* control buttons */
frame = gtk_frame_new("Lattice transformations");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_button_x("Apply to lattice matrix",
               apply_transmat, GINT_TO_POINTER(AT_LATMAT),
               vbox);
gtksh_button_x("Create new lattice model from linear combination",
               apply_transmat, GINT_TO_POINTER(AT_LATTICE),
               vbox);

hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

label = gtk_label_new("Alter lattice periodicity ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

periodicity_spin = gtk_spin_button_new_with_range(0, 3, 1);
gtk_box_pack_start(GTK_BOX(hbox), periodicity_spin, FALSE, FALSE, 0);

gtksh_button_x(NULL, cb_modify_periodicity, NULL, hbox);
}

/********************************************/
/* periodicity/lattice transformations page */
/********************************************/
/* TODO - too few options to justify its own page */
/*
void periodicity_page(GtkWidget *box)
{
GtkWidget *frame, *vbox;

frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX (box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_button_x("Force structure to P1", make_current_p1, NULL, vbox);
}
*/

/*************************************/
/* save a diffraction (DIFFAX) setup */
/*************************************/
struct model_pak *diffract_model=NULL;

void diffract_save(gchar *name)
{
write_diffax(name, diffract_model);

dialog_destroy_type(FILE_SELECT);
}

/******************************************/
/* create a file dialog for a DIFFAX save */
/******************************************/
void diffract_save_dialog(GtkWidget *w, struct model_pak *model)
{
diffract_model = model;
file_dialog("Save DIFFAX file", model->basename, FILE_SAVE,
            (gpointer) diffract_save, DIFFAX_INP);
}

/*******************/
/* add a new layer */
/*******************/
GtkWidget *diffract_layer_total;

#define DEBUG_DIFFRACT_LAYER_SETUP 0
void diffract_layer_setup(struct model_pak *data)
{
gint n, region;
gint num_layer, tot_layer;
gdouble start, stop;
GSList *clist, *list1=NULL;
struct layer_pak *layer;
struct core_pak *core;

if (!data)
  return;

if (GTK_IS_SPIN_BUTTON(diffract_layer_total))
  tot_layer = SPIN_IVAL(GTK_SPIN_BUTTON(diffract_layer_total));
else
  return;

/* get rid of old list */
free_slist(data->layer_list);
data->layer_list = NULL;

for (num_layer=0 ; num_layer<tot_layer ; num_layer++)
  {

/* create the new layer */
layer = g_malloc(sizeof(struct layer_pak));
layer->width = 1.0/(gdouble) tot_layer;
VEC3SET(layer->centroid, 0.0, 0.0, 0.0);

start = (gdouble) num_layer / (gdouble) tot_layer;
stop = (gdouble) (num_layer+1) / (gdouble) tot_layer;

#if DEBUG_DIFFRACT_LAYER_SETUP
printf("new layer: %f-%f (%f)\n", start, stop, layer->width);
#endif

region = num_layer;

/* add cores that satisfy the start/stop condition */
list1 = NULL;
for (clist=data->cores ; clist ; clist=g_slist_next(clist))
  {
  core = (struct core_pak *) clist->data;

/* FIXME - floating point boundary problems? */
  if (core->x[2] >= start && core->x[2] < stop)
    {
    list1 = g_slist_prepend(list1, core); 
    ARR3ADD(layer->centroid, core->x);
    core->region = region;
/* update region colouring (if necessary) */
    if (data->colour_scheme == REGION)
      atom_colour_scheme(REGION, core, data);
    }
  }

/* centroid calc. */
n = g_slist_length(list1);
if (n)
  {
#if DEBUG_DIFFRACT_LAYER_SETUP
printf("[%d] added layer with %d cores.\n", num_layer, n);
#endif

  layer->cores = list1;
  VEC3MUL(layer->centroid, 1.0/(gdouble) n);
  }

/* always save the layer - may *want* to have an empty layer */
/* eg to simulate a vacuum gap */
data->layer_list = g_slist_prepend(data->layer_list, layer);
  }

#if DEBUG_DIFFRACT_LAYER_SETUP
printf("Total layers: %d\n", g_slist_length(data->layer_list));
#endif

redraw_canvas(SINGLE);
}

/***********************************************/
/* create a defect structure from input layers */
/***********************************************/
GtkWidget *diffract_layer_order;

#define DEBUG_DIFFRACT_MODEL_CREATE 0
void diffract_model_create(GtkWidget *w, struct model_pak *model)
{
gint i, n, num_layer, tot_layer;
gdouble new_width, old_width, offset;
const gchar *text;
GSList *list;
struct model_pak *dest_model;
struct layer_pak *layer;
struct core_pak *core;

if (!model)
  return;

diffract_layer_setup(model);
text = gtk_entry_get_text(GTK_ENTRY(diffract_layer_order));

/* create a new model */
dest_model = model_new();
g_return_if_fail(dest_model != NULL);
/* NB: not DIFFAX_INP, since this is demonstrating a particular layer packing */
dest_model->id = CREATOR;

strcpy(dest_model->filename, "new_model");
g_free(dest_model->basename);
dest_model->basename = g_strdup("new model");

/* find how many valid layers we will create */
tot_layer = 0;
for (i=0 ; i<strlen(text) ; i++)
  {
  if (g_ascii_isdigit(text[i]))
    {
    n = text[i] - '0';
    layer = g_slist_nth_data(model->layer_list, n-1);
    if (layer)
      tot_layer++;
    }
  }

#if DEBUG_DIFFRACT_MODEL_CREATE
printf("Requested layers: %d\n", tot_layer);
#endif

/* build the new model from the input layer string */
num_layer=0;
new_width=1.0/tot_layer;
old_width=1.0;

for (i=strlen(text) ; i-- ; )
  {
  if (g_ascii_isdigit(text[i]))
    {
    n = text[i] - '0';
/* NB: numbering starts from 0 */
    layer = g_slist_nth_data(model->layer_list, n-1);
    if (layer)
      {
      old_width = layer->width;

/* add j based z offset */
      offset = num_layer+0.5;
      offset *= new_width;

#if DEBUG_DIFFRACT_MODEL_CREATE
printf("[%d] inserting layer: %d (scale: %f) (offset: %f)\n",
        num_layer, n, new_width/old_width, offset);
#endif

/* add the layer's cores */
      for (list=layer->cores ; list ; list=g_slist_next(list))
        {
        core = dup_core(list->data);

        core->region = tot_layer - i - 1;

/* remove z centroid */
        core->x[2] -= layer->centroid[2];

/* z values need to be SCALED (to meet the new layer width) */
        core->x[2] *= new_width/old_width;

        core->x[2] += offset;

        dest_model->cores = g_slist_prepend(dest_model->cores, core);
        }
      num_layer++;
      }
    else
      printf("Layer number %d not defined.\n", n);
    }
  }

/* test if anything was created */
if (!num_layer)
  {
  model_delete(dest_model);
  return;
  }

dest_model->cores = g_slist_reverse(dest_model->cores);

/* setup */
dest_model->periodic = 3;
dest_model->fractional = TRUE;
ARR3SET(&dest_model->pbc[0], &model->pbc[0]);
ARR3SET(&dest_model->pbc[3], &model->pbc[3]);
dest_model->pbc[2] *= num_layer*old_width;

#if DEBUG_DIFFRACT_MODEL_CREATE
printf("Setting c to: %f\n", dest_model->pbc[2]);
#endif

prep_model(dest_model);

model_colour_scheme(model->colour_scheme, dest_model);

tree_model_add(dest_model);
}

/**********************************/
/* callback for forcefield typing */
/**********************************/
void cb_type_model(GtkWidget *w, gpointer data)
{
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

type_model(gtk_entry_get_text(GTK_ENTRY(data)), model);

/* possible label update */
redraw_canvas(ALL);
}

/**************************/
/* cell sculpting globals */
/**************************/
gdouble sculpt_length = 20.0;

/***************************/
/* cell sculpting callback */
/***************************/
#define DEBUG_SCULPT_CREATE 0
void sculpt_model_create(void)
{
gint i, t[3], limit[3];
gdouble r, x[3];
GSList *list, *clist, *slist, *plist;
struct model_pak *model, *temp, *dest;
struct core_pak *core;
struct shel_pak *shell;
struct plane_pak *plane;

/* get source model and template */
model = sysenv.active_model;
/*
temp = model_ptr(sysenv.active+1, RECALL);
*/
temp = NULL;

if (!model || !temp)
  {
  show_text(ERROR, "Not enough input models.\n");
  return;
  }
if (model->periodic != 3 || temp->id != MORPH)
  {
  show_text(ERROR, "Bad input models.\n");
  return;
  }

/* compute required periodic images (assumed symmetric) */
VEC3SET(limit, 0, 0, 0);
for (i=0 ; i<model->periodic ; i++)
  limit[i] = 1 + 0.5*sculpt_length/model->pbc[i];

/* init destination model for sculpture */
dest = model_new();
if (!dest)
  {
  show_text(ERROR, "Failed to allocate for new model.\n");
  return;
  }
template_model(dest);

#if DEBUG_SCULPT_CREATE 
printf("maximum length: %f\n", sculpt_length);
printf("periodic images: %d %d %d\n", limit[0], limit[1], limit[2]);
#endif

/* periodic image creation */
/* TODO - combine this with the plane test (so we don't create/alloc what we don't need) */
for (t[0] = -limit[0] ; t[0] <= limit[0] ; t[0]++)
  {
  for (t[1] = -limit[1] ; t[1] <= limit[1] ; t[1]++)
    {
    for (t[2] = -limit[2] ; t[2] <= limit[2] ; t[2]++)
      {

/* duplicate cores, add periodic image offset & convert to cartesian */
      clist = dup_core_list(model->cores);
      slist = dup_shell_list(model->shels);
      dest->cores = g_slist_concat(dest->cores, clist);
      dest->shels = g_slist_concat(dest->shels, slist);
      for (list=clist ; list ; list=g_slist_next(list))
        {
        core = list->data;
        ARR3ADD(core->x, t);
        vecmat(model->latmat, core->x);
        }
      for (list=slist ; list ; list=g_slist_next(list))
        {
        shell = list->data;
        ARR3ADD(shell->x, t);
        vecmat(model->latmat, shell->x);
        }
      }
    }
  }

/* plane cutoff tests */
#if DEBUG_SCULPT_CREATE 
printf("rmax = %f\n", temp->rmax);
#endif

for (plist = temp->planes ; plist ; plist=g_slist_next(plist))
  {
  plane = plist->data;
  if (plane->present)
    {
/* compute distance to plane facet */
    switch(temp->morph_type)
      {
      case EQUIL_UN:
        r = plane->esurf[0];
        break;
      case GROWTH_UN:
        r = fabs(plane->eatt[0]);
        break;
      case EQUIL_RE:
        r = plane->esurf[1];
        break;
      case GROWTH_RE:
        r = fabs(plane->eatt[1]);
        break;
      case DHKL:
      default:
        r = 1.0/plane->dhkl;
        break;
      }
    r *= 0.5*sculpt_length/temp->rmax;

#if DEBUG_SCULPT_CREATE 
printf("[%d %d %d] : %f\n ", plane->index[0], plane->index[1], plane->index[2], r);
#endif

/* project coords onto plane normal & remove if greater than distance to facet */
/* TODO - do cutoff by molecule centroid when combining with periodic image loop */
    for (list=dest->cores ; list ; list=g_slist_next(list))
      {
      core = list->data;

      ARR3SET(x, core->x);
      ARR3MUL(x, plane->norm);

      if ((x[0]+x[1]+x[2]) > r)
        {
        core->status |= DELETED;
        }
      }
    }
  }

/* init for display */
delete_commit(dest);
prep_model(dest);
tree_model_add(dest);
redraw_canvas(ALL);
}

/*******************************************/
/* unit cell sculpting (eg via morphology) */
/*******************************************/
void sculpting_page(GtkWidget *box)
{
GtkWidget *hbox, *hbox2;

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 0);
gtksh_direct_spin("Maximum length", &sculpt_length, 10.0, 100.0, 10.0, NULL, NULL, hbox);

/* action buttons */
hbox2 = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(box), hbox2, FALSE, FALSE, PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_end(GTK_BOX(hbox2), hbox, TRUE, FALSE, 0);

gtksh_icon_button(GTK_STOCK_APPLY, "Create ",
                  sculpt_model_create, NULL,
                  hbox);
}

/*******************************/
/* core labelling manipulation */
/*******************************/
void labelling_page(GtkWidget *box)
{
GtkWidget *frame, *hbox, *vbox, *label, *spin;
struct model_pak *data;

data = sysenv.active_model;
g_return_if_fail(data != NULL);

/* Surface options */
frame = gtk_frame_new("Regions");
gtk_box_pack_start(GTK_BOX (box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);
gtksh_button_x("Move selection up", region_move, (gpointer) UP, vbox);
gtksh_button_x("Move selection down", region_move, (gpointer) DOWN, vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

label = gtk_label_new ("Mark selection as region ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

spin = gtk_spin_button_new_with_range(0, 9, 1);
gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 0);

gtksh_button_x(NULL, region_set, (gpointer) spin, hbox);

/* Surface options */
frame = gtk_frame_new("Ghosts");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);
gtksh_button_x("Mark selection as normal", set_selection_normal, NULL, vbox);
gtksh_button_x("Mark selection as ghosts", set_selection_ghosts, NULL, vbox);

#if FF_TYPING
/* NEW - forcefield assignment */
frame = gtk_frame_new("Typing");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER (frame), PANEL_SPACING);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* typing setup */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

label = gtk_label_new("Assign atom: ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

list = NULL;
list = g_list_prepend(list, "QEq charges");
list = g_list_prepend(list, "CVFF labels");

combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, PANEL_SPACING);

gtksh_button_x(NULL, cb_type_model, GTK_COMBO(combo)->entry, hbox);
#endif

#if DIFFAX
/* DIFFAX stuff - shunted here */
/* frame */
frame = gtk_frame_new("DIFFAX");
gtk_box_pack_start(GTK_BOX(box), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);

/* source */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new(g_strdup_printf("Source model: %s", data->basename));
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* layer subdivision */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Number of layers ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

diffract_layer_total = gtk_spin_button_new_with_range(0, 10, 1);
gtk_spin_button_set_value(GTK_SPIN_BUTTON(diffract_layer_total), 0);
gtk_box_pack_end(GTK_BOX(hbox), diffract_layer_total, FALSE, FALSE, 0);

/* layer stacking */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, PANEL_SPACING);

label = gtk_label_new("Stacking sequence ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
diffract_layer_order = gtk_entry_new();
gtk_box_pack_end(GTK_BOX(hbox), diffract_layer_order, FALSE, FALSE, 0);

/* action buttons */
hbox2 = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox2, FALSE, FALSE, PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_end(GTK_BOX(hbox2), hbox, TRUE, FALSE, 0);

gtksh_icon_button(GTK_STOCK_APPLY, "Create ",
                  diffract_model_create, data,
                  hbox);
gtksh_icon_button(GTK_STOCK_SAVE, "Save  ",
                  diffract_save_dialog, data,
                  hbox);
#endif
}

/****************************/
/* the model editing dialog */
/****************************/
void med_interface()
{
gint i;
gpointer dialog;
GtkWidget *window, *frame, *label;
GtkWidget *notebook, *page;
struct model_pak *data;

data = sysenv.active_model;
if (!data)
  return;

/* request a new dialog */
dialog = dialog_request(CREATOR, "Model editing", NULL, NULL, data);
if (!dialog)
  return;
window = dialog_window(dialog);

/* notebook frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_container_add(GTK_CONTAINER(frame), notebook);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);

/* add page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new("Connectivity");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
connect_page(page);

/* add page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new("Objects");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
objects_page(page);

/* add page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new("Transformations");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
trans_page(page);

/* add page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new("Periodicity");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
periodicity_page(page);


/* add page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new("Sculpting");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
sculpting_page(page);


/* add page */
page = gtk_vbox_new(FALSE,0);
label = gtk_label_new("Labelling");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
labelling_page(page);


/* terminating buttons */
gtksh_stock_button(GTK_STOCK_REFRESH, connect_refresh, NULL,
                   GTK_DIALOG(window)->action_area);
gtksh_stock_button(GTK_STOCK_CLOSE, dialog_destroy, dialog,
                   GTK_DIALOG(window)->action_area);

gtk_widget_show_all(window);
 
/* init the transformation values */
reset_transmat();
for (i=0 ; i<12; i++)
  g_signal_connect(GTK_OBJECT(transmat[i]), "changed", 
                   GTK_SIGNAL_FUNC(change_transmat), (gpointer) i);
}
