/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <GL/gl.h>

#ifndef __WIN32
#include <sys/wait.h>
#endif

#include "gdis.h"
#include "coords.h"
#include "edit.h"
#include "file.h"
#include "parse.h"
#include "task.h"
#include "morph.h"
#include "matrix.h"
#include "opengl.h"
#include "render.h"
#include "select.h"
#include "spatial.h"
#include "gtkshorts.h"
#include "interface.h"
#include "dialog.h"

#define DEBUG 0
#define DELETE_POV_FILE 0

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];
extern GtkWidget *window;

/* global render parameters */
struct light_pak current_light;
GtkWidget *scale_spin;

#define SCALE_MAG 10
#define PIX2ANG 0.03

/* this is an OpenGL limitation (ie shininess) */
#define MAX_HL 128

/****************/
/* render setup */
/****************/
void render_setup(GtkWidget *w, gpointer dummy)
{
/* checks */
if (!sysenv.povray_path)
  {
  show_text(ERROR, "POVRay executable was not found.\n");
  return;
  }

/* single model background job */
run_pov();
}

/******************************/
/* font cancel button handler */
/******************************/
void font_dialog_close(GtkWidget *w, gpointer fsd)
{
gtk_widget_destroy(GTK_WIDGET(fsd));
}

/**************************/
/* font ok button handler */
/**************************/
void gl_font_selected(GtkWidget *w, gpointer fsd)
{
gchar *fn;

fn = gtk_font_selection_dialog_get_font_name ((GtkFontSelectionDialog *) fsd);
strcpy(sysenv.gl_fontname, fn);
redraw_canvas(ALL);
}

/**********************/
/* drawing font selection */
/**********************/
void gl_font_dialog(void)
{
GtkWidget *fsd;

fsd =  gtk_font_selection_dialog_new("Font selection");

/* setup events for the file selection widget */
g_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(fsd)->ok_button),
                 "clicked", (GtkSignalFunc) gl_font_selected, fsd);
g_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(fsd)->cancel_button),
                 "clicked", (GtkSignalFunc) font_dialog_close, fsd);

gtk_widget_show(fsd);
}

/************************************/
/* event handler for display dialog */
/************************************/
void render_refresh(void)
{
redraw_canvas(ALL);
}

/********************/
/* Property toggles */
/********************/
void toggle_axes_type(void)
{
struct model_pak *model;

model = sysenv.active_model;
if (!model)
 return;

if (model->periodic)
  {
  if (model->axes_type == CARTESIAN)
    model->axes_type = OTHER;
  else
    model->axes_type = CARTESIAN;
  }
calc_coords(REFRESH, model);
redraw_canvas(SINGLE);
}
void cs_toggle(GtkWidget *w, gpointer data)
{
struct model_pak *model;

model = sysenv.active_model;
if (!model)
 return;

if (model->sof_colourize)
  init_objs(SOF_COLOUR, model);
else
  init_objs(REFRESH_COLOUR, model);
redraw_canvas(SINGLE);
}
void morph_toggle(GtkWidget *w, gpointer data)
{
redraw_canvas(SINGLE);
}

/************************/
/* zoom factor callback */
/************************/
void render_zoom_changed(GtkWidget *w, struct model_pak *model)
{
gdouble zoom;

/* FIXME - for some reason model->zoom hasn't been updated */
/* at this point so we need to get the widget's value */
zoom = SPIN_FVAL(GTK_SPIN_BUTTON(w));

g_assert(zoom > 0.001);

model->scale = model->rmax/zoom;

init_objs(REDO_COORDS, model);

redraw_canvas(SINGLE);
}

/****************/
/* unified hook */
/****************/
gint event_render_modify(GtkWidget *w, gpointer *obj)
{
gint id, refresh=0;
const gchar *entry;
struct model_pak *data;

/* checks */
g_return_val_if_fail(obj != NULL, FALSE);

/* ascertain type of modification required */
id = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(obj), "id"));

data = sysenv.active_model;

sysenv.moving = FALSE;

switch(id)
  {

/*
  case STICK:
  case BALL_STICK:
  case LIQUORICE:
  case CPK:
    sysenv.render.type = id;

if (data->selection)
  core_render_mode_set(id, data->selection);
else
  {
  core_render_mode_set(id, data->cores);
  sysenv.render.type = id;
  }

    refresh++;
    break;
*/

  case ANIM_GIF:
    sysenv.render.animate_type = ANIM_GIF;
    break;
  case ANIM_MPEG:
    sysenv.render.animate_type = ANIM_MPEG;
    break;
  case ANIM_NAME:
    entry = gtk_entry_get_text(GTK_ENTRY(obj));
    g_free(sysenv.render.animate_file);
    sysenv.render.animate_file = g_strdup(entry);
    break;
  case LIGHT_TYPE:
    entry = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(obj)->entry));
    if (g_ascii_strncasecmp(entry, "Positional", 10) == 0)
      current_light.type = POSITIONAL;
    else
      current_light.type = DIRECTIONAL;
    break;
  case MORPH_FINISH:
    entry = (gchar *) gtk_entry_get_text(GTK_ENTRY(obj));
    g_free(sysenv.render.morph_finish);
    sysenv.render.morph_finish = g_strdup(entry);
    refresh++;
    break;
  }

if (refresh)
  redraw_canvas(ALL);

return(FALSE);
}

/****************************/
/* rendering mode for model */
/****************************/
void render_mode_set(GtkWidget *w, gpointer data)
{
gint mode;
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

mode = GPOINTER_TO_INT(data);

spatial_delete_type(SPATIAL_POLYHEDRA, model);

/* CURRENT - fine grained rendering modes */
if (model->selection)
  core_render_mode_set(mode, model->selection);
else
  {
  core_render_mode_set(mode, model->cores);

/* deprec. */
  sysenv.render.type = mode;
  }
redraw_canvas(SINGLE);
}

/*************************************/
/* wire-frame / solid atom rendering */
/*************************************/
void render_wire_atoms(void)
{
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

if (model->selection)
  core_render_wire_set(TRUE, model->selection);
else
  core_render_wire_set(TRUE, model->cores);

redraw_canvas(SINGLE);
}

void render_solid_atoms(void)
{
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

if (model->selection)
  core_render_wire_set(FALSE, model->selection);
else
  core_render_wire_set(FALSE, model->cores);

redraw_canvas(SINGLE);
}

/************************************/
/* toggle the render to file option */
/************************************/
/*
void toggle_animate(GtkWidget *w, GtkWidget *a_frame)
{
if (sysenv.render.animate)
  gtk_widget_set_sensitive(GTK_WIDGET(a_frame), TRUE);
else
  gtk_widget_set_sensitive(GTK_WIDGET(a_frame), FALSE);
}
*/

/*****************************/
/* task orientated rendering */
/*****************************/
void exec_pov_task(gpointer *ptr)
{
GString *cmd;

g_return_if_fail(ptr != NULL);
cmd = g_string_new(NULL);

/* build the command line */
g_string_sprintf(cmd, "%s +I%s -Ga -P +W%d +H%d +FT",
                       sysenv.povray_path, (gchar *) ptr, 
                       (gint) sysenv.render.width, (gint) sysenv.render.height); 
if (sysenv.render.antialias)
  g_string_sprintfa(cmd, " +A +AM2");

task_sync(cmd->str);

g_string_free(cmd, TRUE);
}

/***************************/
/* task orientated viewing */
/***************************/
void exec_img_task(gpointer *ptr)
{
gchar *cmd, *basename, *fullname;

/* post povray command */
g_return_if_fail(ptr != NULL);

/* construct image name */
basename = strdup_basename((gchar *) ptr);
g_free(ptr);
fullname = g_build_filename(sysenv.cwd, basename, NULL);

/* remove .pov file */
if (sysenv.render.no_keep_tempfiles)
  {
  cmd = g_strdup_printf("%s.pov", fullname);
  unlink(cmd);
  g_free(cmd);
  }

/* construct viewing task command */
cmd = g_strdup_printf("%s %s.tga", sysenv.viewer_path, fullname);
g_spawn_command_line_async(cmd, NULL);

/* cleanup */
g_free(basename);
g_free(fullname);
g_free(cmd);
}

/*************************/
/* foreground rendering  */
/*************************/
void exec_pov(gchar *name)
{
GString *cmd;

cmd = g_string_new(NULL);

/* build the command line */
g_string_sprintf(cmd,"%s +I%s -GA -P +W%d +H%d +FT ",
                      sysenv.povray_path, name,
                      (gint) sysenv.render.width , (gint) sysenv.render.height);
if (sysenv.render.antialias)
  g_string_sprintfa(cmd,"+A +AM2 ");

/* after rendering delete input file, */
g_string_sprintfa(cmd,"; rm -rf %s ",name);

/* execute */
system(cmd->str);
printf("\n");
g_string_free(cmd, TRUE);
}

/************************/
/* background rendering */
/************************/
void run_pov()
{
gchar *basename, *fullname;
struct model_pak *model;

model = sysenv.active_model;
if (!model)
  return;

/* make an input file */
basename = gun("pov");

fullname = g_build_filename(sysenv.cwd, basename, NULL);

write_povray(fullname, model);

task_new("POVRay", &exec_pov_task, basename, &exec_img_task, basename, NULL);

g_free(fullname);
}

/*************/
/* callbacks */
/*************/
void geom_label_toggle(void)
{
redraw_canvas(SINGLE);
}

void update_geom_line_width(GtkWidget *w, gpointer *ptr)
{
sysenv.render.geom_line_width = GTK_ADJUSTMENT(w)->value;
redraw_canvas(SINGLE);
}

void render_polyhedral(void)
{
struct model_pak *model;

model = sysenv.active_model;

create_polyhedra(model);
init_objs(REDO_COORDS, model);
redraw_canvas(SINGLE);
}

/***********************/
/* main rendering page */
/***********************/
void render_main_page(GtkWidget *box, struct model_pak *data)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame;
GtkWidget *spin;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* left pane */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add (GTK_CONTAINER(frame), vbox);

gtksh_button_x("Ball & Stick", render_mode_set, GINT_TO_POINTER(BALL_STICK), vbox);
gtksh_button_x("CPK", render_mode_set, GINT_TO_POINTER(CPK), vbox);
gtksh_button_x("Liquorice", render_mode_set, GINT_TO_POINTER(LIQUORICE), vbox);
gtksh_button_x("Polyhedral", render_polyhedral, NULL,  vbox);
gtksh_button_x("Stick", render_mode_set, GINT_TO_POINTER(STICK), vbox);

gtksh_button_x("Wire frame molecules", render_wire_atoms, NULL, vbox);
gtksh_button_x("Solid molecules", render_solid_atoms, NULL, vbox);


/* extra toggles */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

gtksh_direct_check("Antialias", &sysenv.render.antialias,
                   render_refresh, NULL, vbox);
gtksh_direct_check("Wire frame surfaces", &sysenv.render.wire_surface,
                   render_refresh, NULL, vbox);
gtksh_direct_check("Hidden lines", &sysenv.render.wire_show_hidden,
                   render_refresh, NULL, vbox);


/* zoom frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

spin = gtksh_auto_spin("Zoom factor", &data->zoom, 0.1, 500.0, 1.0,
                render_zoom_changed, data, vbox);
gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 2);

/* surface transmission frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
/* ghost atoms and shells */
gtksh_direct_spin("Ghost atom opacity",
                  &sysenv.render.ghost_opacity, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);
/* molecular surfaces */
gtksh_direct_spin("Surface opacity",
                  &sysenv.render.transmit, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);


/* radii frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);

vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_spin("Ball radius",
                  &sysenv.render.ball_rad, 0.1, 0.5, 0.02,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Cylinder radius",
                  &sysenv.render.stick_rad, 0.02, 0.5, 0.01,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Stick thickness",
                  &sysenv.render.stick_thickness, 0.1, 5.0, 0.05,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Frame radius",
                  &sysenv.render.frame_thickness, 0.1, 5.0, 0.05,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("CPK scaling",
                  &sysenv.render.cpk_scale, 0.1, 3.0, 0.05,
                  render_refresh, NULL, vbox);

/* highlighting frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 5);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_direct_spin("Atom highlight power",
                   &sysenv.render.ahl_strength, 0.0, 1.0, 0.05,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Atom highlight focus",
                   &sysenv.render.ahl_size, 0.0, MAX_HL, 5.0,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Surface highlight power",
                   &sysenv.render.shl_strength, 0.0, 1.0, 0.05,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Surface highlight focus",
                   &sysenv.render.shl_size, 0.0, MAX_HL, 5.0,
                   render_refresh, NULL, vbox);

/* ribbon control frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_direct_spin("Ribbon curvature control",
                  &sysenv.render.ribbon_curvature, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Ribbon thickness",
                  &sysenv.render.ribbon_thickness, 0.4, 6.0, 0.2,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Ribbon quality",
                  &sysenv.render.ribbon_quality, 1.0, 30.0, 1.0,
                  render_refresh, NULL, vbox);
}

/************************************/
/* colouring method change callback */
/************************************/
void set_colour_scheme(GtkWidget *w, gpointer *data)
{
const gchar *tmp;
struct model_pak *model;

/*
tmp = gtk_entry_get_text(GTK_ENTRY(w));
*/
tmp = gtk_entry_get_text(GTK_ENTRY(data));

model = sysenv.active_model;
if (!model)
  return;

/* default to atom selection */
if (g_ascii_strncasecmp(tmp, "element", 7) == 0)
  model_colour_scheme(ELEM, model);

if (g_ascii_strncasecmp(tmp, "molecule", 8) == 0)
  model_colour_scheme(MOL, model);

if (g_ascii_strncasecmp(tmp, "region", 6) == 0)
  model_colour_scheme(REGION, model);

redraw_canvas(SINGLE);
}

/**************************/
/* colour control options */
/**************************/
void render_colours_section(GtkWidget *box)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame, *combo;
GtkWidget *label, *button;
GtkStyle *style;
GList *list=NULL;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);


/* TODO - simplify by having a combo of names, and a single colour editing box */
/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* background colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Background colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.bg_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.bg_colour[0]*65535.0;
style->bg[0].green = sysenv.render.bg_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.bg_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);

/* crystal colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Crystal colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.morph_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.morph_colour[0]*65535.0;
style->bg[0].green = sysenv.render.morph_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.morph_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);

/* re-entrant colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Re-entrant colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.rsurf_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.rsurf_colour[0]*65535.0;
style->bg[0].green = sysenv.render.rsurf_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.rsurf_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);

/* ribbon colour editing */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Ribbon colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), sysenv.render.ribbon_colour);
style = gtk_style_copy(gtk_widget_get_style(button));
style->bg[0].red   = sysenv.render.ribbon_colour[0]*65535.0;
style->bg[0].green = sysenv.render.ribbon_colour[1]*65535.0;
style->bg[0].blue  = sysenv.render.ribbon_colour[2]*65535.0;
gtk_widget_set_style(GTK_WIDGET(button), style);


/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* NEW - colouring scheme */
/* TODO - radio button */
list = NULL;
list = g_list_append(list, "element");
list = g_list_append(list, "molecule");
list = g_list_append(list, "region");

hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

label = gtk_label_new ("Colouring scheme ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);

/*
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(set_colour_scheme), NULL);

g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "activate", 
                 GTK_SIGNAL_FUNC(set_colour_scheme), NULL);
*/

gtksh_stock_button(GTK_STOCK_REFRESH, set_colour_scheme, GTK_COMBO(combo)->entry, hbox);
}

/**********************/
/* light list globals */
/**********************/
enum 
{
RENDER_LIGHT_COLOUR,
RENDER_LIGHT_VECTOR,
RENDER_LIGHT_TYPE,
RENDER_LIGHT_ATTRIBUTE,
RENDER_LIGHT_NCOLS
};

GtkWidget *render_light_tv;
GtkListStore *render_light_ls;


/****************************************************/
/* convert 3 doubles [0:1] into a hex colour string */
/****************************************************/
gchar *get_hex_colour(gdouble *colour)
{
guint r, g, b;
gdouble rgb[3];

ARR3SET(rgb, colour);
VEC3MUL(rgb, 255.0);

r = rgb[0];
g = rgb[1];
b = rgb[2];

return(g_strdup_printf("#%2X%2X%2X", r, g, b));
}

/**********************************/
/* get the currently selected row */
/**********************************/
gint render_light_selected(void)
{
gint row=0;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
GtkTreeIter iter;

treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(render_light_tv));
selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(render_light_tv));

if (gtk_tree_model_get_iter_first(treemodel, &iter))
  {
  do
    {
    if (gtk_tree_selection_iter_is_selected(selection, &iter))
      return(row);
    row++;
    }
  while (gtk_tree_model_iter_next(treemodel, &iter));
  }
return(-1);
}

/***********************************/
/* construct the light source list */
/***********************************/
void update_light_list(void)
{
gint row, num_rows=0;
gchar *text;
GSList *list;
GtkTreeIter iter;
GtkTreeModel *treemodel;
GtkTreeSelection *selection;
struct light_pak *light;

/* checks */
g_assert(render_light_ls != NULL);

/* store */
row = render_light_selected();

/* re-populate */
gtk_list_store_clear(render_light_ls);
for (list=sysenv.render.light_list ; list ; list=g_slist_next(list))
  {
  light = (struct light_pak *) list->data;
  gtk_list_store_append(render_light_ls, &iter);

/* position or direction vector */
  text = g_strdup_printf(" (%5.1f, %5.1f, %5.1f) ", light->x[0],
                                                    light->x[1],
                                                    light->x[2]);
  gtk_list_store_set(render_light_ls, &iter, RENDER_LIGHT_VECTOR, text, -1);
  g_free(text);

/* type */
  if (light->type == DIRECTIONAL)
    text = g_strdup(" Directional");
  else
    text = g_strdup(" Positional");
  gtk_list_store_set(render_light_ls, &iter, RENDER_LIGHT_TYPE, text, -1);
  g_free(text);

/* FIXME - easier way to specify the RGB colour? */
  text = get_hex_colour(light->colour);
  gtk_list_store_set(render_light_ls, &iter, RENDER_LIGHT_ATTRIBUTE, text, -1);
  g_free(text);

  num_rows++;
  }

/* restore selected row, or the previous (if possible) if deleted  */
if (row >= num_rows && row)
  row--;
if (row >= 0)
  {
  treemodel = gtk_tree_view_get_model(GTK_TREE_VIEW(render_light_tv));
  if (gtk_tree_model_iter_nth_child(treemodel, &iter, NULL, row))
    {
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(render_light_tv)); 
    if (selection)
      gtk_tree_selection_select_iter(selection, &iter);
    }
  }
}

/***********************************************************/
/* modify light position to match current rotational frame */
/***********************************************************/
void mod_light_pos(gdouble *x, struct model_pak *model)
{
gdouble mat[9];

g_assert(model != NULL);

/* apply current rotational reference */
vecmat(model->rotmat, x);
/* apply the inverse of the initial rotmat (ie get "pure" rotation) */
/* NB: initial rotmat is NOT the identity */
init_rotmat(mat);
invmat(mat);
vecmat(mat, x);
}

/*******************************************/
/* light list manipulation, add new source */
/*******************************************/
void add_current_light(GtkWidget *w, gpointer dummy)
{
struct light_pak *light;
struct model_pak *model;

/* duplicate data for the list */
light = g_malloc(sizeof(struct light_pak));
memcpy(light, &current_light, sizeof(struct light_pak));

/* NEW */
model = sysenv.active_model;
if (model)
  mod_light_pos(light->x, model);

/* append & update */
sysenv.render.light_list = g_slist_append(sysenv.render.light_list, light);
update_light_list();

redraw_canvas(SINGLE);
}

/******************************************/
/* light list manipulation, modify source */
/******************************************/
void mod_selected_light(GtkWidget *w, gpointer dummy)
{
gint row;
struct light_pak *light;
struct model_pak *model;

row = render_light_selected();
if (row < 0)
  return;

/* get the light's data from the list */
light = (struct light_pak *) g_slist_nth_data(sysenv.render.light_list, row);

/* overwrite with the current data */
memcpy(light, &current_light, sizeof(struct light_pak));

/* NEW */
model = sysenv.active_model;
if (model)
  mod_light_pos(light->x, model);

update_light_list();

redraw_canvas(SINGLE);
}

/******************************************/
/* light list manipulation, delete source */
/******************************************/
void del_selected_light(GtkWidget *w, gpointer dummy)
{
gint row;
struct light_pak *light;

row = render_light_selected();
if (row < 0)
  return;

light = (struct light_pak *) g_slist_nth_data(sysenv.render.light_list, row);

sysenv.render.light_list = g_slist_remove(sysenv.render.light_list, light);

update_light_list();

redraw_canvas(SINGLE);
}

/******************/
/* render cleanup */
/******************/
void render_cleanup(void)
{
gtk_list_store_clear(render_light_ls);
render_light_ls = NULL;
render_light_tv = NULL;
}

/*********************************************/
/* callback to fill dialog with light values */
/*********************************************/
void render_light_activate(GtkTreeView *treeview, GtkTreePath *treepath)
{
gint n;
gdouble mat[9];
struct model_pak *model;
struct light_pak *light;

/* get selected light */
n = render_light_selected();
if (n < 0)
  return;

/* get the light's data from the list */
light = g_slist_nth_data(sysenv.render.light_list, n);

/* transfer to current displayed values */
if (light)
  {
/* overwrite */
  memcpy(&current_light, light, sizeof(struct light_pak));

/* CURRENT - convert to current orientation */
  model = sysenv.active_model;
  if (model)
    {
/* inverse of mod_light_pos() */
    init_rotmat(mat);
    vecmat(mat, current_light.x);
    vecmat(model->irotmat, current_light.x);
    }

/* update */
  gtksh_relation_update(NULL);
  }
}

/****************************/
/* lighting control options */
/****************************/
void render_lighting_section(GtkWidget *box)
{
gint i;
gchar *titles[] = {" Colour ", "  Absolute vector  ", " Type "};
GList *list=NULL;
GtkWidget *swin, *vbox1, *vbox2, *vbox, *hbox, *frame;
GtkWidget *label, *combo, *button;
GtkStyle *style1;
GtkCellRenderer *renderer;
GtkTreeViewColumn *column;

/* NB: for the coloured boxes we need to copy a style from a realized widget! */

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* box 1 - colour (and edit button) */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
label = gtk_label_new("Current light source colour");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

/* light colour */
button = gtk_button_new_with_label("    ");
gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(button), "clicked",
                 GTK_SIGNAL_FUNC(modify_colour_dialog), current_light.colour);
style1 = gtk_style_copy(gtk_widget_get_style(window));
style1->bg[0].red   = current_light.colour[0] * 65535;
style1->bg[0].green = current_light.colour[1] * 65535; 
style1->bg[0].blue  = current_light.colour[2] * 65535; 
gtk_widget_set_style(GTK_WIDGET(button), style1);


/* box 2 - light components */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);

gtksh_direct_spin("Ambient component", &current_light.ambient,
                  0.0, 1.0, 0.1, NULL, NULL, vbox);

gtksh_direct_spin("Diffuse component", &current_light.diffuse,
                  0.0, 1.0, 0.1, NULL, NULL, vbox);

gtksh_direct_spin("Specular component", &current_light.specular,
                  0.0, 1.0, 0.1, NULL, NULL, vbox);


/* left box - light type */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* type label */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);
label = gtk_label_new("  Type  ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
/* TODO - pulldown dir/pos */
/* NEW - combo box for the selection mode */
list = g_list_append(list, "Directional");
list = g_list_append(list, "Positional");
combo = gtk_combo_new();
gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(combo)->entry), FALSE);
gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
gtk_box_pack_start(GTK_BOX(hbox), combo, FALSE, FALSE, 0);
g_signal_connect(GTK_OBJECT(GTK_COMBO(combo)->entry), "changed", 
                 GTK_SIGNAL_FUNC(event_render_modify), (gpointer) combo);
g_object_set_data(G_OBJECT(combo), "id", (gpointer) LIGHT_TYPE);


/* position/direction vector input */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

/* x component */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
label = gtk_label_new("X");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

gtksh_direct_spin(NULL, &current_light.x[0],
                  -1000.0, 1000.0, 0.1, NULL, NULL, vbox);

/* y component */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
label = gtk_label_new("Y");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

gtksh_direct_spin(NULL, &current_light.x[1],
                  -1000.0, 1000.0, 0.1, NULL, NULL, vbox);

/* z component */
vbox = gtk_vbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
label = gtk_label_new("Z");
gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

gtksh_direct_spin(NULL, &current_light.x[2],
                  -1000.0, 1000.0, 0.1, NULL, NULL, vbox);


/* next frame - actions */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);
hbox = gtk_hbox_new(TRUE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
gtksh_button(" Add ", add_current_light, NULL, hbox, TT);
gtksh_button(" Modify ", mod_selected_light, NULL, hbox, TT);
gtksh_button(" Delete ", del_selected_light, NULL, hbox, TT);

/* right box 1 - light source listing */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, TRUE, TRUE, 0);
vbox = gtk_vbox_new(FALSE,0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* scrolled win for the list */
swin = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(swin),
                               GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), swin, TRUE, TRUE, 0);
/*
gtk_widget_set_size_request(swin, 24*sysenv.gfontsize, -1);
*/

/* NEW - light list in a list store */
render_light_ls = gtk_list_store_new(RENDER_LIGHT_NCOLS, 
                                     G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
render_light_tv = gtk_tree_view_new_with_model(GTK_TREE_MODEL(render_light_ls));
gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(swin), render_light_tv);
for (i=0 ; i<=RENDER_LIGHT_TYPE ; i++)
  {
  renderer = gtk_cell_renderer_text_new();
  column = gtk_tree_view_column_new_with_attributes(titles[i], renderer, "text", i, NULL);

if (!i)
  gtk_tree_view_column_add_attribute(column, renderer, "background", RENDER_LIGHT_ATTRIBUTE);

  gtk_tree_view_append_column(GTK_TREE_VIEW(render_light_tv), column);
  }

g_signal_connect(render_light_tv, "row_activated",
                 G_CALLBACK(render_light_activate), NULL);
}

/***************************/
/* OpenGL specific options */
/***************************/
void render_opengl_section(GtkWidget *box, struct model_pak *model)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* perspective projection */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_check("Perspective projection", &sysenv.render.perspective,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Vanishing point",
                  &sysenv.render.vp_dist, 1.0, 100.0, 1.0,
                  render_refresh, NULL, vbox);

/* depth queuing */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_check("Depth queueing", &sysenv.render.fog,
                   render_refresh, NULL, vbox);

gtksh_direct_spin("Depth queueing start",
                  &sysenv.render.fog_start, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);

gtksh_direct_spin("Depth queueing strength",
                  &sysenv.render.fog_density, 0.0, 1.0, 0.1,
                  render_refresh, NULL, vbox);

/* new frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* sphere quality */
gtksh_direct_spin("Sphere quality",
                  &sysenv.render.sphere_quality, 0.0, 8.0, 1.0,
                  render_refresh, NULL, vbox);

/* cylinder quality */
gtksh_direct_spin("Cylinder quality",
                  &sysenv.render.cylinder_quality, 4.0, 20.0, 1.0,
                  render_refresh, NULL, vbox);

/* quality */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

gtksh_direct_check("Fast rotation", &sysenv.render.fast_rotation,
                   NULL, NULL, vbox);
gtksh_direct_check("Selection halos", &sysenv.render.halos,
                   render_refresh, NULL, vbox);

/* new frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);

hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, PANEL_SPACING);

gtksh_auto_check("Label geometric measurements", geom_label_toggle, NULL,
                                          &model->show_geom_labels, hbox);

gtksh_direct_spin("Line width",
                  &sysenv.render.geom_line_width, 0.5, 9.0, 0.5,
                  render_refresh, NULL, vbox);
}

/***************************/
/* POVRay specific options */
/***************************/
void render_povray_section(GtkWidget *box)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame;
GtkWidget *label, *entry;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* new frame */
frame = gtk_frame_new("Image size");
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

/* create a vbox in the frame */
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_direct_spin("Width",
                  &sysenv.render.width, 100, 2000, 100,
                  NULL, NULL, vbox);

gtksh_direct_spin("Height",
                  &sysenv.render.height, 100, 2000, 100,
                  NULL, NULL, vbox);

/* new frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(FALSE, 0);
gtk_container_add(GTK_CONTAINER(frame),vbox);

gtksh_direct_check("Shadowless", &sysenv.render.shadowless, NULL, NULL, vbox);

/* misc options */
gtksh_direct_check("Create povray input files then stop",
                   &sysenv.render.no_exec_povray,
                   NULL, NULL, vbox);
gtksh_direct_check("Delete intermediate input/image files",
                   &sysenv.render.no_keep_tempfiles,
                   NULL, NULL, vbox);


/* new frame */
frame = gtk_frame_new("Glass morphology");
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, 5);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(GTK_BOX(vbox)), PANEL_SPACING);

/* refractive index */
gtksh_direct_spin("Refractive index",
                  &sysenv.render.ref_index, 1.0, 3.0, 0.1,
                  NULL, NULL, vbox);

/* texture - glassy, metallic, etc. */
hbox = gtk_hbox_new (FALSE, 0);
gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
label = gtk_label_new("Surface finish ");
gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
entry = gtk_entry_new_with_max_length(LINELEN-1);
gtk_box_pack_end(GTK_BOX(hbox), entry, FALSE, TRUE, 0);

/* this causes GTK to complain (open , change ribbon colour, close, open */
/* but I cannot figure out why */
/*
gtk_entry_set_text(GTK_ENTRY(entry), sysenv.render.morph_finish);
*/

/* update hook */
g_signal_connect(GTK_OBJECT(entry), "activate",
                 GTK_SIGNAL_FUNC(event_render_modify), (gpointer) entry);
g_object_set_data(G_OBJECT(entry), "id", (gpointer) MORPH_FINISH);

/* run POVRay job in the background */
hbox = gtk_hbox_new(FALSE, 0);
gtk_box_pack_end(GTK_BOX(box), hbox, TRUE, FALSE, PANEL_SPACING);
gtksh_button("    Render    ", render_setup, NULL, hbox, TF);
}

/*****************/
/* misc. options */
/*****************/
void render_misc_section(GtkWidget *box)
{
GtkWidget *vbox1, *vbox2, *vbox, *hbox, *frame;
struct model_pak *model;

/* TODO - allow render dialog even when no models, but dont draw */
/* or make insensitive model specific buttons */
model = sysenv.active_model;

/* left & right pane split */
hbox = gtk_hbox_new(FALSE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(box), hbox);
gtk_container_set_border_width(GTK_CONTAINER(hbox), PANEL_SPACING);
vbox1 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox1, TRUE, TRUE, 0);
vbox2 = gtk_vbox_new(FALSE, PANEL_SPACING);
gtk_box_pack_start(GTK_BOX(hbox), vbox2, TRUE, TRUE, 0);

/* left pane */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox1), frame, FALSE, FALSE, 0);

/* create a vbox in the frame */
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* relation check buttons */
gtksh_auto_check("Show axes", render_refresh, NULL, &model->show_axes, vbox);
gtksh_auto_check("Show cell", render_refresh, NULL, &model->show_cell, vbox);
gtksh_auto_check("Show cell images", render_refresh, NULL, &model->show_cell_images, vbox);
gtksh_auto_check("Show cell lengths", render_refresh, NULL, &model->show_cell_lengths, vbox);
gtksh_auto_check("Show charge", render_refresh, NULL, &model->show_charge, vbox);
gtksh_auto_check("Show cores", render_refresh, NULL, &model->show_cores, vbox);
gtksh_auto_check("Show shells", render_refresh, NULL, &model->show_shells, vbox);
gtksh_auto_check("Show atom charges", render_refresh, NULL, &model->show_atom_charges, vbox);
gtksh_auto_check("Show atom labels", render_refresh, NULL, &model->show_atom_labels, vbox);
gtksh_auto_check("Show atom types", render_refresh, NULL, &model->show_atom_types, vbox);

/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

/* change font options */
/* GTK 2.x - text font is no longer changeable - Duncan Stupid */
gtksh_button_x("Change graphics font", gl_font_dialog, NULL, vbox);
gtksh_button_x("Toggle axes type", toggle_axes_type, NULL, vbox);

/* next frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(vbox2), frame, FALSE, FALSE, 0);
vbox = gtk_vbox_new(TRUE, PANEL_SPACING);
gtk_container_add(GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width(GTK_CONTAINER(vbox), PANEL_SPACING);

gtksh_auto_check("Colour using SOF ", cs_toggle, NULL, &model->sof_colourize, vbox);
gtksh_auto_check("Labelled morphology ", morph_toggle, NULL, &model->morph_label, vbox);
}

/***********************/
/* render setup widget */
/***********************/
void render_dialog()
{
gpointer dialog;
GtkWidget *window, *frame;
GtkWidget *label, *notebook, *page;
struct light_pak *ldata;
struct model_pak *model;

/* NB: some check boxes depend on having a valid model */
model = sysenv.active_model;
if (!model)
  return;

/* request dialog slot */
dialog = dialog_request(POVRAY, "Display properties", NULL, render_cleanup, model);
if (!dialog)
  return;
window = dialog_window(dialog);

/* init current light */
ldata = (struct light_pak *) ((GSList *) sysenv.render.light_list)->data;

if (ldata)
  memcpy(&current_light, ldata, sizeof(struct light_pak));
else
  {
  show_text(ERROR, "Empty light list.\n");
  return;
  }

/* notebook frame */
frame = gtk_frame_new(NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), frame, FALSE, FALSE, 0);
gtk_container_set_border_width(GTK_CONTAINER(frame), PANEL_SPACING);

/* create notebook */
notebook = gtk_notebook_new();
gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
gtk_container_add(GTK_CONTAINER(frame), notebook);
gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);

/* main page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Main");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_main_page(page, model);

/* colour page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Colours");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_colours_section(page);

/* lighting page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Lights");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_lighting_section(page);

/* OpenGL page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("OpenGL");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_opengl_section(page, model);

/* POVRay page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("POVRay");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_povray_section(page);

/* misc page */
page = gtk_vbox_new(FALSE, PANEL_SPACING);
label = gtk_label_new("Misc");
gtk_notebook_append_page(GTK_NOTEBOOK(notebook), page, label);
render_misc_section(page);

/* terminating button */
gtksh_stock_button(GTK_STOCK_CLOSE, dialog_destroy, dialog,
                   GTK_DIALOG(window)->action_area);

/* display the dialog */
gtk_widget_show_all(window);

update_light_list();
}
