/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "config.h"
#ifdef ZVT
#   include <zvt/zvtterm.h>
#endif

#include "world.h"
#include "notebook.h"
#include "dialog.h"
#include "settings.h"
#include "toolbar.h"
#include "menu.h"
#include "editor.h"
#include "run.h"

GtkWidget *notebook  = NULL;
GList     *notebook_tabs_list = NULL;
world     *current_world = NULL;

typedef struct _TabKeeper TabKeeper;
struct _TabKeeper {
    int   type;
    void *data;
    void (*close) (void *data);
    void (*focused) (void *data);
};

void gm_notebook_set_tab_position(int position) {
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), position);
}
void gm_notebook_set_homogeneous_tabs(int on) {
    gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(notebook), on);
}

void gm_notebook_set_tabs_show_lines_waiting(int on) {
    GList *list, *l;
    list = gm_notebook_get_worlds();
    for(l = list; l; l = g_list_next(l))
	gm_world_update_tab_label((world *) (l->data));
    g_list_free(list);
}


GList *gm_notebook_get_worlds() {
    GList *list = NULL, *l;
    for(l = notebook_tabs_list; l; l = g_list_next(l)) {
	if(((TabKeeper *) l->data)->type == TK_WORLD) {
	    list = g_list_append(list, ((TabKeeper *) l->data)->data);
	}
    }
    return list;
}

world *get_current_world() {
    TabKeeper *t;
    int page;
    world *w = NULL;
    GList *node;

    page = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
    if(page >= 0) {
        node = g_list_nth(notebook_tabs_list, page);
        t = (TabKeeper *)(node->data);
        if(t && t->type == TK_WORLD) {
            w = t->data;
        }
    }

    return w;
}

void notebook_page_switched(GtkNotebook *n, GtkNotebookPage *page, guint num) {
    GList *node = g_list_nth(notebook_tabs_list, num);
    TabKeeper *t = (TabKeeper *) node->data;

    current_world = get_current_world();

    gm_toolbar_switched_to(t->type);
    gm_menu_switched_to(t->type);
    if(t->type == TK_WORLD) {
        gm_world_got_focus((world *) t->data);
    } else if(t->type == TK_EDITOR) {
        gm_editor_got_focus((editor_t *) t->data);
#ifdef ZVT
    } else if(t->type == TK_RUN) {
        gm_run_got_focus((t_run_t *) t->data);
#endif
    }
}

void gm_notebook_force_world_focused(world *w) {
    int pos;
    if(w && w != current_world) {
	pos = gm_notebook_get_pos(w);
	if(pos < 0)
	    return;
	gm_notebook_set(pos);
#if 0
        gm_toolbar_switched_to(TK_WORLD);
        gm_menu_switched_to(TK_WORLD);
        gm_world_got_focus(w);
#endif
    }
}

void gm_notebook_changed() {
    GList *l;
    TabKeeper *t;
    int i = 0;

    if(debug) printf("Notebook changed, updating positions\n");
    current_world = get_current_world();
    for(l = notebook_tabs_list; l; l = g_list_next(l)) {
        t = (TabKeeper *) (l->data);
        switch(t->type) {
        case TK_WORLD:
            gm_world_set_position((world *) (t->data), i);
            break;
        case TK_EDITOR:
            gm_editor_set_position((editor_t *) (t->data), i);
            break;
#ifdef ZVT
        case TK_RUN:
            gm_run_set_position((t_run_t *) (t->data), i);
            break;
#endif
        }
        i++;
    }
}

GtkWidget *gm_notebook_create() {
    notebook = gtk_notebook_new();
    gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);
    gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);
    gtk_notebook_set_tab_border(GTK_NOTEBOOK(notebook), 1);

    gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(notebook), 
				      settings->homo_tabs);
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), settings->tab_position);

    gtk_widget_show(notebook);
    gtk_signal_connect_after(GTK_OBJECT(notebook), "switch_page",
			     GTK_SIGNAL_FUNC(notebook_page_switched), NULL);
    gtk_signal_connect_after(GTK_OBJECT(notebook), "add",
			     GTK_SIGNAL_FUNC(gm_notebook_changed), NULL);
    gtk_signal_connect_after(GTK_OBJECT(notebook), "remove",
			     GTK_SIGNAL_FUNC(gm_notebook_changed), NULL);

    return notebook;
}

void gm_notebook_next() {
    int pages = g_list_length(notebook_tabs_list);
    int page  = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
    if(pages) {
	gm_notebook_set((page + 1) % pages);
    }
}

void gm_notebook_prev() {
    int pages = g_list_length(notebook_tabs_list);
    int page  = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
    if(pages) {
	gm_notebook_set((page - 1) % pages);
    }
}

void gm_notebook_set(int page) {
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook), page);
}


void gm_notebook_add_type(int pos, int type, void *data,
                          GtkWidget *main, GtkWidget *tab) {
    TabKeeper *tk = g_malloc(sizeof(TabKeeper));

    tk->type = type;
    tk->data = data;

    if(pos < 0) {
        notebook_tabs_list = g_list_append(notebook_tabs_list, (void *) tk);
        gtk_notebook_append_page(GTK_NOTEBOOK(notebook), main, tab);
        pos = g_list_length(notebook_tabs_list) - 1;
    } else {
        notebook_tabs_list = g_list_insert(notebook_tabs_list, (void *) tk,
                                           pos);
        gtk_notebook_insert_page(GTK_NOTEBOOK(notebook), main, tab, pos);
    }
    gtk_notebook_set_page(GTK_NOTEBOOK(notebook), pos);
    gm_notebook_changed();
}

void gm_notebook_append_type(int type, void *data, GtkWidget *main,
                             GtkWidget *tab) {
    gm_notebook_add_type(g_list_length(notebook_tabs_list),
                         type, data, main, tab);
}

void gm_notebook_add_world(world *w) {
    gm_notebook_add_type(-1, TK_WORLD, (void *) w, w->vpaned, w->tab_label);
#if 0
    gm_world_println(w, "\e[32mGROEN\e[1mBOLD\e[0;42m acher\e[4;1;35monder!\e[0m");
#endif
    gm_world_connect(w);
}

void notebook_remove(int page_num) {
    GList *node;
    gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page_num);

    node = g_list_nth(notebook_tabs_list, page_num);
    g_free(node->data);
    notebook_tabs_list = g_list_remove_link(notebook_tabs_list, node);

    if(!notebook_tabs_list) { /* last tab has been removed */
        gm_toolbar_switched_to(TK_NONE);
        gm_menu_switched_to   (TK_NONE);
    }
    gm_notebook_changed();
}

int gm_notebook_try_add_world(const char *name) {
    world *w = gm_world_create(name);

    if(w == NULL) {
        gm_dialog_popup(_("Error"), _("Failed to open world"), B_OK, FALSE);
        return FALSE;
    }
    gm_notebook_add_world(w);
    return TRUE;
}

void gm_notebook_close_world(world *w) {
    GList *l;
    TabKeeper *t;
    int page_num = 0;

    for(l = notebook_tabs_list; l != NULL; l = g_list_next(l)) {
        t = l->data;
        if(t->type == TK_WORLD && t->data == w) {
            notebook_remove(page_num);
            gm_world_close(w);
            return;
        }
        page_num++;
    }
    if(debug) printf("MAJOR ERROR: World not found to close !!!\n");
}

#ifdef ZVT
void gm_notebook_close_run(t_run_t *r) {
    GList *l;
    TabKeeper *t;
    int page_num = 0;

    for(l = notebook_tabs_list; l != NULL; l = g_list_next(l)) {
        t = l->data;
        if(t->type == TK_RUN && t->data == r) {
            notebook_remove(page_num);
            gm_run_close(r);
            return;
        }
        page_num++;
    }
    if(debug) printf("MAJOR ERROR: Run not found to close !!!\n");
}
#endif

void gm_notebook_close_worlds_with_name(const char *name) {
    GList *l;
    TabKeeper *t;
    int page_num;

  repeat_this:
    page_num = 0;
    for(l = notebook_tabs_list; l != NULL; l = g_list_next(l)) {
        t = l->data;
        if(t->type == TK_WORLD &&
           strcmp(((world *) t->data)->p->name, name) == 0) {
            notebook_remove(page_num);
            gm_world_close((world *) t->data);
            goto repeat_this;
        }
        page_num++;
    }
}

void gm_notebook_try_close_world(world *w) {
    if(gm_world_ok_to_close(w)) {
        gm_notebook_close_world(w);
    }
}


void gm_notebook_close_current_world() {
    world *w;
    w = gm_notebook_current_world();
    if(w) {
        gm_notebook_try_close_world(w);
    } else if(debug) {
        printf("Not a world\n");
    }
}

world *gm_notebook_current_world() {
    return current_world;
}

world *gm_notebook_find_world (const char *name) {
    GList *l;
    TabKeeper *t;
    for(l = notebook_tabs_list; l != NULL; l = g_list_next(l)) {
	t = l->data;
	if(t->type == TK_WORLD && strcmp(((world *) t->data)->p->name,
					 name) == 0)
	    return (world *) t->data;
    }
    return NULL;
}

world *gm_notebook_get_world_by_id(int id) {
    GList *l;
    TabKeeper *t;
    for(l = notebook_tabs_list; l != NULL; l = g_list_next(l)) {
        t = l->data;
        if(t->type == TK_WORLD && ((world *) t->data)->id == id)
            return (world *) t->data;
    }
    return NULL;
}

void gm_notebook_remove_editor(editor_t *e) {
    GList *l;
    TabKeeper *t;
    int page_num = 0;

    if(debug) printf("removeing editor \"%s\" from notebook\n", e->name);
    for(l = notebook_tabs_list; l != NULL; l = g_list_next(l)) {
        t = l->data;
        if(t->type == TK_EDITOR && t->data == e) {
            notebook_remove(page_num);
            return;
        }
        page_num++;
    }
    if(debug) printf("MAJOR ERROR: EDITOR not found to close !!!\n");
}

int gm_notebook_ok_to_close() {
    GList *l;
    TabKeeper *t;

    for(l = notebook_tabs_list; l != NULL; l = g_list_next(l)) {
        t = (TabKeeper *) l->data;
        switch(t->type) {
        case TK_WORLD:
            if(!gm_world_ok_to_close((world *) t->data)) {
                return FALSE;
            }
            break;
        case TK_EDITOR:
            break;
#ifdef ZVT
        case TK_RUN:
            break;
#endif
        default:
            if(debug) printf("MAJOR ERROR: Unknown tab to close !!\n");
        }
    }
    return TRUE;
}

void gm_notebook_close() {
    GList *item;
    TabKeeper *t;

    gm_notebook_save_status();
    item = notebook_tabs_list;
    while(item) {
        t = item->data;
        switch(t->type) {
        case TK_WORLD:
            gm_notebook_close_world((world *) t->data);
            break;
        case TK_EDITOR:
            break;
#ifdef ZVT
        case TK_RUN:
            gm_notebook_close_run((t_run_t *) (t->data));
            break;
#endif
        default:
            if(debug) printf("MAJOR ERROR: Unknown tab to close !!\n");
        }
        item = notebook_tabs_list;
    }
}

int gm_notebook_get_pos(void *page) {
    int ret = 0;
    GList *l;
    for(l = notebook_tabs_list; l; l = g_list_next(l)) {
	if(((TabKeeper *) l->data)->data == page)
	    return ret;
	ret++;
    }
    return -1;
}

int gm_notebook_is_world(const void *data) {
    GList *l;
    TabKeeper *tk;
    for(l = notebook_tabs_list; l; l = g_list_next(l)) {
	tk = (TabKeeper *) l->data;
	if(tk->data == data && tk->type == TK_WORLD) {
	    return TRUE;
	}
    }
    return FALSE;
}

void gm_notebook_save_status() {
    char *filename;
    FILE *file;
    GList *l;
    TabKeeper *tk;

    filename = g_strconcat(gm_settings_get_config_dir(), "/",
			   NOTEBOOK_STATUS_FILE, NULL);
    if(settings->restore_open_worlds && notebook_tabs_list) {
	if((file = fopen(filename, "w"))) {
	    for(l = notebook_tabs_list; l; l = g_list_next(l)) {
		tk = l->data;
		switch(tk->type) {
		case TK_WORLD:
		    if(!((world *) (tk->data))->quick) {
			fputs("W\n", file);
			fputs(((world *) (tk->data))->p->name, file);
			fputc('\n', file);
		    }
		    break;
#ifdef ZVT
		case TK_RUN:
		    fputs("R\n", file);
		    fputs(((t_run_t *) (tk->data))->name, file);
		    fputc('\n', file);
		    fputs(((t_run_t *) (tk->data))->command, file);
		    fputc('\n', file);
		    break;
#endif
		}
	    }
	    fclose(file);
	}
    } else {
	remove(filename);
    }
    g_free(filename);
}

void gm_notebook_try_restore_status() {
    char *filename;
    FILE *file;
#define LENGTH 255
    char buf[LENGTH + 1];
    int len;
#ifdef ZVT
    char *name, *command;
#endif
#define chop() len = strlen(buf); if(len) buf[len - 1] = '\0'

    filename = g_strconcat(gm_settings_get_config_dir(), "/",
                           NOTEBOOK_STATUS_FILE, NULL);
    if(settings->restore_open_worlds) {
        if((file = fopen(filename, "r"))) {
            while(fgets(buf, LENGTH, file)) {
                if(buf[0] == 'W') {
                    fgets(buf, LENGTH, file);
                    chop();
                    if(debug) printf("Restoring world \"%s\"\n", buf);
                    gm_notebook_try_add_world(buf);
#ifdef ZVT
                } else if(buf[0] == 'R') {
                    fgets(buf, LENGTH, file);
                    chop();
                    name = g_strdup(buf);
                    command = fgets(buf, LENGTH, file);
                    chop();
                    if(debug) printf("Restoring run \"%s\" => \"%s\"\n",
                                     name, command);
                    gm_run_try_run(name, command);
                    g_free(name);
#endif
                }
            }
            fclose(file);
            gm_notebook_set(0);
        }
    } else {
        remove(filename);
    }
    g_free(filename);
}


void gm_notebook_update_input_styles() {
    GList *widgets = NULL;
    GList *l;
    TabKeeper *tk;
#ifdef ZVT
    gushort reds[N_COLORS];
    gushort greens[N_COLORS];
    gushort blues[N_COLORS];
    int i;
#endif

    for(l = notebook_tabs_list; l; l = g_list_next(l)) {
        tk = l->data;
        if(tk->type == TK_WORLD) {
            widgets = g_list_append(widgets, ((world *) (tk->data))->input);
        } else if(tk->type == TK_EDITOR && IS_I_EDITOR(tk->data)) {
            widgets = g_list_append(widgets, I_EDITOR(tk->data)->text);
#ifdef ZVT
        } else if(tk->type == TK_EDITOR && IS_E_TAB_EDITOR(tk->data)) {
            widgets = g_list_append(widgets, E_TAB_EDITOR(tk->data)->term);
        } else if(tk->type == TK_RUN) {
            widgets = g_list_append(widgets, ((t_run_t *) (tk->data))->term);
#endif
        }
    }
    if(widgets) {
#ifdef ZVT
        for(i = 0; i < N_COLORS; i++) {
            reds[i]   = colors[i].red;
            greens[i] = colors[i].green;
            blues[i]  = colors[i].blue;
        }
#endif
        for(l = widgets; l; l = g_list_next(l)) {
#ifdef ZVT
            if(ZVT_IS_TERM(l->data)) {
                zvt_term_set_color_scheme(ZVT_TERM(l->data), reds, greens, blues);
                gtk_widget_queue_draw(GTK_WIDGET(l->data));
            } else {
#endif
                gtk_widget_set_style(GTK_WIDGET(l->data), input_style);
#ifdef ZVT
            }
#endif
        }
        g_list_free(widgets);
    }
}

void gm_notebook_close_current_tab() {
    GList *l;
    TabKeeper *t;
    int cur = gtk_notebook_get_current_page(GTK_NOTEBOOK(notebook));
    if((l = g_list_nth(notebook_tabs_list, cur))) {
        t = l->data;
        switch(t->type) {
        case TK_WORLD:
            gm_notebook_close_world(t->data);
            return;
#ifdef ZVT
        case TK_RUN:
            gm_notebook_close_run(t->data);
            return;
#endif
        }
    }
}

