/******************************************************************************\
 gnofin/ui-record-select.c   $Revision: 1.5 $
 Copyright (C) 2000 Luc Saillard <luc.saillard@alcove.fr>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

// #define ENABLE_DEBUG_TRACE

#include "common.h"
#include <gtk/gtktable.h>
#include <gtk/gtkbox.h>
#include <gtk/gtkarrow.h>
#include <gtk/gtklist.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktooltips.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkradiobutton.h>
#include <gnome.h>
#include "gtk-auto-combo.h"
#include "data-if.h"
#include "date.h"
#include "ui.h"
#include "dialogs.h"
#include "data-structure/data-if.h"
#include "data-structure/account.h"
#include "data-structure/record-select.h"
#include "ui-record-list.h"
#include "ui-record-select.h"
#include "record-stringizer.h"
#include "record-parser.h"
#include "row-color.h"
#include <time.h>

typedef struct _MyGtkCalendar MyGtkCalendar;
struct _MyGtkCalendar
{
  GtkEntry  *date;
  GtkWidget *button;
  GtkWidget *calendar;
  GtkWidget *popup;
  int block_calendar_update;
  int block_calendar_date_changed;
  int calendar_month_changed_id;
  int calendar_day_changed_id;
};

typedef struct _UI_RecordSelect UI_RecordSelect;
struct _UI_RecordSelect
{
  MyGtkCalendar       date_begin;
  MyGtkCalendar       date_end;
  MyGtkCalendar       date_between_begin;
  MyGtkCalendar       date_between_end;
  int                 period_number, period_type;
  GtkRadioButton      *before,*after,*between,*last,*notchecked,*all;
  GtkWidget           *dialog;
  Account             *account;
};

/* Warning, if you change order, or add another period, you must modify
 * all code who access to period_number and period_type , for now
 * it's in the fonction on_radio_button_changed_last and in the file
 * data-structures/record-select.c */
const char *periods_name[]={N_("Day"),N_("Week"),N_("Month"),N_("Year"),NULL};
#define MAX_NUMBER_OF_PERIOD 20

/******************************************************************************
 * Static functions
 */

static void create_calendar_dialog (MyGtkCalendar *cal);

/******************************************************************************
 * Signal handlers
 */

static void
on_delete_popup (GtkCalendar *calendar, GdkEvent *event, MyGtkCalendar *cal)
{
  gtk_widget_hide (cal->popup);
  gtk_grab_remove (cal->popup);
}

static gint
on_key_press_popup (GtkWidget *widget, GdkEventKey *event, MyGtkCalendar *cal)
{
  if (event->keyval != GDK_Escape)
    return FALSE;
  gtk_widget_hide (cal->popup);
  gtk_grab_remove (cal->popup);
  return TRUE;
}

static gint
on_button_press_popup (GtkWidget *widget, GdkEventButton *event, MyGtkCalendar *cal)
{
  GtkWidget *child;

  child = gtk_get_event_widget ((GdkEvent *) event);
  if (child != widget)
   {
     while (child)
      {
	if (child == widget)
	  return FALSE;
	child = child->parent;
      }
   }
  gtk_widget_hide (cal->popup);
  gtk_grab_remove (cal->popup);
  return TRUE;
}

static void
on_close_dialog (GtkWidget *widget, GdkEvent *event, gboolean apply)
{
  /* Perhaps use another listener, or create a specific listeners
   * to redraw the list of record */
  if (apply)
    notification_list_notify (&row_color_change_listeners);
  gtk_widget_hide(widget);
}

static void
on_entry_date_changed (GtkWidget *w, MyGtkCalendar *cal)
{
  GtkRadioButton *button;
  g_return_if_fail (cal);

  if (cal->calendar && GTK_WIDGET_VISIBLE (cal->calendar)
      && (!cal->block_calendar_update))
   {
     GDate date;

     g_date_clear (&date, 1);
     g_date_set_parse (&date, gtk_entry_get_text ((GtkEntry *)cal->date));

     if (g_date_valid (&date))
      {
	/* Synchronize date with calendar widget */
	guint day, month, year;

	day = g_date_day (&date);
	month = g_date_month (&date) - 1;
	year = g_date_year (&date);

	gtk_calendar_freeze ((GtkCalendar *)cal->calendar);
	gtk_calendar_select_month ((GtkCalendar *)cal->calendar, month, year);
	gtk_calendar_select_day ((GtkCalendar *)cal->calendar, day);
	gtk_calendar_thaw ((GtkCalendar *)cal->calendar);

	/* Mark the today date in bold */
	date_now (&date);
	if (   (g_date_month (&date) - 1 == month)
	    && (g_date_year (&date) == year))
	  gtk_calendar_mark_day((GtkCalendar *)cal->calendar,day);
	else
	  gtk_calendar_clear_marks((GtkCalendar *)cal->calendar);
      } /* if (g_date_valid (&date)) */
   } /* if calendar is open and visible */

  /* The entry is modified so notify the radio button witch change value into the
   * account struct */
  if (w)
   {
     button = gtk_object_get_user_data((GtkObject *)w);
     if (button)
      {
	gtk_toggle_button_set_active((GtkToggleButton *)button,TRUE);
	gtk_signal_emit_by_name((GtkObject *)button,"clicked");
      }
   }
}

static void
on_date_format_changed (MyGtkCalendar *cal)
{
  GDate date;
  gchar *text;

  trace ("");

  text = gtk_entry_get_text (cal->date);
  if (date_parse (text, &date))
   {
     text = date_stringize (NULL, 0, &date);
     gtk_entry_set_text (cal->date, text);
     g_free (text);
   }
}

static void
on_entry_date_key_pressed (GtkWidget *w, GdkEventKey *event, MyGtkCalendar *cal)
{
  GDate date;
  gchar *text;
  trace ("");

  switch (event->keyval)
   {
    case GDK_KP_Add:
      event->keyval='+';
      break;
    case GDK_KP_Multiply:
      event->keyval=']';
      break;
    case GDK_KP_Subtract:
      event->keyval='-';
      break;
    case GDK_KP_Divide:
      event->keyval='[';
      break;
   }

  switch (event->keyval)
   {
    case '+':
    case '-':
    case '[':
    case ']':
      g_date_clear (&date, 1);
      g_date_set_parse (&date, gtk_entry_get_text (cal->date));
      if (g_date_valid (&date))
       {
	 if (event->keyval=='+')
	   g_date_add_days(&date,1);
	 else if (event->keyval=='-')
	   g_date_subtract_days(&date,1);
	 else if (event->keyval==']')
	   g_date_add_months(&date,1);
	 else if (event->keyval=='[')
	   g_date_subtract_months(&date,1);

	 text = date_stringize (NULL, 0, &date);
	 cal->block_calendar_update = 1;
	 gtk_entry_set_text (cal->date, text);
	 cal->block_calendar_update = 0;
	 g_free (text);
       }
      gtk_signal_emit_stop_by_name(GTK_OBJECT(w),"key-press-event");
      break;
   }
}

static void
on_calendar_date_changed (GtkCalendar *calendar, MyGtkCalendar *cal)
{
  GDate date;
  gchar *text;
  guint year, month, day;
  GtkRadioButton *button;

  g_return_if_fail (calendar);
  g_return_if_fail (cal);

  gtk_calendar_get_date (calendar, &year, &month, &day);
  month++;

  /* Generate date string */
  g_date_clear (&date, 1);
  g_date_set_dmy (&date, day, month, year);

  text = date_stringize (NULL, 0, &date);

  cal->block_calendar_update = 1;
  gtk_entry_set_text (cal->date, text);
  cal->block_calendar_update = 0;

  /* Mark the today date in bold */
  date_now (&date);
  if (   (g_date_month (&date) == month)
      && (g_date_year (&date) == year))
    gtk_calendar_mark_day((GtkCalendar *)cal->calendar,g_date_day(&date));
  else
    gtk_calendar_clear_marks((GtkCalendar *)cal->calendar);

  /* Activate the radio button, and notify to change value in the struct account */
  button = gtk_object_get_user_data((GtkObject *)cal->date);
  if (button)
   {
     gtk_toggle_button_set_active((GtkToggleButton *)button,TRUE);
     gtk_signal_emit_by_name((GtkObject *)button,"clicked");
   }
  g_free (text);
}

static void
on_show_calendar (GtkWidget *w, MyGtkCalendar *cal)
{
  int x,y,bwidth,bheight;
  g_return_if_fail (cal);

  if (cal->calendar == NULL)
    create_calendar_dialog (cal);

  if (GTK_WIDGET_VISIBLE (cal->popup))
   {
     gtk_widget_hide (cal->popup);
     gtk_grab_remove (cal->popup);
   }
  else
   {
     if (strlen (gtk_entry_get_text (cal->date)) > 0)
      {
	/* Force the calendar to be updated before it is displayed */
	on_entry_date_changed (NULL, cal);
      }
     else
      {
	/* Set the date entry to the value of the calendar, and
	 * make the calendar show today's date. */
	guint day, month, year;
	GDate date;

	date_now (&date);

	day = g_date_day (&date);
	month = g_date_month (&date) - 1;
	year = g_date_year (&date);

	gtk_calendar_freeze ((GtkCalendar *)cal->calendar);
	gtk_calendar_select_month ((GtkCalendar *)cal->calendar, month, year);
	gtk_calendar_select_day ((GtkCalendar *)cal->calendar, day);
	gtk_calendar_thaw ((GtkCalendar *)cal->calendar);

	on_calendar_date_changed ((GtkCalendar *)cal->calendar, cal);
      }

     gtk_widget_size_request (cal->popup, &cal->popup->requisition);

     gdk_window_get_origin (cal->button->window, &x, &y);
     gdk_window_get_size (cal->button->window, &bwidth, &bheight);

     x += bwidth - cal->popup->requisition.width;
     y += bheight;

     if (x < 0)
       x = 0;

     if (y < 0)
       y = 0;

     gtk_widget_set_uposition (cal->popup, x, y);

     gtk_widget_show (cal->popup);
     gtk_widget_grab_focus (cal->popup);
     gtk_grab_add(cal->popup);
   }
}

static void
on_hide_calendar (GtkCalendar *calendar, MyGtkCalendar *cal)
{
  gtk_widget_hide (cal->popup);
  gtk_grab_remove (cal->popup);
}

static void
on_radio_button_changed_before (GtkRadioButton *radiobutton, UI_RecordSelect *info)
{
  GDate date;

  if (gtk_toggle_button_get_active((GtkToggleButton *)radiobutton))
   {
     g_date_clear (&date, 1);
     g_date_set_parse (&date, gtk_entry_get_text (info->date_begin.date));
     if (g_date_valid(&date))
      {
	info->account->records_select.method=(RecordSelectType)BEFORE;
	memcpy(&info->account->records_select.date1,&date,sizeof(GDate));
      }
   }
}

static void
on_radio_button_changed_after (GtkRadioButton *radiobutton, UI_RecordSelect *info)
{
  GDate date;

  if (gtk_toggle_button_get_active((GtkToggleButton *)radiobutton))
   {
     g_date_clear (&date, 1);
     g_date_set_parse (&date, gtk_entry_get_text (info->date_end.date));
     if (g_date_valid(&date))
      {
	info->account->records_select.method=(RecordSelectType)AFTER;
	memcpy(&info->account->records_select.date1,&date,sizeof(GDate));
      }
   }
}

static void
on_radio_button_changed_between (GtkRadioButton *radiobutton, UI_RecordSelect *info)
{
  GDate date[2];

  if (gtk_toggle_button_get_active((GtkToggleButton *)radiobutton))
   {
     g_date_clear (date, 2);
     g_date_set_parse (&date[0], gtk_entry_get_text (info->date_between_begin.date));
     g_date_set_parse (&date[1], gtk_entry_get_text (info->date_between_end.date));
     if (g_date_valid(&date[0]) && g_date_valid(&date[1]))
      {
	if (g_date_compare(&date[0],&date[1])>0)
	 {
	   dialog_error (GTK_WINDOW (gtk_widget_get_toplevel (info->dialog)),
	       _("The begin date must be lower than the end date !!!"));
	 }
	else
	 {
	   info->account->records_select.method=(RecordSelectType)BETWEEN;
	   memcpy(&info->account->records_select.date1,&date[0],sizeof(GDate));
	   memcpy(&info->account->records_select.date2,&date[1],sizeof(GDate));
	 }
      }
   }
}

static void
on_radio_button_changed_last (GtkRadioButton *radiobutton, UI_RecordSelect *info)
{
  GDate date;

  trace("");
  if (gtk_toggle_button_get_active((GtkToggleButton *)radiobutton))
   {
     if (info->period_number<1 || info->period_number>MAX_NUMBER_OF_PERIOD)
       return;

     /* 
      * To speed up the test, we save the number of day AND the date.
      * The date is use to compare 2 date, and the number of date is save
      * in the user preference.
      */ 
     date_now (&date);

     switch (info->period_type)
      {
       case 0: /* Days */
	 g_date_subtract_days(&date,info->period_number);
	 break;
       case 1: /* Week */
	 g_date_subtract_days(&date,info->period_number*7);
	 break;
       case 2: /* Month */
	 g_date_subtract_months(&date,info->period_number);
	 break;
       case 3: /* Year */
	 g_date_subtract_years(&date,info->period_number);
	 break;
      }

     if (g_date_valid(&date))
      {
	info->account->records_select.method=(RecordSelectType)LAST;
	memcpy(&info->account->records_select.date1,&date,sizeof(GDate));
	info->account->records_select.period_type=info->period_type;
	info->account->records_select.period_number=info->period_number;
      }
   }
}

static void
on_menuitem_changed_num(GtkWidget *item, UI_RecordSelect *info)
{
  info->period_number=(int)gtk_object_get_user_data(GTK_OBJECT(item));
  gtk_toggle_button_set_active((GtkToggleButton *)info->last,TRUE);
}


static void
on_menuitem_changed_period(GtkWidget *item, UI_RecordSelect *info)
{
  info->period_type=(int)gtk_object_get_user_data(GTK_OBJECT(item));
  gtk_toggle_button_set_active((GtkToggleButton *)info->last,TRUE);
}


static void
on_radio_button_changed_not_checked (GtkRadioButton *radiobutton, UI_RecordSelect *info)
{
  if (gtk_toggle_button_get_active((GtkToggleButton *)radiobutton))
    info->account->records_select.method=(RecordSelectType)NOTCHECKED;
}

static void
on_radio_button_changed_all (GtkRadioButton *radiobutton, UI_RecordSelect *info)
{
  if (gtk_toggle_button_get_active((GtkToggleButton *)radiobutton))
    info->account->records_select.method=(RecordSelectType)ALL;
}


/******************************************************************************
 * Interface functions
 */

/******************************************************************************
 * Calendar dialog (not modal)
 */

static void
create_calendar_dialog (MyGtkCalendar *cal)
{
  GtkWidget *popup;
  GtkWidget *frame;

  popup = gtk_window_new (GTK_WINDOW_POPUP);
  cal->popup = GTK_WIDGET (popup);
  gtk_widget_set_events (popup,
      gtk_widget_get_events (popup) | GDK_KEY_PRESS_MASK);
  gtk_signal_connect (GTK_OBJECT (popup), "delete_event",
      GTK_SIGNAL_FUNC(on_delete_popup), popup);
  gtk_signal_connect (GTK_OBJECT (popup), "key_press_event",
      GTK_SIGNAL_FUNC(on_key_press_popup), cal);
  gtk_signal_connect (GTK_OBJECT (popup), "button_press_event",
      GTK_SIGNAL_FUNC(on_button_press_popup), cal);
  gtk_window_set_policy (GTK_WINDOW (popup), FALSE, FALSE, TRUE);


  frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
  gtk_container_add (GTK_CONTAINER (popup), frame);
  gtk_widget_show (frame);

  cal->calendar = gtk_calendar_new ();
  gtk_calendar_display_options (GTK_CALENDAR (cal->calendar),
      (GTK_CALENDAR_SHOW_DAY_NAMES | GTK_CALENDAR_SHOW_HEADING));
  gtk_container_add (GTK_CONTAINER (frame), cal->calendar);
  gtk_signal_connect (GTK_OBJECT (cal->calendar), "month_changed",
      on_calendar_date_changed, cal);
  gtk_signal_connect (GTK_OBJECT (cal->calendar), "day_selected",
      on_calendar_date_changed, cal);
  gtk_signal_connect (GTK_OBJECT (cal->calendar), "day_selected_double_click",
      on_hide_calendar, cal);
  gtk_widget_show (cal->calendar);
}

static GtkWidget *gnofin_calendar_new(MyGtkCalendar *cal)
{
  GtkWidget *entry, *hbox, *button, *arrow;

  cal->calendar=NULL;
  cal->popup=NULL;
  cal->block_calendar_update=0;

  notification_list_add (&date_format_change_listeners,
      NOTIFICATION_FUNC (on_date_format_changed), cal);

  /* Our entry to display date at own format */
  hbox = gtk_hbox_new (FALSE, 0);
  entry = gtk_entry_new ();
  gtk_widget_show(entry);
  gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
  cal->date = GTK_ENTRY (entry);

   {
     GDate date;
     gchar *text;

     date_now (&date);
     text = date_stringize (NULL, 0, &date);
     gtk_entry_set_text (cal->date, text);
     g_free (text);
   }

  gtk_signal_connect (GTK_OBJECT (entry), "activate",
      GTK_SIGNAL_FUNC (on_entry_date_changed), cal);
  gtk_signal_connect (GTK_OBJECT (entry), "key-press-event",
      GTK_SIGNAL_FUNC (on_entry_date_key_pressed), cal);

  /* Calendar popup button */
  button = gtk_button_new ();
  GTK_WIDGET_UNSET_FLAGS (button, GTK_CAN_FOCUS);
  gtk_box_pack_end (GTK_BOX (hbox), button, FALSE, FALSE, 0);
  gtk_widget_show(button);
  cal->button = button;

  arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
  gtk_container_add (GTK_CONTAINER (button), arrow);
  gtk_widget_show(arrow);

  gtk_signal_connect (GTK_OBJECT (button), "clicked",
      GTK_SIGNAL_FUNC (on_show_calendar), cal);

  return hbox;
}



/*
 * Function: make_select_dialog
 * Input:
 *   info: a pointer on the structure who contains all our information
 * Output:
 *   return the window widget
 * Notes:
 */
static GtkWidget *ui_record_editor_init(UI_RecordSelect *info)
{
  GtkWidget *hbox, *vbox, *radio;
  GtkWidget *dialog, *widget, *menu;
  GtkWidget *menuitem, *table, *cancel;

  trace("");

  dialog = gtk_dialog_new();
  gtk_window_set_title (GTK_WINDOW(dialog), _("Select Record"));
  gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
      GTK_SIGNAL_FUNC (on_close_dialog), (gpointer)TRUE);

  vbox=GTK_DIALOG(dialog)->vbox;

  table=gtk_table_new(4,2,FALSE);
  gtk_box_pack_start_defaults(GTK_BOX(vbox), table);

  /* 1st row: choose date before ... */
  radio=gtk_radio_button_new_with_label(NULL,_("Before the "));
  gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 0, 1,
      GTK_FILL, 0, 0, 0);
  widget=gnofin_calendar_new(&info->date_begin);
  gtk_object_set_user_data(GTK_OBJECT(info->date_begin.date),(gpointer)radio);
  gtk_table_attach (GTK_TABLE (table), widget, 1, 2, 0, 1,
                    GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (radio), "clicked",
  		      GTK_SIGNAL_FUNC (on_radio_button_changed_before),
		      (gpointer)info);
  info->before=GTK_RADIO_BUTTON(radio);

  /* 2st row: choose date after ... */
  radio=gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radio),_("After the "));
  gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 1, 2,
      (GtkAttachOptions) (GTK_FILL),
      (GtkAttachOptions) (0), 0, 0);
  widget=gnofin_calendar_new(&info->date_end);
  gtk_object_set_user_data(GTK_OBJECT(info->date_end.date),(gpointer)radio);
  gtk_table_attach (GTK_TABLE (table), widget, 1, 2, 1, 2,
      GTK_EXPAND | GTK_FILL, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (radio), "clicked",
      GTK_SIGNAL_FUNC (on_radio_button_changed_after),
      (gpointer)info);
  info->after=GTK_RADIO_BUTTON(radio);

  /* 3rd row: choose date between ... */
  radio=gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radio),_("Between the\nand the"));
  gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 2, 3,
      GTK_FILL, 0, 0, 0);
  vbox=gtk_vbox_new (FALSE, 0);
  gtk_table_attach (GTK_TABLE (table), vbox, 1, 2, 2, 3,
      GTK_FILL, GTK_FILL, 0, 0);
  widget=gnofin_calendar_new(&info->date_between_begin);
  gtk_object_set_user_data(GTK_OBJECT(info->date_between_begin.date),(gpointer)radio);
  gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 0);
  widget=gnofin_calendar_new(&info->date_between_end);
  gtk_object_set_user_data(GTK_OBJECT(info->date_between_end.date),(gpointer)radio);
  gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 0);
  gtk_signal_connect (GTK_OBJECT (radio), "clicked",
      GTK_SIGNAL_FUNC (on_radio_button_changed_between),
      (gpointer)info);
  info->between=GTK_RADIO_BUTTON(radio);

  /* 4th row: choose date before a non fixed date ... */
  radio=gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radio),_("Since"));
  gtk_table_attach (GTK_TABLE (table), radio, 0, 1, 3, 4,
      GTK_FILL, 0, 0, 0);
  info->last=GTK_RADIO_BUTTON(radio);
  gtk_signal_connect (GTK_OBJECT (radio), "clicked",
  		      GTK_SIGNAL_FUNC (on_radio_button_changed_last),
		      (gpointer)info);

  hbox=gtk_hbox_new (TRUE, 0);
  gtk_table_attach (GTK_TABLE (table), hbox, 1, 2, 3, 4,
      GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);

   {
     int i;
     char tmp[4];
     widget=gtk_option_menu_new();
     menu=gtk_menu_new();
     for (i=1;i<MAX_NUMBER_OF_PERIOD;i++)
      {
	sprintf(tmp,"%d",i);
	menuitem=gtk_menu_item_new_with_label(tmp);
	gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                        GTK_SIGNAL_FUNC(on_menuitem_changed_num),
			(gpointer)info);
	gtk_menu_append (GTK_MENU (menu), menuitem);
	gtk_object_set_user_data(GTK_OBJECT(menuitem),(gpointer)i);
      }
     gtk_option_menu_set_menu(GTK_OPTION_MENU(widget),menu);
     gtk_box_pack_start(GTK_BOX(hbox),widget,FALSE,TRUE,0);
   }

   {
     int i;
     widget=gtk_option_menu_new();
     menu=gtk_menu_new();
     i=0;
     while (periods_name[i])
      {
	menuitem=gtk_menu_item_new_with_label(periods_name[i]);
	gtk_signal_connect (GTK_OBJECT (menuitem), "activate",
                        GTK_SIGNAL_FUNC(on_menuitem_changed_period),
			(gpointer)info);
	gtk_menu_append (GTK_MENU (menu), menuitem);
	gtk_object_set_user_data(GTK_OBJECT(menuitem),(gpointer)i);
	i++;
      }
     gtk_option_menu_set_menu(GTK_OPTION_MENU(widget),menu);
     gtk_box_pack_start(GTK_BOX(hbox),widget,FALSE,TRUE,0);
   }

  /* 5th row: view all record */
  radio=gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radio),_("All records not checked"));
  gtk_table_attach (GTK_TABLE (table), radio, 0, 2, 4, 5,
      GTK_FILL, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (radio), "clicked",
  		      GTK_SIGNAL_FUNC (on_radio_button_changed_not_checked),
		      (gpointer)info);
  info->notchecked=GTK_RADIO_BUTTON(radio);

  /* 6th row: view all record */
  radio=gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(radio),_("All records"));
  gtk_table_attach (GTK_TABLE (table), radio, 0, 2, 5, 6,
      GTK_FILL, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (radio), "clicked",
  		      GTK_SIGNAL_FUNC (on_radio_button_changed_all),
		      (gpointer)info);
  info->all=GTK_RADIO_BUTTON(radio);

  /* Last Line: Action button */
  cancel = gtk_button_new_with_label(_("Cancel"));
  gtk_signal_connect_object (GTK_OBJECT (cancel), "clicked",
      GTK_SIGNAL_FUNC (gtk_widget_hide), (gpointer)dialog);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
      cancel);

  return dialog;
}




/******************************************************************************
 * Interface functions
 */

void
ui_select_records_account (UI *ui, Account *account)
{
  static UI_RecordSelect *info=NULL;

  trace ("");

  if (info==NULL)
   { /* The window need to be created */
     info=calloc(1,sizeof(UI_RecordSelect));
     info->account = account;
     info->period_number=1;
     //info.account_name = if_account_get_name (account);
     info->dialog=ui_record_editor_init(info);
     gtk_window_set_title((GtkWindow *)info->dialog,if_account_get_name(account));
   }
  else
   {
     gchar *text1, *text2;
     /* Popup the window and change there name */
     info->account=account;
     gtk_window_set_title((GtkWindow *)info->dialog,if_account_get_name(account));

     text1 = date_stringize (NULL, 0, &account->records_select.date1);
     text2 = date_stringize (NULL, 0, &account->records_select.date2);
     switch(account->records_select.method)
      {
       case BEFORE:
	 gtk_toggle_button_set_active((GtkToggleButton *)info->before,TRUE);
	 gtk_entry_set_text(info->date_begin.date,text1);
	 break;
       case AFTER:
	 gtk_toggle_button_set_active((GtkToggleButton *)info->after,TRUE);
	 gtk_entry_set_text(info->date_end.date,text1);
	 break;
       case BETWEEN:
	 gtk_toggle_button_set_active((GtkToggleButton *)info->between,TRUE);
	 gtk_entry_set_text(info->date_between_begin.date,text1);
	 gtk_entry_set_text(info->date_between_end.date,text2);
	 break;
       case LAST:
	 gtk_toggle_button_set_active((GtkToggleButton *)info->last,TRUE);
	 break;
       case NOTCHECKED:
	 gtk_toggle_button_set_active((GtkToggleButton *)info->notchecked,TRUE);
	 break;
       case ALL:
	 gtk_toggle_button_set_active((GtkToggleButton *)info->all,TRUE);
	 break;
      }
     g_free (text1);
     g_free (text2);
   }
  gtk_widget_show_all(info->dialog);

}

