/* GTK - The GIMP Toolkit
 *
 * Copyright (C) 2003 Sun Microsystems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors:
 *	Mark McLoughlin <mark@skynet.ie>
 */

#include <config.h>

#include "gtkexpander.h"

#include <gtk/gtklabel.h>
#include <gtk/gtkcontainer.h>
#include <gdk/gdkkeysyms.h>

#define _(x) x

#define DEFAULT_EXPANDER_SIZE 10
#define DEFAULT_EXPANDER_SPACING 2

enum {
  PROP_0,
  PROP_EXPANDED,
  PROP_EXPANDED_LABEL,
  PROP_COLLAPSED_LABEL,
  PROP_PADDING
};

enum {
  TOGGLE,
  LAST_SIGNAL
};

struct _GtkExpanderPrivate {
  gchar            *expanded_label;
  gchar            *collapsed_label;
  gint              padding;

  GtkWidget        *label;

  GtkExpanderStyle  expander_style;
  guint             animation_timeout;

  guint      expanded : 1;
  guint      button_down : 1;
};

static void gtk_expander_class_init (GtkExpanderClass *klass);
static void gtk_expander_init       (GtkExpander      *expander);

static void gtk_expander_finalize     (GObject          *object);
static void gtk_expander_set_property (GObject          *object,
				       guint             prop_id,
				       const GValue     *value,
				       GParamSpec       *pspec);
static void gtk_expander_get_property (GObject          *object,
				       guint             prop_id,
				       GValue           *value,
				       GParamSpec       *pspec);

static void gtk_expander_destroy (GtkObject *object);

static void     gtk_expander_realize        (GtkWidget        *widget);
static void     gtk_expander_size_request   (GtkWidget        *widget,
					     GtkRequisition   *requisition);
static void     gtk_expander_size_allocate  (GtkWidget        *widget,
					     GtkAllocation    *allocation);
static void     gtk_expander_map            (GtkWidget        *widget);
static gboolean gtk_expander_expose         (GtkWidget        *widget,
					     GdkEventExpose   *event);
static gboolean gtk_expander_button_press   (GtkWidget        *widget,
					     GdkEventButton   *event);
static gboolean gtk_expander_button_release (GtkWidget        *widget,
					     GdkEventButton   *event);
static gboolean gtk_expander_motion_notify  (GtkWidget        *widget,
					     GdkEventMotion   *event);
static gboolean gtk_expander_enter_notify   (GtkWidget        *widget,
					     GdkEventCrossing *event);
static gboolean gtk_expander_leave_notify   (GtkWidget        *widget,
					     GdkEventCrossing *event);
static gboolean gtk_expander_focus          (GtkWidget        *widget,
					     GtkDirectionType  direction);

static void gtk_expander_add (GtkContainer *container,
			      GtkWidget    *widget);

static GtkBinClass *parent_class = NULL;

GType
gtk_expander_get_type (void)
{
  static GType expander_type = 0;
  
  if (!expander_type)
    {
      static const GTypeInfo expander_info =
      {
	sizeof (GtkExpanderClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
	(GClassInitFunc) gtk_expander_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */
	sizeof (GtkExpander),
	0,		/* n_preallocs */
	(GInstanceInitFunc) gtk_expander_init,
      };
      
      expander_type = g_type_register_static (GTK_TYPE_BIN,
					      "GtkExpander",
					      &expander_info, 0);
    }
  
  return expander_type;
}

static void
gtk_expander_class_init (GtkExpanderClass *klass)
{
  GObjectClass *gobject_class;
  GtkObjectClass *gtkobject_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class    = (GObjectClass *) klass;
  gtkobject_class  = (GtkObjectClass *) klass;
  widget_class    = (GtkWidgetClass *) klass;
  container_class = (GtkContainerClass *) klass;

  gobject_class->finalize     = gtk_expander_finalize;
  gobject_class->set_property = gtk_expander_set_property;
  gobject_class->get_property = gtk_expander_get_property;

  gtkobject_class->destroy = gtk_expander_destroy;

  widget_class->realize              = gtk_expander_realize;
  widget_class->size_request         = gtk_expander_size_request;
  widget_class->size_allocate        = gtk_expander_size_allocate;
  widget_class->map                  = gtk_expander_map;
  widget_class->expose_event         = gtk_expander_expose;
  widget_class->button_press_event   = gtk_expander_button_press;
  widget_class->button_release_event = gtk_expander_button_release;
  widget_class->motion_notify_event  = gtk_expander_motion_notify;
  widget_class->enter_notify_event   = gtk_expander_enter_notify;
  widget_class->leave_notify_event   = gtk_expander_leave_notify;
  widget_class->focus                = gtk_expander_focus;

  container_class->add = gtk_expander_add;

  klass->toggle = gtk_expander_toggle;

  g_object_class_install_property (gobject_class,
				   PROP_EXPANDED,
				   g_param_spec_boolean ("expanded",
							 _("Expanded"),
							 _("Whether or not the expander is expanded"),
							 FALSE,
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  g_object_class_install_property (gobject_class,
				   PROP_EXPANDED_LABEL,
				   g_param_spec_string ("expanded-label",
							_("Expanded Label"),
							_("Text display when the expander is expanded"),
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  g_object_class_install_property (gobject_class,
				   PROP_COLLAPSED_LABEL,
				   g_param_spec_string ("collapsed-label",
							_("Collapsed Label"),
							_("Text display when the expander is collapsed"),
							NULL,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
  g_object_class_install_property (gobject_class,
				   PROP_PADDING,
				   g_param_spec_int ("padding",
						     _("Padding"),
						     _("Space to put between the label and the child"),
						     0,
						     G_MAXINT,
						     0,
						     G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

  gtk_widget_class_install_style_property (widget_class,
					   g_param_spec_int ("expander_size",
							     _("Expander Size"),
							     _("Size of the expander arrow"),
							     0,
							     G_MAXINT,
							     DEFAULT_EXPANDER_SIZE,
							     G_PARAM_READABLE));

  gtk_widget_class_install_style_property (widget_class,
					   g_param_spec_int ("expander_spacing",
							     _("Indicator Spacing"),
							     _("Spacing around expander arrow"),
							     0,
							     G_MAXINT,
							     DEFAULT_EXPANDER_SPACING,
							     G_PARAM_READABLE));

  widget_class->activate_signal =
    g_signal_new ("toggle",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET (GtkExpanderClass, toggle),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);
}

static void
gtk_expander_init (GtkExpander *expander)
{
  /* FIXME: use g_type_class_add_private() */
  expander->priv = g_new0 (GtkExpanderPrivate, 1);

  expander->priv->expanded = TRUE;

  GTK_WIDGET_SET_FLAGS (expander, GTK_CAN_FOCUS);
  GTK_WIDGET_UNSET_FLAGS (expander, GTK_NO_WINDOW);

  expander->priv->label = gtk_label_new (NULL);
  gtk_widget_set_parent (expander->priv->label, GTK_WIDGET (expander));
  gtk_widget_show (expander->priv->label);
}

static void
gtk_expander_finalize (GObject *object)
{
  GtkExpander *expander = GTK_EXPANDER (object);

  g_free (expander->priv->expanded_label);
  g_free (expander->priv->collapsed_label);

  g_free (expander->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gtk_expander_set_property (GObject      *object,
			   guint         prop_id,
			   const GValue *value,
			   GParamSpec   *pspec)
{
  GtkExpander *expander = GTK_EXPANDER (object);
                                                                                                             
  switch (prop_id)
    {
    case PROP_EXPANDED:
      gtk_expander_set_expanded (expander, g_value_get_boolean (value));
      break;
    case PROP_EXPANDED_LABEL:
      gtk_expander_set_expanded_label (expander, g_value_get_string (value));
      break;
    case PROP_COLLAPSED_LABEL:
      gtk_expander_set_collapsed_label (expander, g_value_get_string (value));
      break;
    case PROP_PADDING:
      gtk_expander_set_padding (expander, g_value_get_int (value));
      break;
    }
}

static void
gtk_expander_get_property (GObject    *object,
			   guint       prop_id,
			   GValue     *value,
			   GParamSpec *pspec)
{
  GtkExpander *expander = GTK_EXPANDER (object);
                                                                                                             
  switch (prop_id)
    {
    case PROP_EXPANDED:
      g_value_set_boolean (value, expander->priv->expanded);
      break;
    case PROP_EXPANDED_LABEL:
      g_value_set_string (value, expander->priv->expanded_label);
      break;
    case PROP_COLLAPSED_LABEL:
      g_value_set_string (value, expander->priv->collapsed_label);
      break;
    case PROP_PADDING:
      g_value_set_int (value, expander->priv->padding);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
gtk_expander_destroy (GtkObject *object)
{
  GtkExpander *expander = GTK_EXPANDER (object);

  if (expander->priv->animation_timeout)
    g_source_remove (expander->priv->animation_timeout);
  expander->priv->animation_timeout = 0;

  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

static void
gtk_expander_realize (GtkWidget *widget)
{
  GdkWindowAttr attributes;
  gint attributes_mask;
  gint border_width;

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

  border_width = GTK_CONTAINER (widget)->border_width;
                                                                                                             
  attributes.x = widget->allocation.x + border_width;
  attributes.y = widget->allocation.y + border_width;
  attributes.width = widget->allocation.width - 2 * border_width;
  attributes.height = widget->allocation.height - 2 * border_width;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget)     |
				GDK_POINTER_MOTION_MASK      |
				GDK_POINTER_MOTION_HINT_MASK |
				GDK_BUTTON_PRESS_MASK        |
				GDK_BUTTON_RELEASE_MASK      |
				GDK_EXPOSURE_MASK            |
				GDK_ENTER_NOTIFY_MASK        |
				GDK_LEAVE_NOTIFY_MASK;

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
				   &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, widget);

  widget->style = gtk_style_attach (widget->style, widget->window);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}
                                                                                                             
static void
gtk_expander_size_request (GtkWidget      *widget,
			   GtkRequisition *requisition)
{
  GtkExpander *expander;
  GtkBin *bin;
  gint border_width;
  gint expander_size;
  gint expander_spacing;
  gboolean interior_focus;
  gint focus_width;
  gint focus_pad;
  GtkRequisition label_requisition;

  expander = GTK_EXPANDER (widget);
  bin = GTK_BIN (widget);

  border_width = GTK_CONTAINER (widget)->border_width;

  gtk_widget_style_get (widget,
			"interior-focus", &interior_focus,
			"focus-line-width", &focus_width,
			"focus-padding", &focus_pad,
			"expander_size", &expander_size,
			"expander_spacing", &expander_spacing,
			NULL);

  gtk_widget_size_request (expander->priv->label, &label_requisition);

  requisition->width = expander_size + 2 * expander_spacing +
		       2 * focus_width + 2 * focus_pad +
		       label_requisition.width;
  requisition->height = MAX (expander_size + 2 * expander_spacing,
			     label_requisition.height +
			     (interior_focus ? 2 * focus_width + 2 * focus_pad : 0));
  if (!interior_focus)
    requisition->height += 2 * focus_width + 2 * focus_pad;

  if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
    {
      GtkRequisition child_requisition;

      gtk_widget_size_request (bin->child, &child_requisition);

      if (!interior_focus)
	child_requisition.width += 2 * focus_width + 2 * focus_pad;

      requisition->width = MAX (requisition->width, child_requisition.width);
      requisition->height += child_requisition.height + expander->priv->padding;
    }

  requisition->width  += 2 * border_width;
  requisition->height += 2 * border_width + 2 * expander->priv->padding;
}

static void
gtk_expander_size_allocate (GtkWidget     *widget,
			    GtkAllocation *allocation)
{
  GtkExpander *expander;
  GtkBin *bin;
  GtkRequisition label_requisition;
  GtkRequisition child_requisition;
  GtkAllocation label_allocation;
  gboolean child_visible = FALSE;
  gint border_width;
  gint expander_size;
  gint expander_spacing;
  gboolean interior_focus;
  gboolean ltr;
  gint focus_width;
  gint focus_pad;

  expander = GTK_EXPANDER (widget);
  bin = GTK_BIN (widget);

  border_width = GTK_CONTAINER (widget)->border_width;

  gtk_widget_style_get (widget,
			"interior-focus", &interior_focus,
			"focus-line-width", &focus_width,
			"focus-padding", &focus_pad,
			"expander_size", &expander_size,
			"expander_spacing", &expander_spacing,
			NULL);

  ltr = gtk_widget_get_direction (widget) != GTK_TEXT_DIR_RTL;

  gtk_widget_get_child_requisition (expander->priv->label, &label_requisition);

  child_requisition.width = 0;
  child_requisition.height = 0;
  if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
    {
      child_visible = TRUE;
      gtk_widget_get_child_requisition (bin->child, &child_requisition);
    }

  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget))
    gdk_window_move_resize (widget->window,
			    allocation->x + border_width,
			    allocation->y + border_width,
                            MAX (allocation->width - 2 * border_width, 0),
                            MAX (allocation->height - 2 * border_width, 0));

  label_allocation.x = focus_width + focus_pad;
  if (ltr)
    label_allocation.x = expander_size + 2 * expander_spacing;
  label_allocation.y = expander->priv->padding + focus_width + focus_pad;

  label_allocation.width = MIN (label_requisition.width,
				allocation->width - 2 * border_width -
				expander_size - 2 * expander_spacing -
				2 * focus_width - 2 * focus_pad);
  label_allocation.width = MAX (label_allocation.width, 1);

  label_allocation.height = MIN (label_requisition.height,
				 allocation->height - 2 * border_width -
				 2 * expander->priv->padding -
				 2 * focus_width - 2 * focus_pad -
				 child_requisition.height -
				 (child_visible ? expander->priv->padding : 0));
  label_allocation.height = MAX (label_allocation.height, 1);

  gtk_widget_size_allocate (expander->priv->label, &label_allocation);

  if (child_visible)
    {
      GtkAllocation child_allocation;
      gint top_height;

      top_height = MAX (2 * expander_spacing + expander_size,
			label_allocation.height +
			(interior_focus ? 2 * focus_width + 2 * focus_pad : 0));

      child_allocation.x = 0;
      child_allocation.y = 2 * expander->priv->padding + top_height;

      if (!interior_focus)
	{
	  child_allocation.x += focus_width + focus_pad;
	  child_allocation.y += focus_width + focus_pad;
	}

      child_allocation.width = MIN (child_requisition.width,
				    allocation->width - 2 * border_width -
				    (!interior_focus ? 2 * focus_width + 2 * focus_pad : 0));
      child_allocation.width = MAX (child_allocation.width, 1);

      child_allocation.height = MIN (child_requisition.height,
				     allocation->height - top_height -
				     2 * border_width -
				     3 * expander->priv->padding -
				     (!interior_focus ? 2 * focus_width + 2 * focus_pad : 0));
      child_allocation.height = MAX (child_allocation.height, 1);

      gtk_widget_size_allocate (bin->child, &child_allocation);
    }
}

static void
gtk_expander_map (GtkWidget *widget)
{
  gtk_widget_map (GTK_EXPANDER (widget)->priv->label);

  GTK_WIDGET_CLASS (parent_class)->map (widget);
}

static GdkRectangle
get_expander_bounds (GtkExpander *expander)
{
  GtkWidget *widget;
  GtkBin *bin;
  GdkRectangle bounds;
  GtkAllocation label_allocation;
  gint border_width;
  gint expander_size;
  gint expander_spacing;
  gboolean interior_focus;
  gint focus_width;
  gint focus_pad;
  gboolean ltr;

  widget = GTK_WIDGET (expander);
  bin = GTK_BIN (expander);

  border_width = GTK_CONTAINER (expander)->border_width;

  gtk_widget_style_get (widget,
			"interior-focus", &interior_focus,
			"focus-line-width", &focus_width,
			"focus-padding", &focus_pad,
			"expander_size", &expander_size,
			"expander_spacing", &expander_spacing,
			NULL);

  ltr = gtk_widget_get_direction (widget) != GTK_TEXT_DIR_RTL;

  label_allocation = expander->priv->label->allocation;

  if (ltr)
    bounds.x = expander_spacing;
  else
    bounds.x = widget->allocation.width - 2 * border_width -
	       expander_spacing - expander_size;
  bounds.y = label_allocation.y + MAX ((label_allocation.height - expander_size) / 2, 0);

  if (!interior_focus)
    {
      if (ltr)
	bounds.x += focus_width + focus_pad;
      else
	bounds.x -= focus_width + focus_pad;
      bounds.y += focus_width + focus_pad;
    }

  bounds.width = bounds.height = expander_size;

  return bounds;
}

static void
gtk_expander_paint (GtkExpander *expander)
{
  GtkWidget *widget;
  gint x, y;
  GtkStateType state;
  GdkRectangle clip;

  clip = get_expander_bounds (expander);

  x = clip.x + clip.width / 2;
  y = clip.y + clip.height / 2;

  widget = GTK_WIDGET (expander);

  state = widget->state;
  if (state != GTK_STATE_PRELIGHT)
    state = GTK_STATE_NORMAL;

  gtk_paint_expander (widget->style,
		      widget->window,
		      state,
		      &clip,
		      widget,
		      "expander",
		      x,
		      y,
		      expander->priv->expander_style);
}

static void
gtk_expander_paint_focus (GtkExpander  *expander,
			  GdkRectangle *area)
{
  GtkWidget *widget;
  gint x, y, width, height;
  gboolean interior_focus;
  gint focus_width;
  gint focus_pad;
  gint expander_size;
  gint expander_spacing;
  gboolean ltr;

  widget = GTK_WIDGET (expander);

  gtk_widget_style_get (widget,
			"interior-focus", &interior_focus,
			"focus-line-width", &focus_width,
			"focus-padding", &focus_pad,
			"expander_size", &expander_size,
			"expander_spacing", &expander_spacing,
			NULL);

  ltr = gtk_widget_get_direction (widget) != GTK_TEXT_DIR_RTL;

  if (interior_focus)
    {
      GtkAllocation label_allocation = expander->priv->label->allocation;

      if (ltr)
	x = expander_spacing * 2 + expander_size;
      else
	x = 0;
      y = expander->priv->padding;
      width = label_allocation.width + 2 * focus_pad + 2 * focus_width;
      height = label_allocation.height + 2 * focus_pad + 2 * focus_width;
    }
  else
    {
      x = y = 0;
      width = widget->allocation.width - 2 * GTK_CONTAINER (widget)->border_width;
      height = widget->allocation.height - 2 * GTK_CONTAINER (widget)->border_width;
    }

  gtk_paint_focus (widget->style, widget->window, GTK_WIDGET_STATE (widget),
		   area, widget, "expander",
		   x, y, width, height);
}

static gboolean
gtk_expander_expose (GtkWidget      *widget,
		     GdkEventExpose *event)
{
  if (GTK_WIDGET_DRAWABLE (widget))
    {
      GtkExpander *expander = GTK_EXPANDER (widget);

      gtk_expander_paint (expander);
      gtk_expander_paint_focus (expander, &event->area);
      gtk_container_propagate_expose (GTK_CONTAINER (widget),
				      GTK_EXPANDER (widget)->priv->label,
				      event);

      GTK_WIDGET_CLASS (parent_class)->expose_event (widget, event);
    }

  return FALSE;
}

static gboolean
is_in_expander_panel (GtkExpander *expander,
		      GdkWindow   *window,
		      gint         x,
		      gint         y)
{
  GtkWidget *widget;
  GtkBin *bin;
  GdkRectangle area;
  gint border_width;

  widget = GTK_WIDGET (expander);
  bin = GTK_BIN (expander);

  border_width = GTK_CONTAINER (expander)->border_width;

  area = get_expander_bounds (expander);

  if (widget->window == window)
    {
      if (x >= area.x && x <= (area.x + area.width) &&
	  y >= area.y && y <= (area.y + area.height))
	return TRUE;
    }

  return FALSE;
}

static gboolean
gtk_expander_button_press (GtkWidget      *widget,
			   GdkEventButton *event)
{
  GtkExpander *expander = GTK_EXPANDER (widget);

  if (event->button == 1 && !expander->priv->button_down)
    {
      if (is_in_expander_panel (expander, event->window, event->x, event->y))
	{
	  expander->priv->button_down = TRUE;
	  return TRUE;
	}
    }

  return FALSE;
}

static gboolean
gtk_expander_button_release (GtkWidget      *widget,
			     GdkEventButton *event)
{
  GtkExpander *expander = GTK_EXPANDER (widget);
  gboolean retval = FALSE;;

  if (event->button == 1 && expander->priv->button_down)
    {
      gtk_expander_toggle (expander);
      expander->priv->button_down = FALSE;
      retval = TRUE;
    }

  return FALSE;
}

static void
gtk_expander_maybe_prelight (GtkExpander *expander)
{
  GtkWidget *widget;
  GtkStateType state = GTK_STATE_NORMAL;

  widget = GTK_WIDGET (expander);

  if (!expander->priv->button_down)
    {
      gint x, y;

      gdk_window_get_pointer (widget->window, &x, &y, NULL);

      if (is_in_expander_panel (expander, widget->window, x, y))
	state = GTK_STATE_PRELIGHT;
    }

  gtk_widget_set_state (widget, state);
}

static gboolean
gtk_expander_motion_notify (GtkWidget      *widget,
			    GdkEventMotion *event)
{
  gtk_expander_maybe_prelight (GTK_EXPANDER (widget));

  return FALSE;
}

static gboolean
gtk_expander_enter_notify (GtkWidget        *widget,
			   GdkEventCrossing *event)
{
  gtk_expander_maybe_prelight (GTK_EXPANDER (widget));

  return FALSE;
}

static gboolean
gtk_expander_leave_notify (GtkWidget        *widget,
			   GdkEventCrossing *event)
{
  gtk_widget_set_state (widget, GTK_STATE_NORMAL);

  return FALSE;
}

static gboolean
focus_child_in (GtkWidget        *widget,
		GtkDirectionType  direction)
{
  GtkBin *bin = GTK_BIN (widget);

  if (!bin->child || !GTK_WIDGET_VISIBLE (bin->child))
    return FALSE;

  return gtk_widget_child_focus (bin->child, direction);
}

static gboolean
gtk_expander_focus (GtkWidget        *widget,
		    GtkDirectionType  direction)
{
  GtkWidget *old_focus_child;
  gboolean widget_is_focus;

  widget_is_focus = gtk_widget_is_focus (widget);
  old_focus_child = GTK_CONTAINER (widget)->focus_child;

  if (old_focus_child)
    {
      if (gtk_widget_child_focus (old_focus_child, direction))
	return TRUE;

      switch (direction)
	{
	case GTK_DIR_TAB_BACKWARD:
	case GTK_DIR_UP:
	  gtk_widget_grab_focus (widget);
	  return TRUE;
	case GTK_DIR_DOWN:
	case GTK_DIR_TAB_FORWARD:
	case GTK_DIR_LEFT:
	case GTK_DIR_RIGHT:
	  return FALSE;
	}
    }
  else if (widget_is_focus)
    {
      switch (direction)
	{
	case GTK_DIR_TAB_BACKWARD:
	case GTK_DIR_UP:
	  return FALSE;
	case GTK_DIR_DOWN:
	case GTK_DIR_TAB_FORWARD:
	case GTK_DIR_LEFT:
	case GTK_DIR_RIGHT:
	  return focus_child_in (widget, direction);
	}
    }
  else
    {
      switch (direction)
	{
	case GTK_DIR_DOWN:
	case GTK_DIR_TAB_FORWARD:
	  gtk_widget_grab_focus (widget);
	  return TRUE;
	case GTK_DIR_TAB_BACKWARD:
	case GTK_DIR_UP:
	case GTK_DIR_LEFT:
	case GTK_DIR_RIGHT:
	  return focus_child_in (widget, direction);
	}
    }

  g_assert_not_reached ();
  return FALSE;
}

static void
gtk_expander_add (GtkContainer *container,
		  GtkWidget    *widget)
{
  GTK_CONTAINER_CLASS (parent_class)->add (container, widget);

  g_object_set (G_OBJECT (widget),
		"visible", GTK_EXPANDER (container)->priv->expanded,
		NULL);
}

GtkWidget *
gtk_expander_new (const gchar *expanded_label,
		  const gchar *collapsed_label,
		  gint         padding)
{
  return g_object_new (GTK_TYPE_EXPANDER,
		       "expanded-label", expanded_label,
		       "collapsed-label", collapsed_label,
		       "padding", padding,
		       NULL);
}

void
gtk_expander_set_expanded_label (GtkExpander *expander,
				 const gchar *expanded_label)
{
  g_return_if_fail (GTK_IS_EXPANDER (expander));

  g_free (expander->priv->expanded_label);
  expander->priv->expanded_label = g_strdup (expanded_label);

  if (expander->priv->expanded)
    gtk_label_set_text (GTK_LABEL (expander->priv->label),
			expander->priv->expanded_label);

  g_object_notify (G_OBJECT (expander), "expanded-label");
}

const char *
gtk_expander_get_expanded_label (GtkExpander *expander)
{
  g_return_val_if_fail (GTK_IS_EXPANDER (expander), NULL);

  return expander->priv->expanded_label;
}

void
gtk_expander_set_collapsed_label (GtkExpander *expander,
				  const gchar *collapsed_label)
{
  g_return_if_fail (GTK_IS_EXPANDER (expander));

  g_free (expander->priv->collapsed_label);
  expander->priv->collapsed_label = g_strdup (collapsed_label);

  if (!expander->priv->expanded)
    gtk_label_set_text (GTK_LABEL (expander->priv->label),
			expander->priv->collapsed_label);

  g_object_notify (G_OBJECT (expander), "collapsed-label");
}

const char *
gtk_expander_get_collapsed_label (GtkExpander *expander)
{
  g_return_val_if_fail (GTK_IS_EXPANDER (expander), NULL);

  return expander->priv->collapsed_label;
}

void
gtk_expander_set_padding (GtkExpander *expander,
			  gint         padding)
{
  g_return_if_fail (GTK_IS_EXPANDER (expander));
  g_return_if_fail (padding >= 0);

  if (expander->priv->padding != padding)
    {
      expander->priv->padding = padding;

      gtk_widget_queue_resize (GTK_WIDGET (expander));

      g_object_notify (G_OBJECT (expander), "padding");
    }
}

gint
gtk_expander_get_padding (GtkExpander *expander)
{
  g_return_val_if_fail (GTK_IS_EXPANDER (expander), 0);

  return expander->priv->padding;
}

static gboolean
gtk_expander_animation_timeout (GtkExpander *expander)
{
  GdkRectangle area;
  gboolean finish = FALSE;

  if (GTK_WIDGET_REALIZED (expander))
    {
      area = get_expander_bounds (expander);
      gdk_window_invalidate_rect (GTK_WIDGET (expander)->window, &area, TRUE);
    }

  if (expander->priv->expanded)
    {
      if (expander->priv->expander_style == GTK_EXPANDER_COLLAPSED)
	{
	  expander->priv->expander_style = GTK_EXPANDER_SEMI_EXPANDED;
	}
      else
	{
	  expander->priv->expander_style = GTK_EXPANDER_EXPANDED;
	  finish = TRUE;
	}
    }
  else
    {
      if (expander->priv->expander_style == GTK_EXPANDER_EXPANDED)
	{
	  expander->priv->expander_style = GTK_EXPANDER_SEMI_COLLAPSED;
	}
      else
	{
	  expander->priv->expander_style = GTK_EXPANDER_COLLAPSED;
	  finish = TRUE;
	}
    }

  if (finish)
    {
      expander->priv->animation_timeout = 0;
      g_object_set (G_OBJECT (GTK_BIN (expander)->child),
		    "visible", expander->priv->expanded,
		    NULL);
    }

  return !finish;
}

static void
gtk_expander_start_animation (GtkExpander *expander)
{
  if (expander->priv->animation_timeout)
    g_source_remove (expander->priv->animation_timeout);

  expander->priv->animation_timeout =
		g_timeout_add (50,
			       (GSourceFunc) gtk_expander_animation_timeout,
			       expander);
}

void
gtk_expander_set_expanded (GtkExpander *expander,
			   gboolean     expanded)
{
  g_return_if_fail (GTK_IS_EXPANDER (expander));

  expanded = expanded != FALSE;

  if (expander->priv->expanded != expanded)
    {
      char *text;

      expander->priv->expanded = expanded;

      if (expander->priv->expanded)
	text = expander->priv->expanded_label;
      else
	text = expander->priv->collapsed_label;

      gtk_label_set_text (GTK_LABEL (expander->priv->label), text);

      if (GTK_WIDGET_VISIBLE (expander))
	gtk_expander_start_animation (expander);

      else if (GTK_BIN (expander)->child)
	{
	  expander->priv->expander_style = expanded ? GTK_EXPANDER_EXPANDED :
						      GTK_EXPANDER_COLLAPSED;
	  g_object_set (G_OBJECT (GTK_BIN (expander)->child),
			"visible", expander->priv->expanded,
			NULL);
	}

      gtk_widget_queue_resize (GTK_WIDGET (expander));

      g_object_notify (G_OBJECT (expander), "expanded");
    }
}

gboolean
gtk_expander_get_expanded (GtkExpander *expander)
{
  g_return_val_if_fail (GTK_IS_EXPANDER (expander), FALSE);

  return expander->priv->expanded;
}

void
gtk_expander_expand (GtkExpander *expander)
{
  g_return_if_fail (GTK_IS_EXPANDER (expander));

  gtk_expander_set_expanded (expander, TRUE);
}

void
gtk_expander_collapse (GtkExpander *expander)
{
  g_return_if_fail (GTK_IS_EXPANDER (expander));

  gtk_expander_set_expanded (expander, FALSE);
}

void
gtk_expander_toggle (GtkExpander *expander)
{
  g_return_if_fail (GTK_IS_EXPANDER (expander));

  gtk_expander_set_expanded (expander, !expander->priv->expanded);
}
