/* gtkcombo - combo widget for gtk+
 * Copyright 1997 Paolo Molaro
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

/*
 * Modified by Marcus Bjurman <marbj499@student.liu.se> 2001-2003
 * The orginal comments are left intact above
 */

#include "gnome-cmd-includes.h"
#include "gnome-cmd-combo.h"
#include "gnome-cmd-style.h"
#include "gnome-cmd-data.h"

enum {
  ITEM_SELECTED,
  POPWIN_HIDDEN,
  LAST_SIGNAL
};

const gchar *gnome_cmd_combo_string_key = "gnome-cmd-combo-string-value";

#define COMBO_LIST_MAX_HEIGHT	(400)
#define	EMPTY_LIST_HEIGHT	(15)

static GtkHBoxClass *parent_class = NULL;

static guint combo_signals[LAST_SIGNAL] = { 0 };


static void
gnome_cmd_combo_item_selected (GnomeCmdCombo *combo, gpointer data);


/*******************************
 * Utility functions
 *******************************/

static void
size_allocate (GtkWidget     *widget,
			   GtkAllocation *allocation)
{
	GnomeCmdCombo *combo;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (GNOME_CMD_IS_COMBO (widget));
	g_return_if_fail (allocation != NULL);

	GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);
  
	combo = GNOME_CMD_COMBO (widget);

	if (combo->entry->allocation.height > combo->entry->requisition.height)
    {
		GtkAllocation button_allocation;

		button_allocation = combo->button->allocation;
		button_allocation.height = combo->entry->requisition.height;
		button_allocation.y = combo->entry->allocation.y + 
			(combo->entry->allocation.height - combo->entry->requisition.height) 
			/ 2;
		gtk_widget_size_allocate (combo->button, &button_allocation);
    }
}


static void
get_pos (GnomeCmdCombo *combo, gint *x, gint *y, gint *height, gint *width)
{
	GtkBin *popwin;
	GtkWidget *widget;
	GtkScrolledWindow *popup;
  
	gint real_height;
	GtkRequisition list_requisition;
	gboolean show_hscroll = FALSE;
	gboolean show_vscroll = FALSE;
	gint avail_height;
	gint min_height;
	gint alloc_width;
	gint work_height;
	gint old_height;
	gint old_width;
  
	widget = GTK_WIDGET(combo);
	popup  = GTK_SCROLLED_WINDOW (combo->popup);
	popwin = GTK_BIN (combo->popwin);
  
	gdk_window_get_origin (combo->entry->window, x, y);
	real_height = MIN (combo->entry->requisition.height, 
					   combo->entry->allocation.height);
	*y += real_height;
	avail_height = gdk_screen_height () - *y;
  
	gtk_widget_size_request (combo->list, &list_requisition);
	min_height = MIN (list_requisition.height, 
					  popup->vscrollbar->requisition.height);
	if (!GTK_CLIST (combo->list)->rows)
		list_requisition.height += EMPTY_LIST_HEIGHT;
  
	alloc_width = (widget->allocation.width -
				   2 * popwin->child->style->klass->xthickness -
				   2 * GTK_CONTAINER (popwin->child)->border_width -
				   2 * GTK_CONTAINER (combo->popup)->border_width -
				   2 * GTK_CONTAINER (GTK_BIN (popup)->child)->border_width - 
				   2 * GTK_BIN (popup)->child->style->klass->xthickness) + 100;
  
	work_height = (2 * popwin->child->style->klass->ythickness +
				   2 * GTK_CONTAINER (popwin->child)->border_width +
				   2 * GTK_CONTAINER (combo->popup)->border_width +
				   2 * GTK_CONTAINER (GTK_BIN (popup)->child)->border_width +
				   2 * GTK_BIN (popup)->child->style->klass->xthickness)+20;
  
	do 
    {
		old_width = alloc_width;
		old_height = work_height;
      
		if (!show_hscroll &&
			alloc_width < list_requisition.width)
		{
			work_height += popup->hscrollbar->requisition.height +
				GTK_SCROLLED_WINDOW_CLASS 
				(GTK_OBJECT (combo->popup)->klass)->scrollbar_spacing;
			show_hscroll = TRUE;
		}
		if (!show_vscroll && 
			work_height + list_requisition.height > avail_height)
		{
			if (work_height + min_height > avail_height && 
				*y - real_height > avail_height)
			{
				*y -= (work_height + list_requisition.height + real_height);
				break;
			}
			alloc_width -= 
				popup->vscrollbar->requisition.width +
				GTK_SCROLLED_WINDOW_CLASS 
				(GTK_OBJECT (combo->popup)->klass)->scrollbar_spacing;
			show_vscroll = TRUE;
		}
    } while (old_width != alloc_width || old_height != work_height);
  
	*width = widget->allocation.width;
	if (*width < 200)
		*width = 200;
	
	if (show_vscroll)
		*height = avail_height;
	else
		*height = work_height + list_requisition.height;
  
	if (*x < 0)
		*x = 0;
}


void
gnome_cmd_combo_popup_list (GnomeCmdCombo *combo)
{
	gint height, width, x, y;

	get_pos (combo, &x, &y, &height, &width);

	gtk_widget_set_uposition (combo->popwin, x, y);
	gtk_widget_set_usize (combo->popwin, width, height);
	gtk_widget_realize (combo->popwin);
	gdk_window_resize (combo->popwin->window, width, height);
	gtk_widget_show (combo->popwin);

	if (combo->first_popup) {
		gtk_clist_unselect_all (GTK_CLIST (combo->list));
		GTK_CLIST (combo->list)->focus_row = 0;
		gtk_clist_select_row (GTK_CLIST (combo->list), 0, 0);
		combo->first_popup = FALSE;
	}
}





/*******************************
 * Callbacks
 *******************************/

static gboolean        
on_popup_button_press (GtkWidget        *button,
					   GdkEventButton   *event,
					   GnomeCmdCombo    *combo)
{
	GtkWidget *child;

	if (!event) return FALSE;
	if (event->button != 1) return FALSE;

	// Check to see if we released inside the button
	child = gtk_get_event_widget ((GdkEvent*) event);
	
	while (child && child != (combo->button))
		child = child->parent;
	
	if (child == combo->button) {
		return FALSE;
	}

//	gtk_widget_hide (combo->popwin);
//	combo->is_popped = FALSE;
	return TRUE;
}


static gboolean        
on_popup_button_release (GtkWidget        *button,
						 GdkEventButton   *event,
						 GnomeCmdCombo    *combo)
{
	GtkWidget *child;
	
	if (!event) return FALSE;
	if (event->button != 1) return FALSE;

	// Check to see if we released inside the button
	child = gtk_get_event_widget ((GdkEvent*) event);
	
	while (child && child != (combo->button))
		child = child->parent;
	
	if (child == combo->button) {
		if (combo->is_popped) {
			gtk_widget_hide (combo->popwin);
			combo->is_popped = FALSE;
		}
		else {
			gnome_cmd_combo_popup_list (combo);
			combo->is_popped = TRUE;
		}
		return FALSE;
	}

	return TRUE;
}


static int
on_list_key_press (GtkWidget *widget, GdkEventKey *event, GnomeCmdCombo *combo)
{
	if (event->keyval == GDK_Escape) {
		gtk_widget_hide (combo->popwin);
		combo->is_popped = FALSE;
		return TRUE;
    }

	return FALSE;
}


static gboolean        
on_popwin_button_released (GtkWidget        *button,
						  GdkEventButton   *event,
						  GnomeCmdCombo    *combo)
{
	GtkWidget *child;
	
	if (!event) return FALSE;
	if (event->button != 1) return FALSE;

	// Check to see if we clicked inside the popwin
	child = gtk_get_event_widget ((GdkEvent*) event);	
	while (child && child != (combo->popwin))
		child = child->parent;
	
	if (child != combo->popwin) {
		// We clicked outside the popwin
		gtk_widget_hide (combo->popwin);
		combo->is_popped = FALSE;
		return TRUE;
	}
	
	return FALSE;
}


static int
on_popwin_keypress (GtkWidget *widget, GdkEventKey *event, GnomeCmdCombo *combo)
{
	if (event->keyval == GDK_Return || event->keyval == GDK_KP_Enter) {
		gpointer data;
		gint row;

		row = GTK_CLIST (combo->list)->focus_row;
		if (row < 0) return TRUE;

		data = gtk_clist_get_row_data (GTK_CLIST (combo->list), row);
		gtk_signal_emit (GTK_OBJECT (combo), combo_signals[ITEM_SELECTED], data);
		
		return TRUE;
    }

	return FALSE;
}


static void
on_list_select_row (GtkCList *list, gint row, gint col,
					GdkEventButton *event, GnomeCmdCombo *combo)
{
	GdkEventKey event2;
	if (!event) return;

	event2.type = GDK_KEY_PRESS;
	event2.keyval = GDK_Return;
	event2.state = 0;

	on_popwin_keypress (GTK_WIDGET (combo->popwin), &event2, combo);
}


static void
on_popwin_show (GtkWidget *widget,
				GnomeCmdCombo *combo)
{
	gtk_grab_add (combo->popwin);
	gtk_widget_grab_focus (combo->list);
}


static void
on_popwin_hide (GtkWidget *widget,
				GnomeCmdCombo *combo)
{
	gtk_grab_remove (combo->popwin);
	gtk_signal_emit (GTK_OBJECT (combo), combo_signals[POPWIN_HIDDEN]);
}


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *combo)
{
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (combo);
}


static void
class_init (GnomeCmdComboClass *klass)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = gtk_type_class (gtk_hbox_get_type ());
	object_class = (GtkObjectClass *) klass;
	widget_class = (GtkWidgetClass *) klass;

	combo_signals[ITEM_SELECTED] =
		gtk_signal_new ("item_selected",
			GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCmdComboClass, item_selected),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE,
			1, GTK_TYPE_POINTER);

	combo_signals[POPWIN_HIDDEN] =
		gtk_signal_new ("popwin_hidden",
			GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GnomeCmdComboClass, popwin_hidden),
		    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE,
			0);
	
	gtk_object_class_add_signals (object_class, combo_signals, LAST_SIGNAL);
	
	object_class->destroy = destroy;  
	widget_class->size_allocate = size_allocate;
	klass->item_selected = gnome_cmd_combo_item_selected;
	klass->popwin_hidden = NULL;
}

static void
init (GnomeCmdCombo *combo)
{
	GtkWidget *arrow;
	GtkWidget *frame;
	GtkWidget *event_box;
	GdkCursor *cursor;

	combo->value_in_list = 0;
	combo->ok_if_empty = 1;
	combo->highest_pixmap = 20;
	combo->widest_pixmap = 20;
	combo->is_popped = FALSE;
	
	combo->entry = gtk_entry_new ();
	gtk_widget_ref (combo->entry);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "entry", combo->entry,
							  (GtkDestroyNotify) gtk_widget_unref);
//	gtk_widget_set_style (combo->entry, list_style);
	gtk_widget_show (combo->entry);
	gtk_widget_set_usize (combo->entry, 60, -1);
	
	combo->button = gtk_button_new ();
	gtk_widget_ref (combo->button);
	gtk_button_set_relief (GTK_BUTTON (combo->button), gnome_cmd_data_get_button_relief ());
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "button", combo->button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (combo->button);
	
	arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_widget_ref (arrow);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "arrow", arrow,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (arrow);
	
	gtk_container_add (GTK_CONTAINER (combo->button), arrow);
	gtk_box_pack_start (GTK_BOX (combo), combo->entry, TRUE, TRUE, 0);
	gtk_box_pack_end (GTK_BOX (combo), combo->button, FALSE, FALSE, 0);
	GTK_WIDGET_UNSET_FLAGS (combo->button, GTK_CAN_FOCUS);

	/* connect button signals */
	gtk_signal_connect_after (
		GTK_OBJECT (combo->button), "button-release-event",
		(GtkSignalFunc) on_popup_button_release, combo);
	gtk_signal_connect_after (
		GTK_OBJECT (combo->button), "button-press-event",
		(GtkSignalFunc) on_popup_button_press, combo);
	
	combo->popwin = gtk_window_new (GTK_WINDOW_POPUP);
	gtk_widget_ref (combo->popwin);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "popwin", combo->popwin,
							  (GtkDestroyNotify) gtk_widget_unref);	
	gtk_window_set_policy (GTK_WINDOW (combo->popwin), 1, 1, 0);

	gtk_widget_set_events (combo->popwin, GDK_KEY_PRESS_MASK);
	gtk_widget_set_events (combo->popwin, GDK_BUTTON_PRESS_MASK);

	/* connect popupwin signals */
	gtk_signal_connect (
		GTK_OBJECT (combo->popwin), "button-release-event",
		GTK_SIGNAL_FUNC (on_popwin_button_released), combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->popwin), "key-press-event",
		GTK_SIGNAL_FUNC (on_popwin_keypress), combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->popwin), "show",
		GTK_SIGNAL_FUNC (on_popwin_show), combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->popwin), "hide",
		GTK_SIGNAL_FUNC (on_popwin_hide), combo);
  
	event_box = gtk_event_box_new ();
	gtk_widget_ref (event_box);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "event_box", event_box,
							  (GtkDestroyNotify) gtk_widget_unref);	
	gtk_container_add (GTK_CONTAINER (combo->popwin), event_box);
	gtk_widget_show (event_box);
	gtk_widget_realize (event_box);
	
	cursor = gdk_cursor_new (GDK_TOP_LEFT_ARROW);
	gdk_window_set_cursor (event_box->window, cursor);
	gdk_cursor_destroy (cursor);

	frame = gtk_frame_new (NULL);
	gtk_widget_ref (frame);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "frame", frame,
							  (GtkDestroyNotify) gtk_widget_unref);	
	gtk_container_add (GTK_CONTAINER (event_box), frame);
	gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
	gtk_widget_show (frame);

	combo->popup = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_ref (combo->popup);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "combo->popup", combo->popup,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (combo->popup),
									GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	GTK_WIDGET_UNSET_FLAGS (GTK_SCROLLED_WINDOW (combo->popup)->hscrollbar, GTK_CAN_FOCUS);
	GTK_WIDGET_UNSET_FLAGS (GTK_SCROLLED_WINDOW (combo->popup)->vscrollbar, GTK_CAN_FOCUS);
	gtk_container_add (GTK_CONTAINER (frame), combo->popup);
	gtk_widget_show (combo->popup);
}



/***********************************
 * Public functions
 ***********************************/

guint
gnome_cmd_combo_get_type (void)
{
	static guint combo_type = 0;

	if (!combo_type)
    {
		static const GtkTypeInfo combo_info =
		{
			"GnomeCmdCombo",
			sizeof (GnomeCmdCombo),
			sizeof (GnomeCmdComboClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		combo_type = gtk_type_unique (gtk_hbox_get_type (), &combo_info);
    }
	return combo_type;
}

GtkWidget *
gnome_cmd_combo_new (gint num_cols, gchar **col_titles)
{
	GnomeCmdCombo *combo =  gtk_type_new (gnome_cmd_combo_get_type ());

	if (col_titles)
		combo->list = gnome_cmd_clist_new_with_titles (num_cols, col_titles);
	else
		combo->list = gnome_cmd_clist_new (num_cols);

	combo->first_popup = TRUE;
	
	gtk_widget_ref (combo->list);
	gtk_object_set_data_full (GTK_OBJECT (combo),
							  "combo->list", combo->list,
							  (GtkDestroyNotify) gtk_widget_unref);

	/* We'll use enter notify events to figure out when to transfer
	 * the grab to the list
	 */
	gtk_container_add (GTK_CONTAINER (combo->popup), combo->list);
	gtk_widget_show (combo->list);

	/* connect list signals */
	gtk_signal_connect_after (
		GTK_OBJECT (combo->list), "select-row",
		GTK_SIGNAL_FUNC (on_list_select_row), combo);
	gtk_signal_connect (
		GTK_OBJECT (combo->list), "key-press-event",
		GTK_SIGNAL_FUNC (on_list_key_press), combo);

	return GTK_WIDGET (combo);
}


void
gnome_cmd_combo_clear (GnomeCmdCombo *combo)
{
	gtk_clist_clear (GTK_CLIST (combo->list));
}


gint
gnome_cmd_combo_append (GnomeCmdCombo *combo, gchar **text, gpointer data)
{
	gint row;
	
	g_return_val_if_fail (GNOME_CMD_IS_COMBO (combo), -1);
	g_return_val_if_fail (text != NULL, -1);

	row = gtk_clist_append (GTK_CLIST (combo->list), text);
	gtk_clist_set_row_data (GTK_CLIST (combo->list), row, data);
	return row;
}


gint
gnome_cmd_combo_insert (GnomeCmdCombo *combo, gchar **text, gpointer data)
{
	gint row;
	
	g_return_val_if_fail (GNOME_CMD_IS_COMBO (combo), -1);
	g_return_val_if_fail (text != NULL, -1);

	row = gtk_clist_insert (GTK_CLIST (combo->list), 0, text);
	gtk_clist_set_row_data (GTK_CLIST (combo->list), row, data);
	return row;
}


void
gnome_cmd_combo_set_pixmap (GnomeCmdCombo *combo,
							gint           row,
							gint           col,
							GnomeCmdPixmap *pixmap)
{
	g_return_if_fail (GNOME_CMD_IS_COMBO (combo));
	g_return_if_fail (pixmap != NULL);
	
	gtk_clist_set_pixmap (GTK_CLIST (combo->list), row, col, pixmap->pixmap, pixmap->mask);
	if (pixmap->height > combo->highest_pixmap) {
		gtk_clist_set_row_height (GTK_CLIST (combo->list), pixmap->height);
		combo->highest_pixmap = pixmap->height;
	}
	if (pixmap->width > combo->widest_pixmap) {
		gtk_clist_set_column_width (GTK_CLIST (combo->list), 0, pixmap->width);
		combo->widest_pixmap = pixmap->width;
	}
}


void
gnome_cmd_combo_set_selection (GnomeCmdCombo *combo, const gchar *text)
{
	g_return_if_fail (GNOME_CMD_IS_COMBO (combo));
	g_return_if_fail (text != NULL);

	gtk_entry_set_text (GTK_ENTRY (combo->entry), text);
}


void
gnome_cmd_combo_update_style (GnomeCmdCombo *combo)
{
	gint i;
	
	gtk_widget_set_style (combo->entry, list_style);
	gnome_cmd_clist_update_style (GNOME_CMD_CLIST (combo->list));

	for ( i=0 ; i<GTK_CLIST (combo->list)->columns ; i++ ) {
		GtkWidget *button;
		GtkWidget *child;

		button = GTK_CLIST (combo->list)->column[i].button;
		if (!button) return;

		child = GTK_BIN (button)->child;
		if (!child) return;

		child = GTK_BIN (child)->child;
		if (!child) return;
		
		gtk_widget_modify_style (child, main_rc_style);
	}
}


static void
gnome_cmd_combo_item_selected (GnomeCmdCombo *combo, gpointer data)
{
	g_return_if_fail (GNOME_CMD_IS_COMBO (combo));

	gtk_widget_hide (combo->popwin);
	combo->is_popped = FALSE;
}
