#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# $Id: GDataSource.py 7080 2005-03-02 22:00:12Z reinhard $

"""
Class that implements a provider-independent DataSource object
"""

from gnue.common.apps import i18n, errors
from gnue.common.definitions import GObjects
import sys, string, types, cStringIO
from gnue.common.datasources import GConnections
from gnue.common.formatting import GTypecast
from gnue.common.datasources import GConditions, Exceptions
from gnue.common.definitions.GParserHelpers import GContent
from gnue.common.definitions.GParser import MarkupError


########################################################################
#
#
#
########################################################################

class GDataSource(GObjects.GObj):
  """
  Class that handles DataSources.  This is a subclass of GObj, which
  means this class can be created from XML markup and stored in an
  Object tree (e.g., a Forms tree).
  """
  def __init__(self, parent=None, type="GDataSource"):
    GObjects.GObj.__init__(self, parent, type)
    self.type = "object"
    self.connection = None
    self._connections = None
    self._dataObject = None
    self._connectionComment = ""
    self._fieldReferences = {}
    self._unboundFieldReferences = {}
    self._defaultValues = {}
    self._ignoreDispatchEvent = None

    self._inits =[self.primaryInit, self.secondaryInit, self.tertiaryInit]
    self._currentResultSet = None
    self._resultSetListeners = []
    self._toplevelParent = None # Needs to be set by subclass
                                # so that _topObject gets set
    self._topObject = None
    self.sorting = None

    #
    # trigger support
    #
    self._triggerGlobal = True
    self._triggerFunctions = {'createResultSet':{'function':self.createResultSet,
                                                 },
                              'simpleQuery':{'function':self.triggerSimpleQuery,
                                             },
                              'delete':{'function':self.deleteCurrentRecordsetEntry
                                        },
                              'call':{'function':self.callFuncOfCurrentRecordsetEntry
                                        },
                              'getCondition':{'function':self.getCondition},
                              'setCondition':{'function':self.setCondition},
                              'count' : {'function':self.triggerGetCount},
                              'update': {'function':
                                self.updateCurrentRecordSet}
                              }

    self._triggerProperties = {'extensions':{'get':self.getExtensions,
                                           'direct':1},
                                'recordCount':{'get':self.triggerGetCount,
                                               'direct':1},
                                'order_by':{'get':self.triggerGetOrderBy,
                                            'set':self.triggerSetOrderBy,
                                               'direct':1},
                               }

  def _ignoreDispatchEvent(self, *args, **parms):
    """
    A dummy method that _dispatchEvent is initially set to; does nothing
    """
    pass

  def setEventListener(self, listener):
    self._dispatchEvent = listener
    if self._dataObject:
      self._dataObject._dispatchEvent = listener

  def __getattr__(self, attr):
    if self._dataObject and attr[1] != '_' and hasattr(self._dataObject,attr):
      return getattr(self._dataObject,attr)
    else:
      raise AttributeError, attr

  def _buildObject(self):

    # Added 0.5.0 -- Delete before 1.0
    if hasattr(self,'database'):
      self.connection = self.database
      del self.database

    try:
      if len(self.explicitfields):
        for field in string.split(self.explicitfields,','):
          gDebug (7, "Explicit field %s" % field)
          self._fieldReferences[field] = True
    except AttributeError:
      pass

    self.__getSortOrder ()
    if self.sorting:
      for field in [item ['name'] for item in self.sorting]:
        self._fieldReferences [field] = True

    return GObjects.GObj._buildObject(self)

  def triggerGetOrderBy(self):
    return self.sorting

  def triggerSetOrderBy(self,value):
    self.sorting = self.__convertOrderBy (value)
    if self._dataObject:
      self._dataObject.sorting = self.sorting

  def triggerGetCount(self):
    if self._currentResultSet:
      return len(self._currentResultSet)
    else:
      return 0

  def triggerSimpleQuery(self,maskDict):
    queryDict = {}
    okToProcess = True
    for key in maskDict.keys():
      queryDict[key] = "%s" % maskDict[key]
      if not len(queryDict[key]):
        okToProcess = False
        break

    conditions = GConditions.buildConditionFromDict(queryDict,GConditions.GClike)
    resultSet = self.createResultSet(conditions)
    recordCount = resultSet.getRecordCount()

    returnList = []
    for count in range(recordCount):
      record = resultSet.getRecord(count)
      resultDict = {}
      for key in record._fields.keys():
        resultDict[key]=record.getField(key) or ""
      returnList.append(resultDict)
    return returnList

  def deleteCurrentRecordsetEntry(self):
    self._currentResultSet.getPostingRecordset().delete()

  def callFuncOfCurrentRecordsetEntry(self,name,params):
    n=self._currentResultSet.getRecordNumber()
    rset=self._currentResultSet.getRecord(n)
    if hasattr(rset,'callFunc'):
      return rset.callFunc(name,params)
    else:
      tmsg = u_("Backend doesn't support the trigger 'call' function")
      raise StandardError, tmsg


  # ---------------------------------------------------------------------------
  # Update the current record set
  # ---------------------------------------------------------------------------

  def updateCurrentRecordSet (self):
    """
    If a result set is available having a record all fields of this record
    would be updated. If the backend does not support this operation an
    ApplicationError will be raised.
    """

    if self._currentResultSet is not None:
      nr = self._currentResultSet.getRecordNumber ()
      rs = self._currentResultSet.getRecord (nr)

      if hasattr (rs, 'updateRecordSet'):
        rs.updateRecordSet ()

      else:
        raise errors.ApplicationError, \
            u_("Backend does not support the 'update' function")

  #
  # get/set the static condition assosiated with a datasource
  # the static condition is build out of the <condition> child
  # elements below a datasource XML definition
  #
  def setCondition(self, mycondition):
    self._dataObject._staticCondition = mycondition
#   dataObject.invalidateCachedConditions()

  def getCondition(self):
    return self._dataObject._staticCondition

  #
  # get the dbdriver extension object
  #
  def getExtensions(self):
    return self.extensions

  #
  # This method should be called after the object is created
  # but before any other methods are called
  #
  def setConnectionManager(self, connectionManager):
    self._connections = connectionManager

  def initialize(self):
    if not self.connection:
      # We are a connectionless datasource (virtual?)
      # We have to bind to something, so bind to empty or static driver
      if not self.type=="static":
        from gnue.common.datasources.drivers.special.unbound import Driver
        gDebug (7, 'Using empty data driver')
        dataObject = Driver.supportedDataObjects['object'](None)

      else:
        from gnue.common.datasources.drivers.special.static import Connection
        gDebug (7, 'Using static data driver')
        dataObject = Connection.supportedDataObjects['object'](None)

        for child in self._children:
          if isinstance(child, GStaticSet):
            dataObject._staticSet = child
            break


    elif self._connections:
      self.connection = string.lower(self.connection)
      # This will throw a GConnections.NotFoundError if an unknown
      # connection name is specified.  The calling method should
      # catch this exception and handle it properly (exit w/message)
      dataObject = \
         self._connections.getDataObject(self.connection, self.type)
      gDebug (7, "GDataSource.py bound to %s " % dataObject)

    self.name = string.lower(self.name)
    self._topObject._datasourceDictionary[self.name]=self

    dataObject._fieldReferences = self._fieldReferences
    dataObject._unboundFieldReferences = self._unboundFieldReferences
    dataObject._defaultValues = self._defaultValues
    dataObject._dataSource = self

    # TODO: Short-term hack to allow primary key support
    try:
      dataObject._primaryIdField = self.primarykey
      dataObject._primaryIdFormat = "%s = '%%s'" % self.primarykey
      dataObject._primaryIdChecked = True
    except AttributeError:
      pass

    hasRaw = False
    for child in self._children:
      if isinstance(child, GConditions.GCondition):
        dataObject._staticCondition = child
        break
      elif isinstance(child, GSql):
        dataObject._rawSQL = child
        hasRaw = True


    if self.type == "sql" and not hasRaw:
      raise Exceptions.InvalidDatasourceDefintion, \
        u_("Datasource %s is sql-based, but has no <sql> definition.") \
        % self.name
    elif not self.type == "sql" and hasRaw:
      raise Exceptions.InvalidDatasourceDefintion, \
        u_("Datasource %s is not sql-based, but has a <sql> definition.") \
        % self.name


    # Copy all attributes from XML to the dataObject
    tagAttributes = getXMLelements()['datasource']['Attributes']
    for attribute in tagAttributes.keys():
      if self.__dict__.has_key(attribute):
        dataObject.__dict__[attribute] = self.__dict__[attribute]
      else:
        try:
          dataObject.__dict__[attribute] = tagAttributes[attribute]['Default']
        except KeyError:
          pass
    self._dataObject = dataObject
    self._dataObject.sorting = self.sorting


  def connect(self):
    if self.connection != None:
      self._connections.requestConnection(self._dataObject)


  def getDataObject(self):
    return self._dataObject


  def referenceField(self, field, defaultValue=None):

    gDebug (7, 'Field %s implicitly referenced' % field)
    self._fieldReferences[field] = ""

    if defaultValue != None:
      self._defaultValues[field] = defaultValue

  def referenceFields(self, fields):
    for field in fields:
      if isinstance (field, types.StringType) or \
         isinstance (field, types.UnicodeType):
        self.referenceField(field)
      else:
        self.referenceField(*field)

  def referenceUnboundField(self, field, defaultValue=None):

    gDebug (7,'Unbound Field %s implicitly referenced' % field)
    self._unboundFieldReferences[field] = True

    if defaultValue != None:
      self._defaultValues[field] = defaultValue


  #
  # The following is a simple wrapper around the datasource's dataobject
  # to hide the dataobject from the app programmer
  #
  def hasMaster(self):
    return self._dataObject != None and self._dataObject.hasMaster()

  def createResultSet(self, conditions={}, readOnly=False, sql=""):
      resultSet= self._dataObject.createResultSet(conditions,readOnly,sql=sql)
      self.__setResultSet( resultSet )
      return resultSet

  def addDetailDataObject(self, dataObject, handler=None):
      self._dataObject.addDetailDataObject(dataObject, handler)

  def createEmptyResultSet(self, readOnly=False,masterRecordSet=None):
      resultSet = self._dataObject.createEmptyResultSet(readOnly, masterRecordSet=masterRecordSet)
      self.__setResultSet( resultSet )
      return resultSet

  def getQueryString(self,conditions={},forDetailSQL=None,additionalSQL=""):
    return self._dataObject.getQueryString(conditions, forDetailSQL,additionalSQL)

  #
  # Master/detail stuff
  #

  # Called by dbdrivers whenever this datasource's master has changed
  def masterResultSetChanged(self, masterResultSet, detailResultSet):
    self._masterResultSet = masterResultSet
    self.__setResultSet( detailResultSet )

  def __setResultSet(self, resultSet):
    self._currentResultSet = resultSet
    # Notify all the listeners (i.e., blocks) that the result set changed
    for listener in self._resultSetListeners:
      listener(resultSet)

  def registerResultSetListener(self, listener):
    self._resultSetListeners.append(listener)

  def primaryInit(self):
    self._topObject = self.findParentOfType(self._toplevelParent)
    gDebug (9, "Setting %s to connect mgr %s" \
        % (self.name, self._topObject._connections))
    self.__getSortOrder ()
    self.setConnectionManager(self._topObject._connections)
    self.initialize()
    self.connect()
    self.extensions = self._dataObject.triggerExtensions


  # TODO: Merged into GDataSource per the TODOs in reports and forms however
  # TODO: self._topObject._datasourceDictionary implies that the top object
  # TODO: always has a specifc structure.  This is a bad thing :(  Maybe GRootObj
  # TODO: should contain a getDatasourceDict()?
  #
  def secondaryInit(self):

    if hasattr(self, 'master') and self.master:

      self.master = string.lower(self.master)
      gDebug (7, "Linking detail '%s' to master '%s'" \
                 % (self.name, self.master))

      if self._topObject._datasourceDictionary.has_key(self.master):
        self._topObject._datasourceDictionary[self.master] \
            .getDataObject().addDetailDataObject(self.getDataObject(),
                                                 self)
      else:
        tmsg = u_("Detail source '%(source)s' references non-existant master "
                  "'%(master)s'") \
               % {'source': self.name,
                  'master': self.master}
        raise StandardError, tmsg



  def tertiaryInit(self):
    if hasattr(self, 'prequery'):
      if not self.hasMaster() and self.prequery:
        self.createResultSet()


  # ---------------------------------------------------------------------------
  # Make sure we have either no sort order, or one in the proper format
  # ---------------------------------------------------------------------------

  def __getSortOrder (self):

    # If there is both, an order_by attribute *and* a sorting-tag, we've stop
    child = self.findChildOfType ('GCSortOrder')
    if child is not None and hasattr (self, 'order_by'):
      raise MarkupError, \
          (u_("The use of order_by is depreciated. Please use <sortorder> "
              "instead"), self._url, self._lineNumber)

    # If there's a sorting tag, we'll use this first
    if child is not None:
      self.sorting = []
      for item in child.findChildrenOfType ('GCSortField'):
        self.sorting.append ({'name'      : item.name,
                              'descending': item.descending,
                              'ignorecase': item.ignorecase})

    # otherwise let's investigate the order_by attribute given
    elif hasattr (self, 'order_by'):
      self.sorting = self.__convertOrderBy (self.order_by)
      delattr (self, 'order_by')


  # ---------------------------------------------------------------------------
  # convert an order_by rule into the new 'sorting' format
  # ---------------------------------------------------------------------------

  def __convertOrderBy (self, order_by):
    """
    This function transforms an order_by rule into a proper 'sorting' sequence,
    made of tuples with fieldname and sort-direction.

    @param order_by: string, unicode-string, sequence with sort-order
    @return: sequence of tuples (field, direction)
    """

    result = []

    # If it's a string or a unicode string, we transform it into a tuple
    # sequence, where all items are treated to be in 'ascending' order
    if isinstance (order_by, types.StringType) or \
       isinstance (order_by, types.UnicodeType):
      gDebug (1, "DEPRECIATION WARNING: use of 'order_by' attribute is " \
                 "depreciated. Please use <sortorder> instead.")
      for field in string.split (order_by, ','):
        (item, desc) = (field, field [-5:].lower () == ' desc')
        if desc:
          item = item [:-5].strip ()

        # Since the old 'order_by' format does *not* support 'ignorecase' we
        # will use 'False' as default
        result.append ({'name': item, 'descending': desc})

    # Well, order_by is already a sequence. So we've to make sure it's a
    # sequence of tuples with fieldname and direction.
    elif isinstance (order_by, types.ListType):
      for item in order_by:
        if isinstance (item, types.StringType) or \
           isinstance (item, types.UnicodeType):
          result.append ({'name':item})

        elif isinstance (item, types.DictType):
          result.append (item)

        else:
          element = {'name': item [0]}
          if len (item) > 1: element ['descending'] = item [1]
          if len (item) > 2: element ['ignorecase'] = item [2]

    else:
      raise MarkupError, \
          (u_("Unknown type/format of 'order-by' attribute"), self._url,
            self._lineNumber)

    return result



  # Find a specific record in the resultset by field values
  def findRecord(self, fieldValues):
    self._currentResultSet.findRecord(fieldValues)

  #
  # Hooks for record-level triggers
  #

  def _beforeCommitInsert(self, record):
    return 1

  def _beforeCommitUpdate(self, record):
    return 1

  def _beforeCommitDelete(self, record):
    return 1

  def _onModification(self, record):
    return 1

  def _onRecordLoaded(self, record):
    return 1


  def close (self):

    # If we have a dataObject available, make sure it's reference to the
    # datasource will be cleared, so garbage collection can do it's job
    if self._dataObject is not None:
      self._dataObject._dataSource = None

    self._dataObject = None

    # Make sure we leave no unreachable reference !
    if self._topObject._datasourceDictionary.has_key (self.name):
      del self._topObject._datasourceDictionary

    self._fieldReferences = None
    self._unboundFieldReferences = None
    self._defaultValues = None
    self._ignoreDispatchEvent = None

    self._inits = None
    self._currentResultSet = None
    self._resultSetListeners = None
    self._toplevelParent = None 
    self._topObject = None
    self.sorting = None

    self._triggerFunctions = None
    self._triggerProperties = None


######
#
#
#
######
class GSql(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDSql")

######
#
# Static Datasource Support
#
######
class GStaticSet(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDStaticSet")

class GStaticSetRow(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDStaticSetRow")

class GStaticSetField(GObjects.GObj):
  def __init__(self, parent=None):
     GObjects.GObj.__init__(self, parent, type="GDStaticSetField")


######
#
#
#
######
class GConnection(GObjects.GObj):
  def __init__(self, parent=None):
    GObjects.GObj.__init__(self, parent, "GCConnection")
    self.comment = ""
    self.name = ""
    self._inits =[self.initialize]

  def _buildObject(self):
    self.name = string.lower(self.name)
    return GObjects.GObj._buildObject(self)

  def initialize(self):
    # Add our database connection information to the connections
    # manager, then let it handle everything from there.
    root = self.findParentOfType(None)
    root._instance.connections.\
        addConnectionSpecification(self.name, {
           'name': self.name,
           'provider': self.provider,
           'dbname': self.dbname,
           'host': self.host } )



######
#
# Used by client GParsers to automatically pull supported xml tags
#
######

#
# Return any XML elements associated with
# GDataSources.  Bases is a dictionary of tags
# whose values are update dictionaries.
# For example: bases={'datasource': {'BaseClass':myDataSource}}
# sets xmlElements['datasource']['BaseClass'] = myDataSource
#
def getXMLelements(updates={}):

  xmlElements = {
      'datasource': {
         'BaseClass': GDataSource,
         'Importable': True,
         'Description': 'A datasource provides a link to a database table '
                        'or some similar data store.',
         'Attributes': {
            'name':        {
               'Required': True,
               'Unique':   True,
               'Typecast': GTypecast.name,
               'Description': 'Unique name of the datasource.' },
            'type':        {
               'Label': _('Data Object Type'),
               'Typecast': GTypecast.name,
               'Default':  "object" },
            'connection':    {
               'Label': _('Connection Name'),
               'Typecast': GTypecast.name,
               'Description': 'The name of the connection as in '
                              'connections.conf that points to '
                              'a valid database.' },
            'database':    {
               'Typecast': GTypecast.name,
               'Deprecated': 'Use {connection} attribute instead' },
            'table':       {
               'Label': _('Table Name'),
               'Typecast': GTypecast.name,
               'Description': 'The table in the database this datasource '
                              'will point to.' },
            'cache':       {
               'Label': _('Cache Size'),
               'Description': 'Cache this number of records',
               'Typecast': GTypecast.whole,
               'Default':  5 },
            'prequery':    {
               'Label': _('Query on Startup'),
               'Description': 'If true, the datasource is populated on '
                              'form startup. If false (default), the form '
                              'starts out with an empty record until the user '
                              'or a trigger queries the database.',
               'Typecast': GTypecast.boolean,
               'Default':  False },
            'distinct':    {
               'Typecast': GTypecast.boolean,
               'Default':  False,
               'Description': 'TODO' },
            'order_by':    {
               'Typecast': GTypecast.text ,
               'Deprecated': 'Use {sortorder} tag instead' },
            'master':      {
               'Label': _('M/D Master DataSource'),
               'Description': 'If this datasource is the child in a '
                              'master/detail relationship, this property '
                              'contains the name of the master datasource.',
               'Typecast': GTypecast.name },
            'masterlink':  {
               'Label': _('M/D Master Field(s)'),
               'Description': 'If this datasource is the child in a '
                              'master/detail relationship, this property '
                              'contains a comma-separated list of the '
                              'master datasource\'s field(s) used for '
                              'linking.',
               'Typecast': GTypecast.text },
            'detaillink':  {
               'Label': _('M/D Detail Field(s)'),
               'Description': 'If this datasource is the child in a '
                              'master/detail relationship, this property '
                              'contains a comma-separated list of the '
                              'this (child\'s) datasource\'s field(s) used '
                              'for linking.',
               'Typecast': GTypecast.text },
            # TODO: Short-term hack
            'explicitfields': {
               'Label': _('Explicit Fields'),
               'Typecast': GTypecast.text,
               'Description': 'TODO' },
            'primarykey': {
               'Label': _('Primary Key Field(s)'),
               'Description': 'Comma-separated list of the fields that '
                              'make up the primary key.',
               'Typecast': GTypecast.text },
            'primarykeyseq': {
               'Label': _('Primary Key Sequence'),
               'Description': 'Name of the sequence used to populate a '
                              'primary key (only applies to relational '
                              'backends that support sequences; requires '
                              'a single {primarykey} value.',
               'Typecast': GTypecast.text },
            'requery': {
               'Label': _('Re-query on commit?'),
               'Default': True,
               'Description': 'Requery a record after posting it; requires '
                              '{primarykey} support and a non-null primary '
                              'key value at the time of update (whether '
                              'via a trigger or by the use of '
                              '{primarykeyseq}.',
               'Typecast': GTypecast.text }
               },
         'ParentTags': None },

      'sortorder': {
        'BaseClass': GCSortOrder,
        'Attributes': {},
        'ParentTags': ('datasource',),
      },
      'sortfield': {
        'BaseClass': GCSortField,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Description': 'The name of the field by which the datasource '
                           'will be ordered.',
            'Typecast': GTypecast.name,
            },
          'descending': {
            'Description': 'Selects if the ordering is done in ascending '
                           '(default) or in descending order.',
            'Default' : False,
            'Typecast': GTypecast.boolean,
          },
          'ignorecase': {
            'Default' : False,
            'Typecast': GTypecast.boolean,
            'Description': 'Selects wether the ordering is case-sensitive '
                           'or not.',
          },
        },
        'ParentTags': ('sortorder',),
      },

      'staticset': {
         'BaseClass': GStaticSet,
#  TODO: This should be replaced by a SingleInstanceInParentObject
#        instead of SingleInstance (in the whole file)
#         'SingleInstance': True,
         'Attributes': {
            'fields':        {
               'Typecast': GTypecast.text,
               'Required': True } },
         'ParentTags': ('datasource',) },
      'staticsetrow': {
         'BaseClass': GStaticSetRow,
         'ParentTags': ('staticset',) },
      'staticsetfield': {
         'BaseClass': GStaticSetField,
         'Attributes': {
            'name':        {
               'Typecast': GTypecast.text,
               'Required': True },
            'value':        {
               'Typecast': GTypecast.text,
               'Required': True } },
         'ParentTags': ('staticsetrow',) },
      'sql': {
         'BaseClass': GSql,
         'MixedContent': True,
         'KeepWhitespace': True,
         'ParentTags': ('datasource',) },
      'connection': {
         'BaseClass': GConnection,
         'Attributes': {
            'name': {
               'Required': True,
               'Unique': True,
               'Typecast': GTypecast.name,
               'Description': 'TODO' },
            'provider': {
               'Required': True,
               'Typecast': GTypecast.name,
               'Description': 'TODO' },
            'dbname': {
               'Required': False,
               'Typecast': GTypecast.text,
               'Description': 'TODO' },
            'service': {
               'Required': False,
               'Typecast': GTypecast.text,
               'Description': 'TODO' },
            'comment': {
               'Required': False,
               'Typecast': GTypecast.text,
               'Description': 'TODO' },
            'host': {
               'Required': False,
               'Typecast': GTypecast.text,
               'Description': 'TODO' } },
         'ParentTags': None,
         'Description': 'TODO' },
  }

  # Add conditional elements
  xmlElements.update(
      GConditions.getXMLelements(
          {'condition':{'ParentTags':('datasource',) } } ))

  for alteration in updates.keys():
    xmlElements[alteration].update(updates[alteration])

  # Connections will have the same parent as datasources
  xmlElements['connection']['ParentTags'] = xmlElements['datasource']['ParentTags']

  return xmlElements


# =============================================================================
# Classes for implementing sort options
# =============================================================================

class GCSortOrder (GObjects.GObj):
  def __init__ (self, parent = None):
    GObjects.GObj.__init__ (self, parent, 'GCSortOrder')
    self.sorting = []
    self._inits = [self._build]

  def _build (self):
    for item in self.findChildrenOfType ('GCSortField'):
      self.sorting.append ({'name': item.name,
                            'descending': item.descending,
                            'ignorecase': item.ignorecase})


class GCSortField (GObjects.GObj):
  def __init__ (self, parent = None):
    GObjects.GObj.__init__ (self, parent, 'GCSortField')


#
# Wrapper for standalone DataSources
# (i.e., not in context of a GObj tree)
#
def DataSourceWrapper(connections=None, fields=(), attributes={}, init=True, unicodeMode=False, sql=""):
  source = _DataSourceWrapper()

  if sql:
    s = GSql(source)
    GContent(s,sql)
    attributes['type'] = 'sql'


  if connections:
    source.setConnectionManager(connections)

  if init:
    source.buildAndInitObject(**attributes)
  else:
    source.buildObject(**attributes)

  if fields:
    source.referenceFields(fields)

  return source


class _DataSourceWrapper(GDataSource):
  def __init__(self, *args, **parms):
    GDataSource.__init__(self, *args, **parms)
    self._datasourceDictionary={}
    self._toplevelParent = self._type

  def getIntrospector (self):
    return self._dataObject._connection.introspector

  def getSchemaCreator (self):
    return self._dataObject._connection.getSchemaCreator ()


# =============================================================================
# Load AppServer specific resources
# =============================================================================

class AppServerResourceError (errors.AdminError):
  pass

class InvalidURLError (AppServerResourceError):
  def __init__ (self, url):
    msg = u_("The URL '%s' is not a valid application server resource "
             "locator") % url
    AppServerResourceError.__init__ (self, msg)

class InvalidResourceTypeError (AppServerResourceError):
  def __init__ (self, resourceType):
    msg = u_("Resource type '%s' is not supported") % resourceType
    AppServerResourceError.__init__ (self, msg)

class ResourceNotFoundError (AppServerResourceError):
  def __init__ (self, resType, resName):
    msg = u_("Resource '%(name)s' of type '%(type)s' not found") \
          % {'type': resType,
             'name': resName}
    AppServerResourceError.__init__ (self, msg)

# -----------------------------------------------------------------------------
# Load a resource from appserver and return it as a file-like object
# -----------------------------------------------------------------------------

def getAppserverResource (url, paramDict, connections):
  if url [:12].lower () != 'appserver://':
    raise InvalidURLError, url

  if '?' in url:
    (appUrl, options) = url [12:].split ('?')
  else:
    appUrl  = url [12:]
    options = None

  parts = appUrl.split ('/')
  if len (parts) != 3:
    raise InvalidURLError, url
  (connection, element, elementName) = parts [:3]

  if not len (connection) or not len (element) or not len (elementName):
    raise InvalidURLError, url

  if not element in ['form']:
    raise InvalidResourceTypeError, element

  elementParams = {}

  if options:
    for part in options.split (';'):
      (item, value) = part.split ('=')
      elementParams [item] = value

  debugFileName = None
  if elementParams.has_key ('debug-file'):
    debugFileName = elementParams ['debug-file']
    del elementParams ['debug-file']

  paramDict.update (elementParams)

  attrs = {'name'    : 'dtsClass',
           'database': connection,
           'table'   : 'gnue_class'}
  fieldList = ['gnue_id', 'gnue_name', 'gnue_module']

  dts = DataSourceWrapper (
      connections = connections,
      attributes  = attrs,
      fields      = fieldList,
      unicodeMode = True)

  parts = elementName.split ('_')
  if len (parts) != 2:
    raise ResourceNotFoundError, (element, elementName)

  (moduleName, className) = parts
  mc = GConditions.buildConditionFromDict ({'gnue_name': className,
        'gnue_module.gnue_name': moduleName})

  rs = dts.createResultSet (mc)
  if rs.firstRecord ():
    paramDict ['connection'] = connection

    res = rs.current.callFunc ("gnue_%s" % element, paramDict)

    if debugFileName is not None:
      dfile = open (debugFileName, 'w')
      dfile.write (res.encode ('utf-8'))
      dfile.close ()

    return cStringIO.StringIO (res.encode ('utf-8'))

  else:
    raise ResourceNotFoundError, (element, elementName)

