#!/usr/bin/perl
#
# Squid filters for GoSA
#
# Igor Muratov <migor@altlinux.org>
#

use strict;
use Net::LDAP;
use POSIX qw(strftime);
use DB_File;

my $DEFAULT_URL = "ERR 501 Access denied";
my $DEFAULT_PERIOD = 'month';
my $CACHE_TIME = 10;

my $LDAP_HOST = "localhost";
my $LDAP_PORT = "389";
my $LDAP_BASE = "ou=People,o=infotek";

my %cache;
my %blacklist;
my %quota;
my $quota_format = "L L L";

tie(%blacklist, 'DB_File', 'domains.db', O_RDONLY);
tie(%quota, 'DB_File', 'quota.db', O_RDWR);

my $debug = 0;

$|=1;

sub timestamp
{
	return strftime("%a %b %X clash goSquid[$$]: ", localtime);
}

sub anonBind
{
	my $ldap = Net::LDAP->new( $LDAP_HOST, port => $LDAP_PORT );
	if($ldap)
	{
		my $mesg = $ldap->bind();
		$mesg->code && warn timestamp, "Can't bind to LDAP server: ", $mesg->error, "\n";
		return $ldap;
	}
	else
	{
		warn timestamp, "Can't connect to LDAP server.\n";
		return undef;
	}
}

# Unpack users's data from entry and setup defaults
sub get_entry
{
	my $result = shift;
	my %user;
	my $entry = ($result->entries)[0];
	my ($min,$hour) = (localtime)[1,2];
	my $time = $hour * 60 + $min;

	if($result->count)
	{
		$user{'ts'} = $time + $CACHE_TIME;
		$user{'uid'} = ($entry->get_value('uid'))[0];
		$user{'start'} = ($entry->get_value('gosaProxyWorkingStart'))[0];
		$user{'stop'} = ($entry->get_value('gosaProxyWorkingStop'))[0];
		$user{'flags'} = ($entry->get_value('gosaProxyAcctFlags'))[0];

		$user{'quota'} = ($entry->get_value('gosaProxyQuota'))[0];
		$user{'quota'} *= 1024 if /\d+[Kk]/;
		$user{'quota'} *= 1048576 if /\d+[Mm]/;
		$user{'quota'} *= 1073741824 if /\d+[Gg]/;

		$user{'period'} = ($entry->get_value('gosaProxyQuotaPeriod'))[0] || $DEFAULT_PERIOD;
		$user{'disabled'} = 0;
	} else {
		# Remember unexisted users
		$user{'disabled'} = 1;
	}

	return \%user;
}

# Retrive users's data from LDAP
sub get_user
{
	my $uid = shift;
	my ($min,$hour) = (localtime)[1,2];
	my $time = $hour * 60 + $min;

	# Search user in cache
	if(defined($cache{$uid}) and $cache{$uid}{'ts'} > $time)
	{
		warn timestamp, "User $uid found in cache\n" if $debug;
		return $cache{$uid};
	}

	# User unknown or cache field is expired
	my $ldap = anonBind;
	return undef unless $ldap;
	my $result = $ldap->search( base=>$LDAP_BASE,
		filter=>"(&(objectClass=gosaProxyAccount)(uid=$uid))",
		attrs=>[
			'uid',
			'gosaProxyWorkingStart',
			'gosaProxyWorkingStop',
			'gosaProxyAcctFlags',
			'gosaProxyQuota',
			'gosaProxyQuotaPeriod'
		]
	);
	$result->code && warn timestamp, "Failed to search: ", $result->error;
	$ldap->unbind;

	# Get user's data
	$cache{$uid} = get_entry($result);
	warn timestamp, "Refresh data from LDAP for user $uid\n";

	# Return
	return $cache{$uid};
}

# Check url in our blacklist
sub unwanted_content
{
	my $url = shift;
	my $host = (split(/\//, $url))[2];

	return 1 if exists($blacklist{$host}) and $blacklist{$host} > 0;
	return undef;
}

# Check work time limit
sub work_time
{
	my $user = shift;
	my ($min,$hour) = (localtime)[1,2];
	my $time = $hour * 60 + $min;

	return 1 if $user->{'start'} < $time and $user->{'stop'} > $time;
	return undef;
}

# Check quota
sub quota_exceed
{
	my $user = shift;
	my $uid = $user->{'uid'};
	my ($bytes, $first, $last) = unpack($quota_format, $quota{$uid});
	my $restart_quota = 0;

	# This user newer asks proxy
	return 1 unless $bytes;

	my $time = `date '+%s'` + 0;

	if($user->{'period'} eq 'h')	# hour
	{
		$restart_quota = 1 if $time - $first > 3600;
	}
	elsif($user->{'period'} eq 'd')	# day
	{
		$restart_quota = 1 if $time - $first > 86400;
	}
	elsif($user->{'period'} eq 'w') # week
	{
		$restart_quota = 1 if $time - $first > 604800;
	}
	elsif($user->{'period'} eq 'm')	# month
	{
		$restart_quota = 1 if $time - $first > 2592000;
	}
	elsif($user->{'period'} eq 'y')	# year
	{
		$restart_quota = 1 if $time - $first > 220752000;
	}

	if($restart_quota)
	{
		warn timestamp, "Restart quota for $uid (".$user->{'quota'}."/".$user->{'period'}.")\n";
		$quota{$user->{'uid'}} = "";
		return 1;
	}
	return 1 if $bytes < $user->{'quota'};
	return undef;
}

sub check_access
{
	my ($user,$url) = @_;
	my ($min,$hour) = (localtime)[1,2];
	my $time = $hour * 60 + $min;

	$user->{'timed'} = 0;
	$user->{'quoted'} = 0;
	$user->{'filtered'} = 0;

	if($user->{'flags'} =~ m/N/)
	{
		# Do not filter anything
		return;
	}
	if($user->{'flags'} =~ m/[F]/)
	{
		# Filter unwanted content
		$user->{'filtered'} = 1 if unwanted_content($url);
	}
	if($user->{'flags'} =~ m/[T]/)
	{
		# Filter unwanted content during working hours only
		$user->{'filtered'} = 1 if work_time($user);
	}
	if($user->{'flags'} =~ m/B/)
	{
		$user->{'quoted'} = 1 if quota_exceed($user);
	}
}

while (<>) {
	my ($url, $addr, $ident, $method) = split;
	my ($min,$hour) = (localtime)[1,2];
	my $time = $hour * 60 + $min;

	if(my $user = get_user($ident))
	{
		check_access($user, $url);

		if($user->{'disabled'})
		{
			warn timestamp, "Access denied for unknown user $ident\n";
		}
		elsif($user->{'timed'})
		{
			warn timestamp, "Access denied by worktime for $ident\n";
		}
		elsif($user->{'quoted'})
		{
			warn timestamp, "Access denied by quota for $ident\n";
		}
		elsif($user->{'filtered'})
		{
			warn timestamp, "Content $url filtered for $ident\n";
		}
		else
		{
			print "$url\n";
			next;
		}
	}

	print "$DEFAULT_URL\n";
}

untie %blacklist;
untie %quota;
