#include <sigc++/bind.h>

#include "database-metadata.h"
#include "result-set.h"
#include "util.h"

#include "gql++/mod-result-set.h"

namespace GQL
{

namespace PG
{

using namespace std;

PGDatabaseMetaData::PGDatabaseMetaData(PGConnection *conn)
{
  conn_ = conn;
  conn_->reference();
}

static gchar *table_types[][2] =
{
  { "TABLE", 		"(relkind='r' AND relname !~ '^pg_' AND relname !~ '^xinv')" },
  { "VIEW", 		"(relkind='v' AND relname !~ '^pg_' AND relname !~ '^xinv')" },
  { "INDEX",		"(relkind='i' AND relname !~ '^pg_' AND relname !~ '^xinv')" },
  { "LARGE OBJECT",	"(relkind='r' AND relname ~ '^xinv')" },
  { "SEQUENCE", 	"(relkind='S' AND relname !~ '^pg_')" },
  { "SYSTEM TABLE", 	"(relkind='r' AND relname ~ '^pg_')" },
  { "SYSTEM VIEW", 	"(relkind='r' AND relname ~ '^pg_')" },
  { "SYSTEM INDEX", 	"(relkind='i' AND relname ~ '^pg_')" },
  { NULL,		NULL }
};

ResultSet *PGDatabaseMetaData::get_tables(
        const string& catalog,
        const string& schema_pattern,
        const string& table_name_pattern,
        const vector<string>& types)
{
  PGresult *r, *dr;
  ModResultSet *result;
  bool first;
  GString *type = g_string_new("");
  vector<string>::size_type j;

  string query = 
    "SELECT relname, oid, relkind FROM pg_class "
    "WHERE relname LIKE '" + table_name_pattern + "' AND (";

  first = true;
  for (vector<string>::size_type i = 0; i < types.size(); i++)
  {
    for (j = 0; table_types[j][0]; j++)
      if (types[i] == table_types[j][0])
        break;

    if (table_types[j][0] == NULL) // if we got an unknown type
      continue;

    if (!first)
      query.append(" OR ");
    query.append(table_types[j][1]);
    first = false;
  }
  if (first)
    query.append("TRUE");
  query.append(") ORDER BY relname");

  r = conn_->exec_sql(query);

  result = manage(new ModResultSet(conn_, 10));
  
  for (int i = 0; i < PQntuples(r); i++)
  {
    const char *tuple[10];
    
    tuple[0] = 0;
    tuple[1] = 0;
    
    // Table name
    tuple[2] = getvalue(r, i, 0);

    // Type
    g_string_assign(type, "");
    if (tuple[2] && strncmp(tuple[2], "pg_", 3) == 0)
      g_string_append(type, "SYSTEM ");
    
    const char *kind = PQgetvalue(r, i, 2);
    switch (kind[0])
    {
      case 'r':
        g_string_append(type, "TABLE");
        break;
      case 'v':
        g_string_append(type, "VIEW");
        break;
      case 's':
        g_string_append(type, "SPECIAL RELATION");
        break;
      case 'i':
        g_string_append(type, "INDEX");
        break;
      case 't':
        g_string_append(type, "TOAST");
        break;
      case 'S':
        g_string_append(type, "SEQUENCE");
      default:
        g_string_append(type, "UNKNOWN");
    }
    tuple[3] = type->str;
    
    // Remarks
    query = "SELECT description FROM pg_description WHERE objoid = ";
    query.append(PQgetvalue(r, i, 1));
    
    dr = conn_->exec_sql(query);
    tuple[4] = (PQntuples(dr) > 0) ? getvalue(dr, 0, 0) : 0;

    tuple[5] = 0;
    tuple[6] = 0;
    tuple[7] = 0;
    tuple[8] = 0;
    tuple[9] = 0;

    result->append(tuple);

    PQclear(dr);
  }

  PQclear(r);

  g_string_free(type, TRUE);
  
  return result;
}

ResultSet *PGDatabaseMetaData::get_columns(
        const std::string& catalog,
        const std::string& schema_pattern,
        const std::string& table_name_pattern,
        const std::string& column_name_pattern)
{
  ModResultSet *result = manage(new ModResultSet(conn_, 9));
  PGresult *r;
  char intbuf[32];
  
  string query = 
    "SELECT c.relname, a.attname, format_type(a.atttypid, a.atttypmod),"
    " a.attnotnull, a.atthasdef, a.attnum,"
    " col_description(a.attrelid, a.attnum), c.oid "
    "FROM pg_attribute a, pg_class c "
    "WHERE a.attrelid = c.oid"
    " AND a.attnum > 0"
    " AND a.attname LIKE '" + column_name_pattern + "'"
    " AND c.relname LIKE '" + table_name_pattern + "'";

  r = conn_->exec_sql(query);

  for (int i = 0; i < PQntuples(r); i++)
  {
    const char *tuple[12];
    
    tuple[0] = 0; 	// table catalog
    tuple[1] = 0; 	// table schema
    tuple[2] = getvalue(r, i, 0); // table name
    tuple[3] = getvalue(r, i, 1); // column name
    tuple[4] = getvalue(r, i, 2); // data type
    
    // nullability - I think PostgreSQL doesn't have unknown here
    sprintf(intbuf, "%d", (strcmp(PQgetvalue(r, i, 3), "t") == 0) ? 
            ColumnNoNulls : ColumnNullable);
    tuple[5] = intbuf;
    
    tuple[6] = 0; 	// remarks

    // default value
    if (strcmp(PQgetvalue(r, i, 4), "t") == 0)
    { 
      query = 
        "SELECT adsrc FROM pg_attrdef ad "
        "WHERE ad.adrelid = " + string(PQgetvalue(r, i, 7)) + 
        " AND ad.adnum = " + PQgetvalue(r, i, 5);
      PGresult *r1 = conn_->exec_sql(query);
      tuple[7] = (PQntuples(r1) >= 1) ? PQgetvalue(r1, 0, 0) : 0;
      PQclear(r1);
    }
    else
      tuple[7] = 0;

    // column index
    tuple[8] = getvalue(r, i, 5);
    if (tuple[8])
    {
      int idx = atoi(tuple[8]);
      idx--;
      sprintf(intbuf, "%d", i);
      tuple[8] = intbuf;
    }
    
    result->append(tuple);
  }
  PQclear(r);
  
  return result;
}

ResultSet *PGDatabaseMetaData::get_primary_keys(const std::string& catalog,
                                                const std::string& schema,
                                                const std::string& table)
{
  string query =
    "SELECT "
    "'' AS TABLE_CAT,"
    "'' AS TABLE_SCHEM,"
    "bc.relname AS TABLE_NAME,"
    "a.attname AS COLUMN_NAME,"
    "a.attnum as KEY_SEQ,"
    "ic.relname as PK_NAME "
    " FROM pg_class bc, pg_class ic, pg_index i, pg_attribute a"
    " WHERE bc.relkind = 'r' "        //        -- not indices
    "  AND upper(bc.relname) = upper('" + table + "')"
    "  AND i.indrelid = bc.oid"
    "  AND i.indexrelid = ic.oid"
    "  AND ic.oid = a.attrelid"
    "  AND i.indisprimary='t' "
    " ORDER BY table_name, pk_name, key_seq";

    return manage(new PGResultSet(conn_, conn_->exec_sql(query)));
}

ResultSet *PGDatabaseMetaData::get_xrefs(const string& primary_table,
                                         const string& foreign_table)
{
  const int tuplesz = 14;
  ModResultSet *result = manage(new ModResultSet(conn_, tuplesz));

  // The code for this method is adapted from PostgreSQL jdbc :-0
  string query =
    "SELECT DISTINCT"
    " c.relname as prelname,"
    " c2.relname as frelname,"
    " t.tgconstrname,"
    " a.attnum as keyseq,"
    " ic.relname as fkeyname,"
    " t.tgdeferrable,"
    " t.tginitdeferred,"
    " t.tgnargs,t.tgargs,"
    " p1.proname as updaterule,"
    " p2.proname as deleterule "
    "FROM "
    " pg_trigger t, pg_trigger t1,"
    " pg_class c, pg_class c2,"
    " pg_class ic, "
    " pg_proc p1, pg_proc p2,"
    "pg_index i, pg_attribute a "
    "WHERE "
    // isolate the update rule
    "(t.tgrelid=c.oid "
    "AND t.tgisconstraint "
    "AND t.tgconstrrelid=c2.oid "
    "AND t.tgfoid=p1.oid "
    "AND p1.proname like '%upd') "

    "AND "
    // isolate the delete rule
    "(t1.tgrelid=c.oid "
    "AND t1.tgisconstraint "
    "AND t1.tgconstrrelid=c2.oid "
    "AND t1.tgfoid=p2.oid "
    "AND p2.proname like '%del') " +

    // if we are looking for exported keys then primary table will be used
    ((primary_table != "") ? 
     "AND c.relname='" + primary_table + "' " : "") +

    // if we are looking for imported keys then the foreign table will be used
    ((foreign_table != "") ? 
     "AND c2.relname='" + foreign_table + "' " : "") +
    
    "AND i.indrelid=c.oid "
    "AND i.indexrelid=ic.oid "
    "AND ic.oid=a.attrelid "
    "AND i.indisprimary "
    "ORDER BY " +
    // orderby is as follows getExported, orders by FKTABLE,
    // getImported orders by PKTABLE
    // getCrossReference orders by FKTABLE, so this should work for both,
    // since when getting crossreference, primaryTable will be defined
    ((primary_table != "null") ? "frelname" : "prelname") + ",keyseq";
  
  // returns the following columns
  // and some example data with a table defined as follows
  
  // create table people ( id int primary key);
  // create table policy ( id int primary key);
  // create table users  ( id int primary key, 
  // 			   people_id int references people(id), 
  //                       policy_id int references policy(id));
  
  // prelname | frelname | tgconstrname | keyseq | fkeyName	 | 
  //     1    |    2     |      3       |   4    |    5		 |
  //  people  |  users   |  <unnamed>   |   1    | people_pkey   |

  // tgdeferrable | tginitdeferred | tgnargs | tgargs | updaterule | deleterule
  //       6      |	 7	   |    8    |	  9   |    10      |	  11
  //       f	  |	 f         |	6    | <unnamed>\000users\000people\000UNSPECIFIED\000people_id\000id\000 | RI_FKey_noaction_upd | RI_FKey_noaction_del

  PGresult *r = conn_->exec_sql(query);
  for (int i = 0; i < PQntuples(r); i++)
  {
    string *tuple[14];
    char intbuf[8];
    
    tuple[0] = 0; // ptable catalog
    tuple[1] = 0; // ptable schema
    tuple[2] = get_new_string(r, i, 0); // ptable name
    tuple[4] = 0; // ftable catalog
    tuple[5] = 0; // ftable schema
    tuple[6] = get_new_string(r, i, 1); // ftable name

    // get update rule
    string updaterule = PQgetvalue(r, i, 9);
    int updateval = ImportedKeyNoAction;
    
    if (updaterule.length() > 12)
    {
      // Rules look like this RI_FKey_noaction_del so we want to pull
      // out the part between the 'Key_' and the last '_' s
      string rule = updaterule.substr(8, updaterule.length() - 12);
      
      if (rule == "cascade") 		updateval = ImportedKeyCascade;
      else if (rule == "setnull") 	updateval = ImportedKeySetNull;
      else if (rule == "setdefault") 	updateval = ImportedKeySetDefault;
      else if (rule == "restrict") 	updateval = ImportedKeyRestrict;
    }
    sprintf(intbuf, "%d", updateval);
    tuple[9] = new string(intbuf);
    
    // get delete rule
    string deleterule = PQgetvalue(r, i, 9);
    int deleteval = ImportedKeyNoAction;
    
    if (deleterule.length() > 12)
    {
      // Rules look like this RI_FKey_noaction_del so we want to pull
      // out the part between the 'Key_' and the last '_' s
      string rule = deleterule.substr(8, rule.length() - 12);
      
      if (rule == "cascade") 		deleteval = ImportedKeyCascade;
      else if (rule == "setnull") 	deleteval = ImportedKeySetNull;
      else if (rule == "setdefault") 	deleteval = ImportedKeySetDefault;
      else if (rule == "restrict") 	deleteval = ImportedKeyRestrict;
    }
    sprintf(intbuf, "%d", deleteval);
    tuple[10] = new string(intbuf);

    int seqno = 0;
    tuple[8] = new string("0"); // sequence numer within foreign key
    tuple[11] = get_new_string(r, i, 8); // fkey name. targs will give
                                         // us a unique name for it
    tuple[12] = get_new_string(r, i, 4); // pkey name
    
    // deferrability
    ImportedKeyDeferrability deferrability = ImportedKeyNotDeferrable;
    bool deferrable = (strcmp(PQgetvalue(r, i, 5), "t") == 0);
    bool initially_deferred = (strcmp(PQgetvalue(r, i, 6), "t") == 0);
    if (deferrable)
    {
      if (initially_deferred)
        deferrability = ImportedKeyInitiallyDeferred;
      else
        deferrability = ImportedKeyInitiallyImmediate;
    }
    sprintf(intbuf, "%d", deferrability);
    tuple[13] = new string(intbuf);

    tuple[3] = 0;
    tuple[7] = 0;
    
    // Parse the tgargs data
    {
      // Note, I am guessing at most of this, but it should be close
      // if not, please correct.
      // the keys are in pairs and start after the first four arguments
      // the arguments are seperated by \000
      const gchar *targs = PQgetvalue(r, i, 8);
      
      // args look like this
      //<unnamed>\000ww\000vv\000UNSPECIFIED\000m\000a\000n\000b\000 
      // we are primarily interested in the column names which are the last
      // items in the string
      gchar **values = g_strsplit(targs, "\\000", 0);
      for (int j = 0; values[j]; j++)
      {
        if (j >= 4 && j % 2 == 0)
          tuple[7] = new string(values[j]);
        else if (j >= 5 && j % 2 == 1)
        {
          tuple[3] = new string(values[j]);
          if (values[j + 1] && values[j + 1][0] != '\0')
          {
            // we got another pair, so we append the tuple we have and
            // duplicate it, leaving idx 3 and 7 alone and increasing
            // sequence number
            result->append(tuple);
            tuple[3] = 0;
            tuple[7] = 0;
            tuple[8] = 0;
            for (int k = 0; k < tuplesz; k++)
              tuple[k] = tuple[k] ? new string(*tuple[k]) : 0;
            seqno++;
            sprintf(intbuf, "%d", seqno);
            tuple[8] = new string(intbuf);
          }
          else
            break;
        }
      }
      g_strfreev(values);
    }

    result->append(tuple);
  }
  
  PQclear(r);

  return result;
}

ResultSet *PGDatabaseMetaData::get_index_info(const string& cat,
                                              const string& schema,
                                              const string& table,
                                              bool unique,
                                              bool approximate)
{
  char intbuf[32];
  const int tuplesz = 13;
  ModResultSet *result = manage(new ModResultSet(conn_, tuplesz));
  ModResultSetMetaData *rsmd = manage(new ModResultSetMetaData(tuplesz));
  string *tuple[tuplesz];
  
  rsmd->set_column_info(0, SQLType(SQLType::VARCHAR, -1), "TABLE_CAT");
  rsmd->set_column_info(1, SQLType(SQLType::VARCHAR, -1), "TABLE_SCHEM");
  rsmd->set_column_info(2, SQLType(SQLType::VARCHAR, -1), "TABLE_NAME");
  rsmd->set_column_info(3, SQLType(SQLType::BOOL), "NON_UNIQUE");
  rsmd->set_column_info(4, SQLType(SQLType::VARCHAR, -1), "INDEX_QUALIFIER");
  rsmd->set_column_info(5, SQLType(SQLType::VARCHAR, -1), "INDEX_NAME");
  rsmd->set_column_info(6, SQLType(SQLType::SMALLINT), "TYPE");
  rsmd->set_column_info(7, SQLType(SQLType::SMALLINT), "ORDINAL_POSITION");
  rsmd->set_column_info(8, SQLType(SQLType::VARCHAR, -1), "COLUMN_INFO");
  rsmd->set_column_info(9, SQLType(SQLType::VARCHAR, -1), "ASC_OR_DESC");
  rsmd->set_column_info(10, SQLType(SQLType::INT), "CARDINALITY");
  rsmd->set_column_info(11, SQLType(SQLType::INT), "PAGES");
  rsmd->set_column_info(12, SQLType(SQLType::VARCHAR, -1), "FILTER_CONDITION");

  result->set_meta_data(rsmd);
  
  string query = 
    "SELECT c.relname, x.indisunique, i.relname, x.indisclustered,"
    " a.amname, x.indkey, c.reltuples, c.relpages, x.indexrelid "
    "FROM pg_index x, pg_class c, pg_class i, pg_am a "
    "WHERE ((c.relname = '" + table + "') "
    " AND (c.oid = x.indrelid) "
    " AND (i.oid = x.indexrelid) "
    " AND (i.relam = a.oid)) " + 
    (unique ? "AND x.indisunique " : "") + 
    "ORDER BY x.indisunique DESC, "
    " x.indisclustered, a.amname, i.relname";
  
  PGresult *r = conn_->exec_sql(query);
  for (int row = 0; row < PQntuples(r); row++)
  {
    // indkey is an array of column ordinals (integers). we must split
    // it over the returned rows
    list<string> column_ordinals;
    const char *last_cp = PQgetvalue(r, row, 5);
    for (const char *cp = last_cp; ; ++cp)
      if (*cp == ' ' || *cp == '\0')
      {
        column_ordinals.push_back(string(last_cp, cp - last_cp));
        if (*cp == '\0')
          break;
        last_cp = cp;
      }
    
    if (column_ordinals.size() == 0)
      continue;
    query = 
      string("SELECT attname FROM pg_attribute WHERE attrelid = ") +
      PQgetvalue(r, row, 8);
    PGresult *cname_r = conn_->exec_sql(query);
    for (int i = 0; i < (int)column_ordinals.size(); i++)
    {
      tuple[0] = 0;
      tuple[1] = 0;
      tuple[2] = get_new_string(r, row, 0);
      tuple[3] = get_new_string(r, row, 1);
      tuple[4] = 0;
      tuple[5] = get_new_string(r, row, 2);
      int index_type = (strcmp(PQgetvalue(r, row, 3), "t") == 0) ?
        TableIndexClustered : (strcmp(PQgetvalue(r, row, 4), "hash") == 0) ?
        TableIndexHashed : TableIndexOther;
      sprintf(intbuf, "%d", index_type);
      tuple[6] = new string(intbuf);
      sprintf(intbuf, "%d", i + 1);
      tuple[7] = new string(intbuf);
      if (i < PQntuples(cname_r))
        tuple[8] = get_new_string(cname_r, i, 0);
      else
        tuple[8] = new string;
      tuple[9] = 0;
      tuple[10] = get_new_string(r, row, 6);        // inexact
      tuple[11] = get_new_string(r, row, 7);
      tuple[12] = 0;
      
      result->append(tuple);
    }
    PQclear(cname_r);
  }
  PQclear(r);

  return result;
}

}

}
