/*
 * Groach properties module.
 * Refer to properties.h about details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include <applet-widget.h>

#include "groach.h"

#include "types.h"
#include "amain.h"
#include "properties.h"
#include "theme-search.h"
#include "preview.h"


/* Data structure definitions */
struct _GroachPropPrivate {
	/* Variable GUI parts */
	GnomePropertyBox *pb;
	GtkEntry *entry;		/* Used to get the selected theme name */
	GtkLabel *description;
	GtkLabel *author;

	/* These two work as cache */
	GList *theme_enum;		/* Singly linked list of theme names */
	GHashTable *theme_ht;	/* Hash table of GroTheme, key is theme name */
};

/* Private function declarations */
static void prop_apply(GtkWidget *widget, gint page_num, GroachData *groach_data);
static void update_labels(GroachProp *groach_prop, const GroTheme *cur_theme);
static void name_entry_changed(GtkEntry *entry, GroachData *groach_data);
static void hash_free_key_value(gpointer key, gpointer value, gpointer data);

static void preview_click_cb(GtkButton *button, GroachData *groach_data);
static void preview_destroy_cb(Preview *preview, GroController *controller);


GroachProp*
properties_new(void)
{
	GroachProp *groach_prop;
	GroachPropPrivate *privat;

	groach_prop = g_new(GroachProp, 1);
	groach_prop->theme_name = NULL;

	privat = g_new(GroachPropPrivate, 1);
	groach_prop->privat = privat;
	privat->theme_enum = NULL;
	privat->theme_ht = g_hash_table_new(g_str_hash, g_str_equal);

	return groach_prop;
}

void
properties_delete(GroachProp *groach_prop)
{
	GroachPropPrivate *privat;

	g_return_if_fail(groach_prop != NULL);

	privat = groach_prop->privat;
	if (privat->theme_ht) {
		g_hash_table_foreach(privat->theme_ht, hash_free_key_value, NULL);
		g_hash_table_destroy(privat->theme_ht);
	}
	if (privat->theme_enum) {
		GList *list;
		for (list = privat->theme_enum; list; list = list->next)
			g_free(list->data);
		g_list_free(privat->theme_enum);
	}
	g_free(privat);

	if (groach_prop->theme_name)
		g_free(groach_prop->theme_name);
	
	g_free(groach_prop);
}

void
properties_load(GroachProp *groach_prop, const char *path)
{
	gnome_config_push_prefix(path);
	if (groach_prop->theme_name) {
		g_warning("groach_prop->theme_name should be null.\n");
		g_free(groach_prop->theme_name);
	}
	groach_prop->theme_name = gnome_config_get_string_with_default(APPLET_NAME"/"THEME_NAME_KEY, NULL);
	gnome_config_pop_prefix();
}

void
properties_save(GroachProp *groach_prop, const char *path)
{
	gnome_config_push_prefix(path);
	gnome_config_set_string(APPLET_NAME"/"THEME_NAME_KEY, groach_prop->theme_name);
	gnome_config_pop_prefix();
}


void
property_cb(AppletWidget *widget, gpointer data)
{
	static GtkWidget *prop_dialog = NULL;
	GroachData *groach_data = data;
	GroachProp *groach_prop = groach_data->groach_prop;
	GroachPropPrivate *privat = groach_prop->privat;
	GroTheme *cur_theme;
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *table;
	GtkWidget *combo;
	GtkWidget *preview_button;

	/* Takes care of internal data GroachProp, which mostly works as cache */
	if (privat->theme_enum == NULL) {
		privat->theme_enum = theme_search_enum_themes(groach_data->theme_search);
	}
	
	if (prop_dialog) {
		gdk_window_raise(GTK_WIDGET(prop_dialog)->window);
		return;
	}
	
	prop_dialog = gnome_property_box_new();
	privat->pb = GNOME_PROPERTY_BOX(prop_dialog);
	gtk_window_set_title(GTK_WINDOW(prop_dialog), _("Groach properties"));

	cur_theme = groach_data->controller->cur_theme;

	vbox = gtk_vbox_new(FALSE, 0);
	label = gtk_label_new(_("Theme"));
	gnome_property_box_append_page(GNOME_PROPERTY_BOX(prop_dialog),
								   vbox, label);
	
	table = gtk_table_new(3, 2, FALSE);

	label = gtk_label_new(_("Name"));
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
					 GTK_FILL, GTK_FILL, 8, 0);
	combo = gtk_combo_new();
	privat->entry = GTK_ENTRY(GTK_COMBO(combo)->entry);
	gtk_entry_set_editable(privat->entry, FALSE);
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), privat->theme_enum);
	gtk_entry_set_text(privat->entry, cur_theme->name);
	gtk_signal_connect(GTK_OBJECT(privat->entry), "changed",
			   GTK_SIGNAL_FUNC(name_entry_changed), groach_data);
	gtk_table_attach_defaults(GTK_TABLE(table), combo, 1, 2, 0, 1);
	
	label = gtk_label_new(_("Description"));
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
					 GTK_FILL, GTK_FILL, 8, 0);
	label = gtk_label_new(NULL);
	privat->description = GTK_LABEL(label);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 1, 2);
	
	label = gtk_label_new(_("Author"));
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
					 GTK_FILL, GTK_FILL, 8, 0);
	label = gtk_label_new(NULL);
	privat->author = GTK_LABEL(label);
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 2, 3);

	update_labels(groach_prop, cur_theme);

	gtk_box_pack_start(GTK_BOX(vbox), table, TRUE, TRUE, 0);

	preview_button = gtk_button_new_with_label(_("Preview"));
    gtk_signal_connect(GTK_OBJECT(preview_button), "clicked",
					   GTK_SIGNAL_FUNC(preview_click_cb), groach_data);
	gtk_box_pack_end(GTK_BOX(vbox), preview_button, FALSE, FALSE, 0);
	
    gtk_signal_connect(GTK_OBJECT(prop_dialog), "apply",
					   GTK_SIGNAL_FUNC(prop_apply), groach_data);
	gtk_signal_connect(GTK_OBJECT(prop_dialog), "destroy",
					   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &prop_dialog);

	gtk_widget_show_all(prop_dialog);
}


/* ---The followings are private functions--- */
static void
prop_apply(GtkWidget *widget, gint page_num, GroachData *groach_data)
{
	GroachProp *groach_prop = groach_data->groach_prop;
	GroachPropPrivate *privat = groach_prop->privat;
	GroTheme *cur_theme;
	const char *theme_name;/* GtkEntry's text */

	cur_theme = groach_data->controller->cur_theme;

	theme_name = gtk_entry_get_text(privat->entry);
	if (strcmp(theme_name, cur_theme->name) != 0) {
		/* theme changed */
		GroTheme *new_theme;
		char *new_theme_dirname;
		int cur_num_gmoves;

		cur_num_gmoves = g_list_length(groach_data->controller->gmove_list);
		new_theme_dirname = theme_search_find_path(groach_data->theme_search, theme_name);
		new_theme = gro_theme_load(new_theme_dirname, THEME_LOAD_NORMAL);
		g_free(new_theme_dirname);
		groach_prop->theme_name = g_strdup(theme_name);

		gro_controller_detach_theme(groach_data->controller);
		gro_controller_attach_theme(groach_data->controller, new_theme);
		gro_controller_add_gmoves(groach_data->controller, cur_num_gmoves);
	}
}

static void
update_labels(GroachProp *groach_prop, const GroTheme *cur_theme)
{
	GroachPropPrivate *privat = groach_prop->privat;
	
	gtk_label_set_text(privat->description, cur_theme->description);
	gtk_label_set_text(privat->author, cur_theme->author);
}

static void
name_entry_changed(GtkEntry *entry, GroachData *groach_data)
{
	GroachProp *groach_prop = groach_data->groach_prop;
	GroachPropPrivate *privat = groach_prop->privat;
	GroTheme *theme;
	const char *theme_name;/* GtkEntry's text */

	/* first, look for the name in hash table */
	theme_name = gtk_entry_get_text(entry);
	theme = g_hash_table_lookup(privat->theme_ht, theme_name);
	if (theme == NULL) {	/* not found */
		char *theme_dirname;
		theme_dirname = theme_search_find_path(groach_data->theme_search, theme_name);
		theme = gro_theme_load(theme_dirname, THEME_LOAD_ONLYINFO);
		gro_theme_ref(theme);/* own it */
		g_hash_table_insert(privat->theme_ht, g_strdup(theme_name), theme);
		g_free(theme_dirname);
	}

	update_labels(groach_prop, theme);
	gnome_property_box_changed(privat->pb);
}

/**
 * hash_free_key_value:
 * Free key and value of GHashTable *theme_ht.
 **/
static void
hash_free_key_value(gpointer key, gpointer value, gpointer data)
{
	g_free(key);	/* free theme name */
	gro_theme_unref(value);/* leave it */
}


/**
 * preview_click_cb:
 * Create a preview window, and start preview.
 **/
static void
preview_click_cb(GtkButton *button, GroachData *groach_data)
{
	GroachProp *groach_prop = groach_data->groach_prop;
	GroachPropPrivate *privat = groach_prop->privat;
	GtkWidget *preview;
	GroController *controller;
	GroTheme *theme;
	const char *theme_name;/* GtkEntry's text */
	char *theme_dirname;

	preview = preview_new();
	gtk_widget_show(preview);

	theme_name = gtk_entry_get_text(privat->entry);
	theme_dirname = theme_search_find_path(groach_data->theme_search, theme_name);
	theme = gro_theme_load(theme_dirname, THEME_LOAD_PREVIEW);
	g_free(theme_dirname);

	controller = gro_controller_new(preview_get_gdkwin(preview),
									preview_get_widget(preview));
	gro_controller_attach_theme(controller, theme);
	gro_controller_add_gmoves(controller, 10);
	gro_controller_run(controller);

	gtk_signal_connect(GTK_OBJECT(preview), "destroy",
					   GTK_SIGNAL_FUNC(preview_destroy_cb), controller);
}

static void
preview_destroy_cb(Preview *preview, GroController *controller)
{
	if (gro_controller_is_running(controller) == TRUE) {
		gro_controller_stop(controller);
	}
	gro_controller_detach_theme(controller);
	gro_controller_delete(controller);
}
