#ifndef MUTEX_H
#define MUTEX_H

/*
 * Encapsulated pthread mutex and condition
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <pthread.h>
//#include <stdio.h>

/*
 * encapsulated pthread mutex
 */
class Mutex
{
protected:
	pthread_mutex_t mutex;
	
public:
	Mutex() throw () { pthread_mutex_init(&mutex, 0); }
	~Mutex() throw () { pthread_mutex_destroy(&mutex); }

	/// Lock the mutex
	/// Normally it's better to use MutexLock
	void lock() throw () { pthread_mutex_lock(&mutex); }

	/// Unlock the mutex
	/// Normally it's better to use MutexLock
	void unlock() throw () { pthread_mutex_unlock(&mutex); }

	/// Reinitialize the mutex
	void reinit() throw () { pthread_mutex_init(&mutex, 0); }

	// Let Condition methods access mutex
	friend class Condition;
};

/*
 * pthread mutex lock which is taken by construction and
 * released by deconstruction
 */
class MutexLock
{
protected:
	Mutex& mutex;
	
public:
	MutexLock(Mutex& m) throw () : mutex(m)
	{
		//fprintf(stderr, "LOCK %p\n", &mutex);
		mutex.lock();
	}
	~MutexLock() throw ()
	{
		//fprintf(stderr, "UNLOCK %p\n", &mutex);
		mutex.unlock();
	}

	// Let Condition methods access mutex
	friend class Condition;
};

/*
 * pthread condition which is taken by construction and
 * released by deconstruction
 *
 * Must be used in association with a MutexLock, of course
 */
class Condition
{
protected:
	pthread_cond_t cond;

public:
	Condition() throw ()
	{
		pthread_cond_init(&cond, 0);
	}
	~Condition() throw () { pthread_cond_destroy(&cond); }

	/// Wake up one process waiting on the condition
	void signal() throw () { pthread_cond_signal(&cond); }

	/// Wake up all processes waiting on the condition
	void broadcast() throw () { pthread_cond_broadcast(&cond); }

	/// Wait on the condition, locking with l [for limited time abstime]
	/// (Unlock l before waiting and lock it before returning)
	void wait(MutexLock& l) throw ()
	{
		pthread_cond_wait(&cond, &l.mutex.mutex);
	}
	void wait(MutexLock& l, const struct timespec& abstime) throw ()
	{
		pthread_cond_timedwait(&cond, &l.mutex.mutex, &abstime);
	}
};

#if 0
class Thread
{
protected:
	pthread_t thread;

	/// Short tag describing this thread
	/// This is used in error messages and identification
	virtual const char* threadTag() { return "generic"; }
	
	/// Main thread function
	/// This is executed in the new thread after creation.
	/// When main() exits, the new thread ends and main() result
	/// will be the thread exit result.
	virtual void* main() throw () = 0;

	/// Callback function used to start the thread
	static void* Starter(void* parm) throw ();

public:
	Thread() throw () {}
	virtual ~Thread() throw () {}

	/// Start the thread
	void start() throw (SystemException);

	/// Start the thread in the detached state
	void startDetached() throw (SystemException);

	/// Join the thread
	void* join() throw (SystemException);

	/// Put the thread in the detached state
	void detach() throw (SystemException);

	/// Send a cancellation request to the thread
	void cancel() throw (SystemException);
};
#endif

// vim:set ts=4 sw=4:
#endif
