module Data.Bits
  ( Bits( (.&.), (.|.), xor  -- :: a -> a -> a
        , complement         -- :: a -> a
        , shift              -- :: a -> Int -> a
        , rotate             -- :: a -> Int -> a
        , bit                -- :: Int -> a
        , setBit             -- :: a -> Int -> a
        , clearBit           -- :: a -> Int -> a
        , complementBit      -- :: a -> Int -> a
        , testBit            -- :: a -> Int -> Bool
        , bitSize            -- :: a -> Int
        , isSigned           -- :: a -> Bool

        , shiftL, shiftR     -- :: Bits a => a -> Int -> a
        , rotateL, rotateR   -- :: Bits a => a -> Int -> a
        )
  ) where

import qualified TraceOrigData.Bits

infixl 8 `shift`, `rotate`, `shiftL`, `shiftR`, `rotateL`, `rotateR`
infixl 7 .&.
infixl 6 `xor`
infixl 5 .|.

class Num a => Bits a where
    (.&.)         :: a -> a -> a
    (.|.)         :: a -> a -> a
    xor           :: a -> a -> a
    complement    :: a -> a

    shift         :: a -> Int -> a
    x `shift` i   | i<0  = x `shiftR` (-i)
                  | i==0 = x
                  | i>0  = x `shiftL` i

    rotate        :: a -> Int -> a

    bit           :: Int -> a
    setBit        :: a -> Int -> a
    clearBit      :: a -> Int -> a
    complementBit :: a -> Int -> a
    testBit       :: a -> Int -> Bool
    bitSize       :: a -> Int
    isSigned      :: a -> Bool

    bit i               = 1 `shiftL` i
    x `setBit` i        = x .|. bit i
    x `clearBit` i      = x .&. complement (bit i)
    x `complementBit` i = x `xor` bit i
    x `testBit` i       = (x .&. bit i) /= 0

    shiftL, shiftR   :: a -> Int -> a
    rotateL, rotateR :: a -> Int -> a
    x `shiftL`  i = x `shift`  i
    x `shiftR`  i = x `shift`  (-i)
    x `rotateL` i = x `rotate` i
    x `rotateR` i = x `rotate` (-i)

foreign import haskell "Data.Bits..&."        _andInt    :: Int -> Int -> Int
foreign import haskell "Data.Bits..|."        _orInt     :: Int -> Int -> Int
foreign import haskell "Data.Bits.xor"        _xorInt    :: Int -> Int -> Int
foreign import haskell "Data.Bits.complement" _notInt    :: Int -> Int
foreign import haskell "Data.Bits.shift"      _shiftInt  :: Int -> Int -> Int
foreign import haskell "Data.Bits.rotate"     _rotateInt :: Int -> Int -> Int
foreign import haskell "Data.Bits.bitSize"    _sizeInt   :: Int -> Int

instance Bits Int where
  (.&.)      = _andInt
  (.|.)      = _orInt
  xor        = _xorInt
  complement = _notInt
  shift      = _shiftInt
  rotate     = _rotateInt
  bitSize    = _sizeInt
  isSigned _ = True

foreign import haskell "Data.Bits..&."
    _andInteger    :: Integer -> Integer -> Integer
foreign import haskell "Data.Bits..|."
    _orInteger     :: Integer -> Integer -> Integer
foreign import haskell "Data.Bits.xor"
    _xorInteger    :: Integer -> Integer -> Integer
foreign import haskell "Data.Bits.complement"
    _notInteger    :: Integer -> Integer
foreign import haskell "Data.Bits.shift"
    _shiftInteger  :: Integer -> Int -> Integer
foreign import haskell "Data.Bits.rotate"
    _rotateInteger :: Integer -> Int -> Integer
foreign import haskell "Data.Bits.bitSize"
    _sizeInteger   :: Integer -> Int

instance Bits Integer where
  (.&.)      = _andInteger
  (.|.)      = _orInteger
  xor        = _xorInteger
  complement = _notInteger
  shift      = _shiftInteger
  rotate     = _rotateInteger
  bitSize _  = error "Bits.bitSize(Integer)"
  isSigned _ = True

