// -*- C++ -*-

#ifndef _ALIGNEDMMAP_H_
#define _ALIGNEDMMAP_H_

#if defined(_WIN32)
#error "You should not include this header on Windows."
#endif

#include <map>

#include "sassert.h"
#include "stlallocator.h"
#include "freelistheap.h"
#include "zoneheap.h"

using namespace std;
using namespace HL;

/*

  We create two versions of AlignedMmap:
  one is for the case where the superheap provides the desired alignment,
  and the other is for when we have to align memory ourselves.

  The Helper class below distinguishes between these two cases through
  the magic of template meta-programming.

*/

template <bool SameAlignment,
	  size_t Alignment_,
	  class SuperHeap>
class AlignedMmapHelper;


template <size_t Alignment_,
	  class SuperHeap>
class AlignedMmapHelper<true, Alignment_, SuperHeap> : public SuperHeap {
public:

  enum { Alignment = Alignment_ };

  inline void * malloc (size_t sz) {
    void * ptr = SuperHeap::malloc (sz);
    assert (reinterpret_cast<size_t>(ptr) % Alignment == 0);
    return ptr;
  }

  inline void free (void * ptr) {
    assert (reinterpret_cast<size_t>(ptr) % Alignment == 0);
    SuperHeap::free (ptr);
  }
};


template <size_t Alignment_,
	  class SuperHeap>
class AlignedMmapHelper<false, Alignment_, SuperHeap> : public SuperHeap {
public:

  enum { Alignment = Alignment_ };

  inline void * malloc (size_t sz) {
    char * ptr = reinterpret_cast<char *>(SuperHeap::malloc (sz + Alignment));
    char * newptr = align (ptr);
    MyMap[newptr] = pair<void *, size_t> (ptr, sz);
    assert (reinterpret_cast<size_t>(newptr) % Alignment == 0);
    return newptr;
  }

  inline void free (void * ptr) {
    typename mapType::iterator i;
    i = MyMap.find ((const void *) ptr);
    pair<void *, size_t> pr = (*i).second;
    void * originalPtr = pr.first;
    SuperHeap::free (originalPtr, pr.second);
    MyMap.erase (i);
  }

private:

  struct MyLess2 {
    bool operator()(const void * a, const void * b) const {
      return (size_t) a < (size_t) b;
    }
  };

  typedef pair<const void *, pair<void *, size_t> > objType;

  class SourceHeap : public FreelistHeap<ZoneHeap<SuperHeap, 16384> > {};

  template <class Foo>
  class MyAllocator2 : public STLAllocator<Foo, SourceHeap> {};

  // Finally, here's the map, with our custom allocator in place.
  typedef map<const void *, pair<void *, size_t>, MyLess2, MyAllocator2<objType> > mapType;

  mapType MyMap;

  inline static char * align (char * buf) {
    return reinterpret_cast<char *>((reinterpret_cast<size_t>(buf) + (Alignment-1)) & ~(Alignment-1));
  }
 
};


template <size_t Alignment_,
	  class SuperHeap>
class AlignedMmap : 
  public AlignedMmapHelper<(SuperHeap::Alignment % Alignment_ == 0),
			   Alignment_, SuperHeap> {};

#endif
