/***************************************************************************
 *   Copyright (C) 2004, 2005 Thomas Nagy                                  *
 *   tnagy2^8@yahoo.fr                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qdir.h>
#include <qpixmap.h>
#include <qstring.h>
#include <qregexp.h>

#include <kmessagebox.h>
#include <kdebug.h>
#include <kstandarddirs.h>
#include <karchive.h>
#include <ktar.h>
#include <klocale.h>
#include <kio/job.h>
#include <kio/netaccess.h>

#include "aux.h"
#include "datastruct/DDataControl.h"
#include "datastruct/DDataItem.h"
#include "settings.h"
#include "DGenerator.h"

DGenerator::DGenerator() : QObject()
{
	m_data = NULL;
}

DGenerator::~DGenerator() { }

const QString DGenerator::identifikation()
{
	return "Error";
}

const QString DGenerator::fullName()
{
	return "Error";
}

const QString DGenerator::group()
{
	return "NoGroup";
}

const QString DGenerator::description()
{
	return "Invalid Template : Generator";
}

const QString DGenerator::quickstart()
{
	return i18n("Nothing appropriate");
}

void DGenerator::writeItem(DDataItem* item, QTextStream & /*s*/, int level)
{
	// kdWarning()<<"DGenerator::writeItem on level"<<level<<"with item"<<item<<endl;
	if (!item)
	{
		kdWarning()<<"BUG : DGenerator::writeItem (1)"<<endl;
		return;
	}

	if (level != 0 && item->Parent() == DItem::NOITEM)
	{
		kdWarning()<<"BUG : DGenerator::writeItem (2)"<<endl;
		return;
	}
}

void DGenerator::generate(const QString &str, DDataControl* data)
{
	// kdWarning()<<"kdissert is generating a document.. "<<str<<endl;
	if (!data)
	{
		// kdWarning()<<"missing data"<<endl;
		return;
	}
	m_data = data;

	QDir dir;
	if (! dir.mkdir(str))
	{
		// kdWarning()<<"could not make dir"<<endl;
		return;
	}

	// Write the makefile, etc
	if (! copyfiles(str) )
	{
		QString text = i18n( "Unable to copy the template files, this should not happen" );
		KMessageBox::sorry( NULL, text );

		// At this point kdissert can safely remove the directory
		KIO::del(str);
		return;
	}

	// Write the pictures
	writePics(data, str);

	// Write the main document
	DDataItem *root = (DDataItem*) data->Item( data->rootID() );
	if (root)
	{
		if ( ! writeMaindoc(root, str))
		{
			QString text = i18n( "An error occurred in the document generation, please look in the directory %1" );
			KMessageBox::sorry( NULL, text.arg(str) );
		}
	}
	else
	{
		QString text = i18n( "No root found, kdissert is unable to produce the main document" );
		KMessageBox::sorry( NULL, text );
		KIO::del(str);	
	}
}

void DGenerator::writePics(DDataControl * data, QString path)
{
	KIO::NetAccess::dircopy( KURL(data->getTmpDir()->absPath()), KURL(path+"/pics/"), NULL );
}

bool DGenerator::writeMaindoc(DDataItem * /*root*/, const QString & /*path*/)
{
	kdWarning()<<"BUG: using DGenerator::writeMaindoc"<<endl;
	return false;
}

bool DGenerator::copyfiles(const QString & destdir)
{
	KTar arch(locate("data", QString("kdissert/templatedata/")+identifikation()+".tar.gz"));
	if (! arch.open(IO_ReadOnly) )
	{
		kdWarning()<<"BUG - missing template data :"<<endl;
		kdWarning()<<locate("data", QString("kdissert/templatedata/")+identifikation()+".tar.gz")<<endl;
		return false;
	}

	const KArchiveDirectory *archiveDir = arch.directory();
	QStringList lst = archiveDir->entries();

	const KArchiveEntry* entry = NULL;
	for (unsigned int i=0; i<lst.count(); i++)
	{
		entry = archiveDir->entry(lst[i]);
		if (entry->isFile())
		{
			KArchiveFile* file = (KArchiveFile*) entry;
			file->copyTo(destdir);
		}
		else if (entry->isDirectory())
		{
			KArchiveDirectory* dir = (KArchiveDirectory*) entry;
			dir->copyTo(destdir+"/"+dir->name(), true);
			//kdWarning()<<"writing directory to "<<destdir+"/"+dir->name()<<endl;
		}
	}

	return true;
}

QString DGenerator::protectTex(const QString & str)
{
	QString ret = str;

	ret.replace( "\\", "$\\backslash$");
	ret.replace( "$", "\\$");

	ret.replace( "&", "\\&");
	ret.replace( "#", "\\#");
	ret.replace( "_", "\\_");
	ret.replace( "%", "\\%");

	ret.replace( "{", "$\\lbrace$" );
	ret.replace( "}", "$\\rbrace$");
	ret.replace( "^", "\\^{}");
	ret.replace( "~", "\\~{}" );

	//////////
	// proceed with the richtext stuff (<b>, <p> ..)

	// wipe out any <html> tag

	ret.replace( "<html>", QString::null);
	ret.replace( "</html>", QString::null);
	ret.replace( "</span>", QString::null);
	ret.replace( "</body>", QString::null);

	ret.replace( "<br />", QString::null);

	QRegExp rbody("<body.*><p>");
	rbody.setMinimal(true);
	ret.replace(rbody, QString::null);

	QRegExp rspan("<span.*>");
	rspan.setMinimal(true);
	ret.replace(rspan, QString::null);

	QRegExp rx("<head.*head>");
	rx.setMinimal(true);
	ret.replace( rx, QString::null);

	// process the intesting tags    
	ret.replace( "<b>", "\\textbf{");
	ret.replace( "</b>", "}" );
	ret.replace( "<p>", "\\linebreak\n" );
	ret.replace( "</p>", QString::null);

	// and remove all other tags
	QRegExp rem("<.*>");
	rem.setMinimal(true);
	ret.replace( rem, QString::null);

	// comment the rest
	ret.replace( "<", "$<$");
	ret.replace( ">", "$>$");

	// TODO : it is better to just ignore these for the moment
	ret.replace( "[", " " );
	ret.replace( "]", " " );

	return ret;
}

QString DGenerator::commentTex(const QString & str)
{
	QString ret = str;

	ret.prepend( "% ");
	ret.replace( "\n", "\n% ");
	ret.append( "\n" );

	return ret;
}

QString DGenerator::htmlToTex(const QString &str)
{
	QString thing = str;

	thing.replace("</span>", "SPAN");
	thing.replace("<span", "SPAN<span");
	QStringList lst = QStringList::split("SPAN", thing);

	QString ret;
	QRegExp boldrx("font-weight:600.*>");
	QRegExp italicrx("italic.*>");
	QRegExp underlinerx("underline.*>");
	QRegExp toremove("<span.*>");

	for (int i=0; i<lst.size(); i++)
	{
		QString txt = lst[i];

		if (txt.startsWith("<span"))
		{
			bool bold = false;
			bool italic = false;
			bool underline = false;

			if (txt.contains(boldrx))
				bold = true;
			if (txt.contains(italicrx))
				italic = true;
			if (txt.contains(underlinerx))
				underline = true;


			if (bold)
				ret += "\\textbf{";
			if (italic)
				ret += "\\textit{";
			if (underline)
				ret += "\\underline{";

			txt.remove(toremove);
			ret += protectTex( txt );

			if (bold)
				ret += "}";
			if (italic)
				ret += "}";
			if (underline)
				ret += "}";
		}
		else	
		{
			ret += protectTex( wipeTags(txt) );
		}
	}
	return ret;
}

void DGenerator::notifyDocGenerated( const QString& url )
{
	emit documentGenerated( url, true );
}

QString wipeTags(const QString &str)
{
	QString ret = str;

	// add newlines
	ret.replace("</p><p>", "\n" );
	ret.replace("<br />", "\n");

	// remove all tags
	QRegExp rx("<.*>");
	rx.setMinimal(true);
	ret.replace( rx, QString::null );

	// replace characters by normal ones
	ret.replace("&quot;", "\"");
	ret.replace("&amp;", "&");
	ret.replace("&lt;", "<");
	ret.replace("&gt;", ">");

	return ret;
}

#include "DGenerator.moc"
