// =============================================================================
//
//      --- kvi_irc_server.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviIrcServer"

#include <qfile.h>

#include "kvi_config.h"
#include "kvi_debug.h"
#include "kvi_irc_network.h"
#include "kvi_irc_server.h"
#include "kvi_settings.h"

KviIrcServerManager::KviIrcServerManager()
{
	m_pNetList = new QPtrList<KviIrcNetwork>;
	m_pNetList->setAutoDelete(true);
	m_pCurrentNet = 0;
}

KviIrcServerManager::~KviIrcServerManager()
{
	clear();
	if( m_pNetList ) {
		delete m_pNetList;
		m_pNetList = 0;
	}
}

void KviIrcServerManager::clear()
{
	while( !m_pNetList->isEmpty() )
		m_pNetList->removeFirst();
	m_pCurrentNet = 0;
}

void KviIrcServerManager::save(KviConfig *cfg)
{
	__range_valid(cfg);
	cfg->setGroup("IrcServerManager");
	cfg->writeEntry("Networks", m_pNetList->count());
	for( uint i = 0; i < m_pNetList->count(); i++ ) {
		KviIrcNetwork *net = m_pNetList->at(i);
		__range_valid(net);
		KviStr szNet(KviStr::Format, "Net_%d", i);
		cfg->setGroup(szNet.ptr());
		net->save(cfg);
		if( net == m_pCurrentNet ) {
			cfg->setGroup("IrcServerManager");
			cfg->writeEntry("Current_Net", i);
		}
	}
	cfg->sync();
}

void KviIrcServerManager::load(KviConfig *cfg)
{
	clear();
	__range_valid(cfg);
	cfg->setGroup("IrcServerManager");
	uint count = cfg->readUIntEntry("Networks",    0);
	uint curr  = cfg->readUIntEntry("Current_Net", 0);
	m_pCurrentNet = 0;
	KviIrcNetwork *curNet = 0;
	for( uint i = 0; i < count; i++ ) {
		KviIrcNetwork *net = new KviIrcNetwork();
		__range_valid(net);
		KviStr szNet(KviStr::Format, "Net_%d", i);
		cfg->setGroup(szNet.ptr());
		net->load(cfg);
		insertNetwork(net);
		if( i == curr ) curNet = net;
	}
	if( curNet )
		m_pCurrentNet = curNet;
	else {
		if( m_pNetList->isEmpty() )
			m_pCurrentNet = 0;
		else
			m_pCurrentNet = m_pNetList->first();
	}
}

KviIrcNetwork *KviIrcServerManager::getCurrentNetwork()
{
	if( m_pNetList->isEmpty() ) return 0;
	if( m_pCurrentNet ) {
		__range_valid(m_pNetList->findRef(m_pCurrentNet) != -1);
		return m_pCurrentNet;
	}
	m_pCurrentNet = m_pNetList->first();
	return m_pCurrentNet;
}

QPtrList<KviIrcNetwork> *KviIrcServerManager::networkList()
{
	return m_pNetList;
}

void KviIrcServerManager::insertNetwork(KviIrcNetwork *net)
{
	__range_valid(net);
	m_pCurrentNet = net;
	if( m_pNetList->isEmpty() )
		m_pNetList->append(net);
	else {
		uint index = 0;
		for( KviIrcNetwork *n = m_pNetList->first(); n; n = m_pNetList->next() ) {
			if( qstricmp(net->name(), n->name()) < 0 ) {
				m_pNetList->insert(index, net);
				return;
			}
			index++;
		}
		m_pNetList->append(net);
	}
}

KviIrcNetwork *KviIrcServerManager::getNetworkByName(const char *szName)
{
	__range_valid(szName);
	for( KviIrcNetwork *n = m_pNetList->first(); n; n = m_pNetList->next() ) {
		if( kvi_strEqualCI(szName, n->name()) ) {
			return n;
		}
	}
	return 0;
}

bool KviIrcServerManager::setCurrentNetwork(KviIrcNetwork *net)
{
	if( m_pNetList->isEmpty() ) {
		m_pCurrentNet = 0;
		return false;
	}
	if( m_pNetList->findRef(net) != -1 ) {
		m_pCurrentNet = net;
		return true;
	}
	if( !m_pCurrentNet )
		m_pCurrentNet = m_pNetList->first();
	return false;
}

bool KviIrcServerManager::setCurrentNetwork(const char *szName)
{
	KviIrcNetwork *net = getNetworkByName(szName);
	if( !net ) return false;
	return setCurrentNetwork(net);
}

bool KviIrcServerManager::setCurrentServer(KviIrcServer *srv)
{
	KviIrcNetwork *net = getCurrentNetwork();
	if( !net )return false;
	return net->setCurrentServer(srv);
}

void KviIrcServerManager::iWantThisServerToBeCurrent(const char *szName, const char *szPort, bool bIPv6)
{
	__range_valid(szName);
	for( KviIrcNetwork *n = m_pNetList->first(); n; n = m_pNetList->next() ) {
		for( KviIrcServer *s = n->m_pServerList->first(); s; s = n->m_pServerList->next() ) {
			if( kvi_strEqualCI(s->szHost.ptr(), szName) ) {
				if( szPort ) {
					if( *szPort ) s->szPort = szPort;
				}
				if( !s->szPort.isUnsignedNum() ) s->szPort = "6667";
				setCurrentNetwork(n);
				n->setCurrentServer(s);
				return;
			}
		}
	}
	KviIrcServer *srv = new KviIrcServer;
	srv->szHost = szName;
	if( szPort )
	{
		if( *szPort ) srv->szPort = szPort;
	}
	if( !srv->szPort.isUnsignedNum() ) srv->szPort = "6667";
#ifdef COMPILE_NEED_IPV6
	srv->bIPv6 = bIPv6;
#endif
	insertNewServer(srv, "[UNKNOWNNET]");
	setCurrentNetwork("[UNKNOWNNET]");
	setCurrentServer(srv);
}

void KviIrcServerManager::setCurrentServerPassword(const char *pass)
{
	KviIrcServer *s = getCurrentServer();
	if( s ) s->szPassword = pass;
}

bool KviIrcServerManager::removeNetwork(const char *szName)
{
	__range_valid(szName);
	KviIrcNetwork *net = getNetworkByName(szName);
	if( !net ) return false;
	return removeNetwork(net);
}

bool KviIrcServerManager::removeNetwork(KviIrcNetwork *net)
{
	__range_valid(net);
	bool bWasHere = m_pNetList->removeRef(net);
	__range_valid(bWasHere);
	if( bWasHere ) {
		if( net == m_pCurrentNet ) {
			if( m_pNetList->isEmpty() )
				m_pCurrentNet = 0;
			else
				m_pCurrentNet = m_pNetList->first();
		}
		return true;
	}
	return false;
}

void KviIrcServerManager::copyFrom(KviIrcServerManager *ptr)
{
	__range_valid(ptr);
	clear();
	KviIrcNetwork *curNet = 0;
	for( KviIrcNetwork *n = ptr->m_pNetList->first(); n; n = ptr->m_pNetList->next() ) {
		// Avoid copying empty networks
		if( n->serverCount() > 0 ) {
			KviIrcNetwork *net = new KviIrcNetwork(n);
			if( n == ptr->m_pCurrentNet ) curNet = net;
			insertNetwork(net);
		}
	}
	if( curNet )
		m_pCurrentNet = curNet;
	else {
		if( m_pNetList->isEmpty() )
			m_pCurrentNet = 0;
		else
			m_pCurrentNet = m_pNetList->first();
	}
#ifdef _KVI_DEBUG_CHECK_RANGE_
	// Sanity check
	if( m_pCurrentNet ) {
		if( m_pCurrentNet->m_pCurrentServer && ptr->m_pCurrentNet->m_pCurrentServer ) {
			__range_valid(m_pCurrentNet->m_pCurrentServer->szHost == ptr->m_pCurrentNet->m_pCurrentServer->szHost);
		} else debug(_i18n_("WARNING: current network has no active server"));
	}
#endif
}

KviIrcServer *KviIrcServerManager::getCurrentServer()
{
	KviIrcNetwork *net = getCurrentNetwork();
	if( !net ) return 0;
	return net->currentServer();
}

KviIrcServer *KviIrcServerManager::getNextServer()
{
	KviIrcNetwork *net = getCurrentNetwork();
	if( !net ) return 0;
	return net->nextServer();
}

// ============================ updateServerIp ===============================
void KviIrcServerManager::updateServerIp(const char *szNet, const char *szServer, KviIpAddresses list)
{
	// Cache the IPs
	KviIrcNetwork *net = getNetworkByName(szNet);
	if( !net ) return;
	net->updateServerIp(szServer, list);
}

// ============================ importFromIni ================================

bool KviIrcServerManager::importFromIni(const char *filename)
{
	// Try to load the mIrc native format servers.ini
	//
	// [servers]
	// nXXX=DescriptionSERVER:server.name:port_listGROUP:group_name
	// ...

	QFile serversIni(filename);
	if( !serversIni.exists() ) return false;
	// KviConfig opens in read-only mode if we not write entries or call sync()
	KviConfig cfg(filename);
	cfg.setGroup("servers");

	int curidx = 0;
	KviStr szKey(KviStr::Format, "n%d", curidx);
	KviStr szNetwork, szDescription, szServer, szPorts, szPort;

	// Assume that the entries are numbered correctly...

	while( cfg.hasKey(szKey.ptr()) ) {
		// Read the entry
		KviStr szEntry = cfg.readEntry(szKey.ptr());
		// Look for the description part
		int idx = szEntry.findFirstIdx("SERVER:");
		if( idx > -1 ) {
			// OK... have a description...
			szDescription = szEntry.left(idx);
			szEntry.cutLeft(idx + 7);
			// Now the server name
			szEntry.getToken(szServer, ':');
			szServer.stripWhiteSpace();
			if( szServer.isEmpty() ) szServer = "irc.unknown.net";
			// Defaults
			szPorts   = "6667";
			szNetwork = "UnknownNet";
			// Now look for the ports
			idx = szEntry.findFirstIdx("GROUP:");
			if( idx > -1 ) {
				szPorts = szEntry.left(idx);
				// Only one port allowed
				szPorts.getToken(szPort, ',');
				szEntry.cutLeft(idx + 6);
				// And the network name
				if( szEntry.hasData() ) szNetwork = szEntry;
			}
			KviIrcServer  *lpE = new KviIrcServer();
			lpE->szHost        = szServer;
			lpE->szDescription = szDescription;
			lpE->szPassword    = "";
			lpE->szPort        = szPort.isUnsignedNum() ? szPort.ptr() : "6667";
#ifdef COMPILE_NEED_IPV6
			lpE->bIPv6 = false;
#endif
			insertNewServer(lpE, szNetwork.ptr());
		}
		curidx++;
		szKey.sprintf("n%d", curidx);
	}
	return true;
}

// ==================== insertNewServer =======================

void KviIrcServerManager::insertNewServer(KviIrcServer *srv, const char *szNetName)
{
	// Insert a new server to the list
	// If the network szNetName already exists,
	// add the server to that network.
	// Otherwise create the net szNetName and then insert.
	KviIrcNetwork *net = getNetworkByName(szNetName);
	if( !net ) {
		// Create a new one
		net = new KviIrcNetwork(szNetName);
		insertNetwork(net);
	}
	net->appendServer(srv);
}
