/*
 * This file contains the signal handler which cleanly exists
 * when an appropriate signal is received.
 */

#include "ledd.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>


static void signal_sig(int sig);
static void signal_abort(int sig);
static void signal_exit(void);
static options *sigopts;

/*
 * Sets the exit signal handler for a few common signals.
 */
void signal_set_handler(options *opts) {
	/* Save a local copy of options for handler. */
	sigopts=opts;

	atexit(signal_exit);
#ifdef SIGINT
	signal(SIGINT,signal_sig);
#endif
#ifdef SIGTERM
	signal(SIGTERM,signal_sig);
#endif
#ifdef SIGABRT
	signal(SIGABRT,signal_sig);
#endif
	return;
}

/*
 * Sets abort signal to exit silently with error code 1.
 */
void signal_ignore_abort(void) {
#ifdef SIGABRT
	signal(SIGABRT,signal_abort);
#endif
}


/*
 * Simply exits quietly with error code 1. Used before the real signal
 * handler is placed.
 */
static void signal_abort(int sig) {
	exit(1);
}



/*
 * The actual signal handler. This handles the exit of the program nicely
 * closing all descriptors and killing all children.
 */
static void signal_sig(int sig) {
	static gint doneit=0;
	GSList *list;
	File *p;

	/* Don't do recursion */
	if (doneit)
		return;
	doneit=1;

	/* Report exit. */
	if (sig==-1)
		g_message("exiting cleanly on exit()");
	else
		g_message("exiting cleanly on signal %d (%s)",
			  sig,g_strsignal(sig));

	/* Startup script */
	for (list=sigopts->startup; list; list=g_slist_next(list)) {
		if (list->data==NULL)
			continue;
		p=list->data;
		if (p->pid > 0)
			kill(p->pid,SIGTERM);
		/* wait() not neccessary, as ledd is exiting anyway. */
	}

	/* File handles */
	pipe_close_all(sigopts);

	/* LEDs */
	led_set_all_normal(sigopts);

	/* pidfile */
	startup_pidfile_remove(sigopts);

	/* Memory - how cares? It's freed anyway. */

	/* Logging off... */
	log_close(sigopts->logging);

	exit(0);
}


/*
 * atexit() handler. Only calls sig_handler(-1).
 */
static void signal_exit(void) {
	signal_sig(-1);
	return;
}

