package Lire::Param;

use strict;

use Lire::Config::TypeSpec;
use Lire::DataTypes qw/ check_xml_name check_type /;
use Lire::Utils qw/ check_param /;
use Lire::I18N qw/ dgettext_para /;


=pod

=head1 NAME

Lire::Param - Object which represents a parameter in a XML specification.

=head1 DESCRIPTION

Lire::Param are objects which represent parameters in an XML
specification. The object is used to represent the parameter's
specification as well as its current value.


=head2 new( 'name' => $name, 'type' => $type, [ 'default' => $default ] )

Creates a new Lire::Param object.

=cut

sub new {
    my ( $class, %params ) = @_;

    check_param( $params{'name'}, 'name', \&check_xml_name );
    check_param( $params{'type'}, 'type', \&check_type );

    my $self = bless { 'name'	=> $params{'name'},
                       'type'	=> $params{'type'},
                       'i18n_domain' => $params{'i18n_domain'} || 'lire',
                       'description' => $params{'description'},
                     }, $class;

    $self->default( $params{'default'} )
      if defined $params{'default'};
    $self->value( $params{'value'} )
      if defined $params{'value'};

    return $self;
}

=pod

=head2 name()

Returns the name of this parameter.

=cut

sub name {
    return $_[0]->{'name'};
}

=pod

=head2 type()

Returns this parameter's type.

=cut

sub type {
    return $_[0]->{'type'};
}

=pod

=head2 value( [ $new_value ] )

Returns (and optionnally modifies) the current value for this
parameter. If no value was set, but a default is available, the
default value will be returned.

=cut

sub value {
    my ($self, $value ) = @_;

    if ( @_ == 2 ) {
	if ( defined $value ) {
            check_param( $value, 'value',
                         $Lire::DataTypes::VALIDATORS{$self->{'type'}} );
	    $self->{'value'} = $value;
	} else {
	    $self->{'value'} = undef;
	}
    }

    # Check for default
    if ( defined $self->{'value'}) {
	return $self->{'value'};
    } else {
	return $self->default();
    }
}

=pod

=head2 default( [ $new_default ] )

Returns (and optionnally changes) the parameter's default value.

=cut

sub default {
    my ( $self, $default ) = @_;

    if ( defined $default ) {
        check_param( $default, 'default', 
                     $Lire::DataTypes::VALIDATORS{$self->{'type'}} );
	$self->{'default'} = $default;
    }

    return $self->{'default'};
}

=pod

=head2 description( [ $new_description ] )

Returns (and optionnally changes) the current parameter's description.

=cut

sub description {
    my ( $self, $desc ) = @_;

    if ( @_ == 2 ) {
	$self->{'description'} = $desc;
    }

    return dgettext_para( $self->{'i18n_domain'},
                          $self->{'description'} );
}

=pod

=head2 as_type_spec()

Returns a Lire::Config::TypeSpec object which adequately represents the
current parameter.

=cut

my %types2spec = ( 'int' => 'Lire::Config::IntegerSpec',
                   'bool' => 'Lire::Config::BooleanSpec',
                   'filename', => 'Lire::Config::FileSpec' );
sub as_type_spec {
    my $self = $_[0];

    my $type = $types2spec{$self->{'type'}} || 'Lire::Config::StringSpec';
    my $spec = $type->new( 'name' => $self->{'name'},
                           'i18n_domain' => $self->{'i18n_domain'},
                           'description' => $self->{'description'} );

    $spec->default( $spec->instance( 'value' => $self->{'default'} ) )
      if defined $self->{'default'};
    return $spec;
}

# keep perl happy

1;

__END__

=pod

=head1 SEE ALSO

  Lire::Config::TypeSpec(3pm), Lire::XMLSpecContainer(3pm)

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Param.pm,v 1.12 2004/07/02 16:13:23 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2001,2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
