import gc
import os
import time

import gtk
import pango

import EXIF

import MetaData
import MetaDataDialog
import MetaDataEditor
import Filesel
import Alert

import _lodju


def rotate_pixbuf(src, angle):
    assert angle in [0, 90, 180, 270]
    
    if angle == 0:
    	return src.copy()

    sw = src.get_width()
    sh = src.get_height()
    if angle == 180:
	dw = sw
	dh = sh
    else:
	dw = sh
	dh = sw
    dest = gtk.gdk.Pixbuf(gtk.gdk.COLORSPACE_RGB,
			  src.get_has_alpha(),
			  src.get_bits_per_sample(),
			  dw, dh)
    
    if angle == 90:
    	_lodju.rotate90(src, dest)
    elif angle == 180:
    	_lodju.rotate180(src, dest)
    elif angle == 270:
    	_lodju.rotate270(src, dest)

    return dest


class Thumbnail:

    THUMB_SIZE = 300

    interesting_exif = [
	("EXIF ExposureTime", u"exif:exposure-time"),
	("EXIF Flash", u"exif:flash"),
	("EXIF FocalLength", u"exif:focal-length"),
	("EXIF ISOSpeedRatings", u"exif:iso-speed"),
	("EXIF DateTimeOriginal", u"exif:date-time"),
    ]

    def __init__(self, doc, photo, draw_hook):
    	self.doc = doc
    	self.photo = photo
	self.photo.listen(self.rotate_thumb)
	self.draw_hook = draw_hook
    	self.x = 0
	self.y = 0
	self.width = 0
	self.height = 0
	self.file = None
	self.loader = None
	self.pixbuf_thumb_orig = None
	self.pixbuf_orig = None
	self.pixbuf_delivered = None
	self.pixbuf_delivered_size = 0
	self.pixbuf_delivered_angle = 0

    def load_exif(self):
	try:
	    f = open(self.photo[u"imported-from"], "rb")
	    data = EXIF.process_file(f)
	    f.close()
	    for exifkey, photokey in self.interesting_exif:
		if data.has_key(exifkey):
		    self.photo[photokey] = unicode(data[exifkey])
	except IOError:
	    pass

    def rotate_thumb(self, photo, key):
    	if key == u"angle" and self.pixbuf_thumb_orig and self.draw_hook:
	    self.draw_hook(self)

    def scale_pixbuf(self, src, max_dimension):
	w = float(src.get_width())
	h = float(src.get_height())
	if w < h:
	    sh = max_dimension
	    sw = sh * (w/h)
	else:
	    sw = max_dimension
	    sh = sw * (h/w)
	return src.scale_simple(int(sw), int(sh), gtk.gdk.INTERP_BILINEAR)

    def set_orig_pixbuf(self, orig_pixbuf):
    	self.pixbuf_orig = orig_pixbuf
	if orig_pixbuf:
	    self.pixbuf_delivered = None
	    self.draw_hook(self)

    def get_thumb_pixbuf(self, max_dimension):
    	if self.pixbuf_delivered:
	    if self.pixbuf_delivered_size == max_dimension:
	    	if self.pixbuf_delivered_angle == self.photo.get_angle():
		    return self.pixbuf_delivered

    	if self.pixbuf_orig:
	    pixbuf = self.pixbuf_orig
	elif not self.pixbuf_thumb_orig:
	    filename = \
	    	self.doc.storage.get_thumbnail_filename(self.photo[u"id"])
    	    if not filename:
		return None
    	    self.pixbuf_thumb_orig = gtk.gdk.pixbuf_new_from_file(filename)
	    if not self.pixbuf_thumb_orig:
	    	return None
    	    pixbuf = self.pixbuf_thumb_orig
    	else:
    	    pixbuf = self.pixbuf_thumb_orig

    	pixbuf = self.scale_pixbuf(pixbuf, max_dimension)

    	angle = self.photo.get_angle()
	if angle != 0:
	    pixbuf = rotate_pixbuf(pixbuf, angle)

	self.pixbuf_delivered = pixbuf
	self.pixbuf_delivered_size = max_dimension
	self.pixbuf_delivered_angle = angle
	return self.pixbuf_delivered

    def copy(self):
    	return Thumbnail(self.doc, self.photo, None)

    def get_position(self):
    	return self.x, self.y

    def set_position(self, x, y, width, height):
    	self.x = x
	self.y = y
	self.width = width
	self.height = height

    def contains(self, x, y):
    	return x >= self.x and x < self.x + self.width and \
	       y >= self.y and y < self.y + self.height


class LoadOriginalTask:

    def __init__(self, doc, thumb):
    	self.thumb = thumb
	self.file = doc.storage.get_original(thumb.photo[u"id"])
	self.loader = gtk.gdk.PixbufLoader("jpeg")
	gtk.idle_add(self.work)

    def is_loading(self, thumb):
    	return self.thumb == thumb

    def work(self, *args):
    	if not self.file:
	    return gtk.FALSE
	data = self.file.read(4096)
	if data:
	    self.loader.write(data, len(data))
    	    return gtk.TRUE
	else:
	    self.file.close()
	    self.file = None
	    self.loader.close()
	    self.thumb.set_orig_pixbuf(self.loader.get_pixbuf())
	    self.loader = None
	    return gtk.FALSE

    def stop(self):
    	self.file = None
	if self.loader:
	    self.loader.close()
	    self.loader = None


class ImportTask:

    def __init__(self, view):
    	self.view = view
    	self.unimported = []
	self.total_files = 0
	self.input = None
	self.output = None
	self.loader = None

    def add(self, doc, thumb):
    	self.unimported.append((doc, thumb))
	self.total_files = self.total_files + 1
	self.view.set_status_text("Importing %d photographs" % 
	    	    	    	  self.total_files)
    	self.show_progress()

    def show_progress(self):
    	done = self.total_files - len(self.unimported)
    	self.view.set_progress(float(done) / self.total_files)
	
    def work(self, *args):
    	doc, thumb = self.unimported[0]

	if not self.input:
	    thumb.load_exif()
	    try:
		self.input = open(thumb.photo[u"imported-from"])
	    except IOError:
	    	self.input = None
    	    else:
		self.output = doc.storage.new_original(thumb.photo[u"id"])
		self.loader = gtk.gdk.PixbufLoader("jpeg")

    	if self.input:
	    data = self.input.read(4096)
	    if data:
		self.output.write(data)
		self.loader.write(data, len(data))
	    else:
		self.input.close()
		self.input = None
		doc.storage.close_original(thumb.photo[u"id"], self.output)
		self.output = None
		self.loader.close()
		pixbuf_orig = self.loader.get_pixbuf()
		thumb.pixbuf_thumb_orig = \
		    thumb.scale_pixbuf(pixbuf_orig, thumb.THUMB_SIZE)
		del self.loader
		self.loader = None
		pixbuf_orig = None
    
		name = doc.storage.new_thumbnail_filename(thumb.photo[u"id"])
		thumb.pixbuf_thumb_orig.save(name, "png")
		doc.storage.new_thumbnail_commit(thumb.photo[u"id"])
    
		if thumb.draw_hook:
		    thumb.draw_hook(thumb)
    
    	if self.input is None:
	    self.unimported = self.unimported[1:]
	    gc.collect() # Python heuristics for GC triggering are bad
			 # for us
	    self.show_progress()

	if self.unimported == []:
	    self.view.set_progress(0.0)
	    self.view.set_status_text("Imported %d photograps" % 
	    	    	    	      self.total_files)
    	    self.total_files = 0
	    return gtk.FALSE
	else:
	    return gtk.TRUE


class ExportOriginalsToSimpleFilesTask:

    def __init__(self, win, doc, photos, target_dir):
    	self.win = win
    	self.doc = doc
	self.remaining = photos
	self.dir = target_dir

    	self.input = None
	self.output = None
	self.photo = None
	self.current_file = 0
	self.total_files = len(photos)
	self.copied_bytes = 0
	self.total_bytes = self.compute_total_size()
	gtk.idle_add(self.work)
	
    def compute_total_size(self):
    	bytes = 0
	for photo in self.remaining:
	    bytes = bytes + self.doc.storage.get_original_size(photo[u"id"])
	return bytes

    def update_progress(self):
    	self.win.set_progress(float(self.copied_bytes) / self.total_bytes)
	self.win.set_status_text(u"Exporting photo %d of %d" % 
	    	    	    	 (self.current_file, self.total_files))

    def clear_progress(self):
    	self.win.set_progress(0)
	self.win.set_status_text(u"")

    def output_name(self):
    	id = self.photo[u"id"]
	name = os.path.join(self.dir, u"%s.jpg" % id)
	i = 0
	while os.path.exists(name):
	    i = i + 1
	    name = os.path.join(self.dir, u"%s-%d.jpg" % (id, i))
    	return name

    def error(self, primary, secondary):
	Alert.Alert(gtk.STOCK_DIALOG_ERROR, [(gtk.STOCK_OK,)], 
	    	    primary, secondary)

    def work(self, *args):
    	if not self.photo:
	    if not self.remaining:
	    	self.clear_progress()
	    	return gtk.FALSE
    	    self.photo = self.remaining[0]
	    self.remaining = self.remaining[1:]
	    self.current_file = self.current_file + 1

	    try:
	    	self.input = self.doc.storage.get_original(self.photo[u"id"])
		if self.input is None:
		    raise IOError(u"File does not exist")
	    except IOError, detail:
	    	self.error(u"Could not read original file.",
		    	   u"Database may be corrupted. " + str(detail))
	    	self.clear_progress()
	    	return gtk.FALSE

    	    name = self.output_name()
	    try:
		self.output = open(name, "w")
	    except IOError, detail:
	    	self.error(u"Could not open output file %s" % name,
		    	   u"%s" % str(detail))
	    	self.clear_progress()
	    	return gtk.FALSE

    	assert self.photo != None
    	assert self.input != None
	assert self.output != None
	
    	try:
	    data = self.input.read(4096)
	    if data:
		self.output.write(data)
		self.copied_bytes = self.copied_bytes + len(data)
    	    else:
	    	self.input.close()
	    	self.output.close()
		self.input = None
		self.output = None
		self.photo = None
	except IOError, detail:
	    self.error(u"Error reading or writing a file.",
	    	       u"%s" % str(detail))
	    self.clear_progress()
	    return gtk.FALSE

	self.update_progress()
    	return gtk.TRUE


class Model:

    def __init__(self, doc, draw_thumb, import_task):
    	self.doc = doc
	self.draw_thumb = draw_thumb
    	self.folders = []
	self.thumbs = []
	self.selected = []
	self.focused = None
	self.import_task = import_task

    def set_document(self, doc):
    	self.doc = doc

    def forget_original_pixbufs(self):
    	for thumb in self.thumbs:
	    thumb.pixbuf_orig = None

    def num_photos(self):
    	n = 0
	for folder in self.folders:
	    n = n + len(folder.photos.get())
	return n

    def set_folders(self, folders):
    	self.folders = folders
	self.thumbs = []
    	for folder in folders:
	    for photo in folder.photos.get():
		self.thumbs.append(Thumbnail(self.doc, 
		    	    	    	     photo, 
					     self.draw_thumb))
    	if self.thumbs:
	    self.focused = self.thumbs[0]
    	else:
	    self.focused = None
	self.selected = []

    def append_photo(self):
    	assert len(self.folders) == 1
    	photo = MetaData.Photo()
	thumb = Thumbnail(self.doc, photo, self.draw_thumb)
    	self.folders[0].photos.add(photo)
	self.thumbs.append(thumb)
	return photo, thumb

    def import_photo(self, filename):
	photo, thumb = self.append_photo()
	photo[u"imported-from"] = unicode(filename)
	if not self.import_task.unimported:
	    gtk.idle_add(self.import_task.work)
	self.import_task.add(self.doc, thumb)

    def copy_selected(self):
    	return map(lambda t: t.photo.copy(), self.selected)

    def remove_selection(self, remove_from_storage):
	if self.focused in self.selected:
	    self.focused = None
    	for folder in self.folders:
	    for thumb in self.selected[:]:
	    	if folder.photos.has(thumb.photo):
		    folder.photos.remove(thumb.photo)
		    if remove_from_storage:
			self.doc.storage.remove(thumb.photo[u"id"])
		    self.thumbs.remove(thumb)
		    self.selected.remove(thumb)
    	assert self.selected == []

    def remove_selection_but_not_from_storage(self):
    	self.remove_selection(0)

    def remove_selection_from_storage(self):
    	self.remove_selection(1)

    def paste(self, photos):
    	assert len(self.folders) == 1
    	thumbs = map(lambda p: Thumbnail(self.doc, p, self.draw_thumb), 
	    	     photos)
    	if self.focused:
	    i = self.thumbs.index(self.focused) + 1
	else:
	    i = len(self.thumbs)
	self.thumbs = self.thumbs[:i] + thumbs + self.thumbs[i:]
	self.folders[0].photos.set(map(lambda t: t.photo, self.thumbs))
    	self.selected = thumbs[:]
	if thumbs:
	    self.focused = thumbs[0]
	else:
	    self.thumbs = None

    def insert(self, pos, photos):
    	assert len(self.folders) == 1
    	if pos is None:
	    i = len(self.thumbs)
	else:
	    i = pos + 1
    	thumbs = map(lambda p: Thumbnail(self.doc, p, self.draw_thumb), 
	    	     photos)
	self.thumbs = self.thumbs[:i] + thumbs + self.thumbs[i:]
	self.folders[0].photos.set(map(lambda t: t.photo, self.thumbs))

    def copy_file(self, orig_filename, dest_stream):
    	f = open(orig_filename, "r")
	while 1:
	    data = f.read(64 * 1024)
	    if not data:
	    	break
	    dest_stream.write(data)
    	f.close()

    def copy_from_other_lodju(self, photolist):
    	for photo in photolist:
	    f = self.doc.storage.new_original(photo[u"id"])
	    self.copy_file(photo[u"dnd:original-filename"], f)
	    self.doc.storage.close_original(photo[u"id"], f)
	    f = self.doc.storage.new_thumbnail(photo[u"id"])
	    self.copy_file(photo[u"dnd:thumbnail-filename"], f)
	    self.doc.storage.close_thumbnail(photo[u"id"], f)

    def select(self, thumb):
    	if thumb and thumb not in self.selected:
	    self.selected.append(thumb)

    def unselect(self, thumb):
    	if thumb and thumb in self.selected:
	    self.selected.remove(thumb)

    def select_all(self):
    	self.selected = self.thumbs[:]

    def unselect_all(self):
    	self.selected = []

    def invert_selection(self):
    	new_selection = []
	for thumb in self.thumbs:
	    if thumb not in self.selected:
	    	new_selection.append(thumb)
	self.selected = new_selection

    def extend_selection_to(self, thumb):
	if thumb:
	    if not self.get_selection():
		self.select(thumb)
	    else:
		first = self.selected[0]
		index_first = self.thumbs.index(first)
		index_thumb = self.thumbs.index(thumb)
		if index_first < index_thumb:
		    a = index_first
		    b = index_thumb + 1
		else:
		    a = index_thumb
		    b = index_first + 1
		self.unselect_all()
		self.select(first)
		for t in self.thumbs[a:b]:
		    self.select(t)

    def is_selected(self, thumb):
    	return thumb in self.selected

    def get_selection(self):
    	return self.selected[:]

    def get_drag_data(self):
	photos = map(lambda t: t.photo, self.get_selection())
	for photo in photos:
	    photo.set_image_filenames(self.doc.storage)
	value = MetaData.encode_photos(photos)
	for photo in photos:
	    photo.unset_image_filenames()
	return value

    def set_focused(self, focused):
    	self.focused = focused

    def get_focused(self):
    	return self.focused

    def move_focus_horizontally(self, diff):
    	assert diff == 1 or diff == -1
    	if self.focused:
	    index = self.thumbs.index(self.focused)
	    index = index + diff
	    if index < 0:
	    	index = len(self.thumbs) + index
	    elif index >= len(self.thumbs):
	    	index = index % len(self.thumbs)
    	    self.focused = self.thumbs[index]

    def focus_next(self):
    	self.move_focus_horizontally(1)

    def focus_previous(self):
    	self.move_focus_horizontally(-1)

    def focus_up(self):
    	if self.focused:
	    gotcha = None
	    for thumb in self.thumbs:
	    	if thumb.x == self.focused.x and thumb.y < self.focused.y:
		    if not gotcha or thumb.y > gotcha.y:
			gotcha = thumb
    	    if gotcha:
	    	self.focused = gotcha

    def focus_down(self):
    	if self.focused:
	    gotcha = None
	    for thumb in self.thumbs:
	    	if thumb.x == self.focused.x and thumb.y > self.focused.y:
		    if not gotcha or thumb.y < gotcha.y:
			gotcha = thumb
    	    if gotcha:
	    	self.focused = gotcha

    def focus_page_up(self, page_size):
    	if self.focused:
	    gotcha = None
	    for thumb in self.thumbs:
	    	if thumb.x == self.focused.x and thumb.y < self.focused.y:
		    if not gotcha or thumb.y < gotcha.y:
		    	if self.focused.y - thumb.y <= page_size:
			    gotcha = thumb
    	    if gotcha:
	    	self.focused = gotcha

    def focus_page_down(self, page_size):
    	if self.focused:
	    gotcha = None
	    for thumb in self.thumbs:
	    	if thumb.x == self.focused.x and thumb.y > self.focused.y:
		    if not gotcha or thumb.y > gotcha.y:
		    	if thumb.y - self.focused.y <= page_size:
			    gotcha = thumb
    	    if gotcha:
	    	self.focused = gotcha


class View:

    def __init__(self, window, doc, xml):
    	self.window = window
    	self.model = Model(doc, self.draw_thumb, ImportTask(window))
    	self.controller = Controller(self.model, self)

	table = xml.get_widget("photo_attribute_table")
    	self.metadata_editor = MetaDataEditor.View(table, None)

	self.widget = xml.get_widget("pic_area")
	self.widget.connect("configure-event", self.controller.on_configure)
	self.widget.connect("expose-event", self.controller.on_expose)
	self.widget.connect("button-press-event", self.controller.on_press)
	self.widget.connect("button-release-event", 
	    	    	    self.controller.on_release)
    	self.widget.connect("focus-in-event", self.controller.focus_changed)
    	self.widget.connect("focus-out-event", self.controller.focus_changed)
    	self.widget.connect("key-press-event", self.controller.on_key_press)
	self.widget.connect("scroll-event", self.controller.scroll)
	
    	self.widget.connect("drag-begin", self.controller.drag_begin)
    	self.widget.connect("drag-motion", self.controller.drag_motion)
    	self.widget.connect("drag-leave", self.controller.drag_leave)
    	self.widget.connect("drag-data-get", self.controller.drag_data_get)
    	self.widget.connect("drag-data-delete", 
	    	    	    self.controller.drag_data_delete)
    	self.widget.connect("drag-end", self.controller.drag_end)
    	self.widget.connect("drag-data-received", 
	    	    	    self.controller.drag_data_received)

    	hscale = xml.get_widget("thumb_scale")
    	self.thumb_adjustment = hscale.get_adjustment()
	self.thumb_adjustment.value = 1.0
	self.thumb_adjustment.lower = 30.0
	self.thumb_adjustment.upper = 30.0
	self.thumb_adjustment.step_increment = 10.0
	self.thumb_adjustment.page_increment = 10.0
	self.thumb_adjustment.page_size = 10.0
	self.thumb_adjustment.changed()
	self.thumb_adjustment.connect("value-changed", 
	    	    	    	      self.controller.scale_thumbnails)

    	self.scrollbar = xml.get_widget("thumb_scrollbar")
    	self.adjustment = self.scrollbar.get_adjustment()
	self.adjustment.connect("value-changed", self.controller.scrolled)

	self.thumb_size_in_pixels = 100
	self.thumb_gap = 12
	self.virtual_pixel_row_visible = 0
	
    	context = self.widget.get_pango_context()
	self.layout = pango.Layout(context)
	
	self.drop_indicator = None
	
    	self.sensitive_widgets = {}
	for name in ["rotate_left", "rotate_right", "rotate_left_button",
	    	     "rotate_right_button", "photo_properties",
	    	     "delete_photo", "import", "import_missing",
	    	     "export_originals"]:
    	    self.sensitive_widgets[name] = xml.get_widget(name)

	self.set_sensitivity()
	
	self.dialog = MetaDataDialog.View(xml, "Photo properties")
	
	self.load_orig_task = None

    def set_document(self, doc):
    	self.model.set_document(doc)

    def set_sensitivity(self):
    	num_folders = len(self.model.folders)
	num_photos = self.model.num_photos()
	num_selected = len(self.model.selected)
    	sensitive = []

    	if num_selected > 0:
	    sensitive.append("rotate_left")
	    sensitive.append("rotate_right")
	    sensitive.append("delete_photo")
	    sensitive.append("rotate_left_button")
	    sensitive.append("rotate_right_button")
	    sensitive.append("export_originals")

    	if num_selected == 1:
	    sensitive.append("photo_properties")

    	if num_folders == 1:
	    sensitive.append("import")
	    sensitive.append("import_missing")

    	for name in self.sensitive_widgets.keys():
	    if name in sensitive:
	    	self.sensitive_widgets[name].set_sensitive(gtk.TRUE)
	    else:
	    	self.sensitive_widgets[name].set_sensitive(gtk.FALSE)

    	if num_folders > 0 and num_photos > 0:
	    self.widget.drag_source_set(gtk.gdk.BUTTON1_MASK,
					[("application/x-lodju-photos", 0, 0)],
					gtk.gdk.ACTION_MOVE)
	else:
	    self.widget.drag_source_unset()

	if num_folders == 1:
	    self.widget.drag_dest_set(gtk.DEST_DEFAULT_ALL,
				      [("application/x-lodju-photos", 0, 0)],
				      gtk.gdk.ACTION_MOVE)
	else:
	    self.widget.drag_dest_unset()

    def update_metadata_editor(self):
    	thumbs = self.model.get_selection()
	a = self.model.doc.photoattrs
	self.metadata_editor.setup(a, map(lambda t: t.photo, thumbs))
	
	# This would be prettier if it were in its own function, but I
	# can't be bothered to add calls to it everywhere. This indicates
	# a bad mini-design somewhere.
	self.window.set_status_text("%d photos shown, %d selected" %
				    (self.model.num_photos(), len(thumbs)))

    	self.model.forget_original_pixbufs();
    	if len(thumbs) == 1:
	    if self.load_orig_task:
		if self.load_orig_task.is_loading(thumbs[0]):
		    return
		self.load_orig_task.stop()
	    self.load_orig_task = LoadOriginalTask(self.model.doc, thumbs[0])
    	elif self.load_orig_task:
	    self.load_orig_task.stop()

    def set_folders(self, folders):
    	self.model.set_folders(folders)
	self.recompute_positions()
	self.update_scrollbar(0.0)
	self.draw()
	self.set_sensitivity()
	self.update_metadata_editor()

    def reset_folders(self):
    	self.set_folders(self.model.folders[:])

    def actual_width(self):
    	if self.widget.window:
	    return self.widget.window.get_size()[0]
	else:
	    return 0

    def actual_height(self):
    	if self.widget.window:
	    return self.widget.window.get_size()[1]
	else:
	    return 0

    def virtual_height(self):
    	w = self.actual_width()
	on_one_row = w / (self.thumb_size_in_pixels + self.thumb_gap)
	if on_one_row == 0:
	    on_one_row = 1
	num_thumbs = self.model.num_photos()
	num_rows = num_thumbs / on_one_row
	if num_thumbs % on_one_row > 0:
	    num_rows = num_rows + 1
	return num_rows * (self.thumb_size_in_pixels + self.thumb_gap)

    def update_scrollbar(self, new_value=None):
	actual_height = self.actual_height()
	if actual_height == 0:
	    return
	virtual_height = self.virtual_height()
	if virtual_height == 0:
	    virtual_height = actual_height

    	if new_value != None:
	    value = new_value
	else:
	    value = self.adjustment.value
	if value > virtual_height - actual_height:
	    value = virtual_height - actual_height
	if value < 0:
	    value = 0

        self.adjustment.value = float(value)
        self.adjustment.lower = 0.0
        self.adjustment.upper = float(virtual_height)
        self.adjustment.step_increment = float(self.thumb_size_in_pixels)
        self.adjustment.page_increment = float(actual_height)
        self.adjustment.page_size = float(actual_height)
        self.adjustment.changed()

    def get_graphics_contexts(self, state):
    	fg = self.widget.get_style().fg_gc[state]
    	bg = self.widget.get_style().bg_gc[state]
	return fg, bg

    def set_solid_lines(self):
	fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
	fg.set_line_attributes(1,
			       gtk.gdk.LINE_SOLID,
			       gtk.gdk.CAP_BUTT,
			       gtk.gdk.JOIN_ROUND)

    def set_dashed_lines(self):
	fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
    	fg.set_dashes(0, [3, 3])
	fg.set_line_attributes(1,
			       gtk.gdk.LINE_ON_OFF_DASH,
			       gtk.gdk.CAP_BUTT,
			       gtk.gdk.JOIN_ROUND)

    def undraw_thumb_focus(self):
    	thumb = self.model.get_focused()
    	w = self.widget.window
    	if thumb is None or w is None:
	    return
    	width, height = w.get_size()
	x, y = thumb.get_position()
	if self.thumb_is_visible(y, height):
	    self.set_dashed_lines()
	    fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
	    xx = x - 3
	    yy = y - 3 - self.virtual_pixel_row_visible
	    dim = self.thumb_size_in_pixels + 6
	    w.draw_rectangle(bg, gtk.FALSE, xx, yy, dim, dim)

    def draw_thumb_focus(self):
    	thumb = self.model.get_focused()
    	w = self.widget.window
    	if thumb is None or w is None:
	    return
    	width, height = w.get_size()
	x, y = thumb.get_position()
	if self.thumb_is_visible(y, height):
	    self.set_dashed_lines()
	    fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
	    xx = x - 3
	    yy = y - 3 - self.virtual_pixel_row_visible
	    dim = int(self.thumb_size_in_pixels + 6)
	    w.draw_rectangle(fg, gtk.FALSE, int(xx), int(yy), dim, dim)

    def draw_one_thumbnail(self, thumb, w, fg, bg):
    	x, virtual_y = thumb.get_position()
	y = virtual_y - self.virtual_pixel_row_visible
	thumb_size = self.thumb_size_in_pixels
    	self.set_solid_lines()
	w.draw_rectangle(bg, gtk.TRUE, int(x), int(y), 
	    	    	 int(thumb_size), int(thumb_size))
    	if 0:
	    w.draw_rectangle(fg, gtk.FALSE, x, y, thumb_size-1, thumb_size-1)
	xx = x + thumb_size - 1
	yy = y + thumb_size - 1
	midx = (x + xx) / 2
	midy = (y + yy) / 2
	thumb_pixbuf = thumb.get_thumb_pixbuf(thumb_size)
	if thumb_pixbuf:
	    tx = (thumb_size - thumb_pixbuf.get_width()) / 2
	    ty = (thumb_size - thumb_pixbuf.get_height()) / 2
	    thumb_pixbuf.render_to_drawable(w, fg, 0, 0, int(x+tx), 
	    	    	    	    	    int(y+ty), -1, -1,
					    gtk.gdk.RGB_DITHER_NONE, 0, 0)
    	elif 1:
	    w.draw_rectangle(fg, gtk.FALSE, int(x), int(y), 
	    	    	     int(thumb_size-1), int(thumb_size-1))
    	elif 0:
	    angle = thumb.photo.get_angle()
	    if angle == 0:
		w.draw_line(fg, x, yy, midx, y)
		w.draw_line(fg, midx, y, xx, yy)
	    elif angle == 90:
		w.draw_line(fg, x, y, xx, midy)
		w.draw_line(fg, xx, midy, x, yy)
	    elif angle == 180:
		w.draw_line(fg, xx, y, midx, yy)
		w.draw_line(fg, midx, yy, x, y)
	    elif angle == 270:
		w.draw_line(fg, xx, yy, x, midy)
		w.draw_line(fg, x, midy, xx, y)
	    self.layout.set_text(thumb.photo[u"id"])
	    w.draw_layout(fg, x, y, self.layout)

    def draw_thumb(self, thumb):
    	w = self.widget.window
	if not w:
	    return
	if self.model.is_selected(thumb):
	    fg, bg = self.get_graphics_contexts(gtk.STATE_SELECTED)
	else:
	    fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
	self.draw_one_thumbnail(thumb, w, fg, bg)

    def undraw_selection(self):
    	w = self.widget.window
	if not w:
	    return
	fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
    	for thumb in self.model.get_selection():
	    self.draw_one_thumbnail(thumb, w, fg, bg)

    def draw_selection(self):
    	w = self.widget.window
	if not w:
	    return
	fg, bg = self.get_graphics_contexts(gtk.STATE_SELECTED)
    	for thumb in self.model.get_selection():
	    self.draw_one_thumbnail(thumb, w, fg, bg)

    def draw_focus(self):
    	w = self.widget.window
	if not w:
	    return
    	normal = self.get_graphics_contexts(gtk.STATE_NORMAL)
	width, height = w.get_size()
	self.set_solid_lines()
	if self.widget.is_focus():
	    gc = normal[0]
	else:
	    gc = normal[1]
    	w.draw_rectangle(gc, gtk.FALSE, 0, 0, width - 1, height - 1)

    def draw(self):
    	w = self.widget.window
	if not w:
	    return
    	normal = self.get_graphics_contexts(gtk.STATE_NORMAL)
    	selected = self.get_graphics_contexts(gtk.STATE_SELECTED)
	width, height = w.get_size()
    	w.draw_rectangle(normal[1], gtk.TRUE, 0, 0, width, height)
    	thumb_size = self.thumb_size_in_pixels
    	for thumb in self.model.thumbs:
	    x, virtual_y = thumb.get_position()
    	    if self.thumb_is_visible(virtual_y, height):
	    	if self.model.is_selected(thumb):
		    fg, bg = selected
    	    	else:
		    fg, bg = normal
		self.draw_one_thumbnail(thumb, w, fg, bg)
    	self.draw_thumb_focus()
	self.draw_focus()

    def draw_drop_indicator_with_gc(self, coords, gc):
    	x, y = coords
	y = y - self.virtual_pixel_row_visible
    	w = self.widget.window
    	self.set_solid_lines()
	w.draw_line(gc, int(x), int(y), 
	    	    	int(x), int(y + self.thumb_size_in_pixels))

    def undraw_drop_indicator(self):
    	if self.drop_indicator:
	    fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
	    self.draw_drop_indicator_with_gc(self.drop_indicator, bg)
	    self.drop_indicator = None

    def draw_drop_indicator(self, thumb_index):
	if thumb_index is None:
	    if len(self.model.thumbs) == 0:
	    	x = self.thumb_gap / 2
		y = self.thumb_gap / 2
	    else:
	    	thumb = self.model.thumbs[-1]
		x = thumb.x + self.thumb_size_in_pixels + self.thumb_gap / 2
		y = thumb.y
    	elif thumb_index == -1:
	    x = self.thumb_gap / 4
	    y = self.thumb_gap / 2
	else:
	    thumb = self.model.thumbs[thumb_index]
	    x = thumb.x + self.thumb_size_in_pixels + self.thumb_gap / 2
	    y = thumb.y

    	self.undraw_drop_indicator()
	fg, bg = self.get_graphics_contexts(gtk.STATE_NORMAL)
	self.drop_indicator = (x, y)
    	self.draw_drop_indicator_with_gc(self.drop_indicator, fg)

    def thumb_is_visible(self, y, window_height):
    	ts = self.thumb_size_in_pixels
    	return y + ts >= self.virtual_pixel_row_visible and \
	       y <= self.virtual_pixel_row_visible + window_height

    def scroll_to_virtual_pixel_row(self, y):
        self.adjustment.value = float(y)
        self.adjustment.value_changed()

    def scroll_vertically(self, deltay):
	actual_height = self.actual_height()
	if actual_height == 0:
	    return
	virtual_height = self.virtual_height()
	if virtual_height == 0:
	    virtual_height = actual_height
    	y = self.adjustment.value + deltay
	if y < 0:
	    y = 0
	elif y > virtual_height - actual_height:
	    y = virtual_height - actual_height
    	self.scroll_to_virtual_pixel_row(y)

    def scroll_to_focused(self):
    	thumb = self.model.get_focused()
	if thumb is None:
	    return
	x, y = thumb.get_position()
	y = y - self.thumb_gap/2
	if y < self.virtual_pixel_row_visible:
	    self.scroll_to_virtual_pixel_row(y)
	width, height = self.widget.window.get_size()
	yy = y + self.thumb_size_in_pixels + self.thumb_gap + self.thumb_gap/2
	if yy > self.virtual_pixel_row_visible + height:
	    yy = yy - height
	    if yy >= 0:
	    	self.scroll_to_virtual_pixel_row(yy)

    def recompute_positions(self):
	w = self.widget.window
    	if not w:
	    return
	width, height = w.get_size() 
	thumb_size = self.thumb_size_in_pixels 
	x = self.thumb_gap / 2 
	y = self.thumb_gap / 2 
	for thumb in self.model.thumbs:
	    thumb.set_position(x, y, thumb_size, thumb_size) 
	    x = x + thumb_size + self.thumb_gap
	    if x + thumb_size >= width:
		x = self.thumb_gap / 2 
		y = y + thumb_size + self.thumb_gap

    def find_thumbnail(self, window_x, window_y):
    	x = window_x
	y = self.virtual_pixel_row_visible + window_y
	for thumb in self.model.thumbs:
	    if thumb.contains(x, y):
	    	return thumb
    	return None


class Controller:

    def __init__(self, model, view):
    	self.model = model
	self.view = view
	self.events_have_been_set = 0
	self.button_press_position = None
	self.select_on_release_if_click = None
	self.filesel = None
	self.filesel_location = u""
	self.export_filesel_location = u""
	self.internal_drag = 0
	self.export_task = None

    def find_drop_neighbor(self, x, y):
	ts = self.view.thumb_size_in_pixels
	tg = self.view.thumb_gap
	pos = None
	for i in range(len(self.model.thumbs)):
	    thumb = self.model.thumbs[i]
	    if y <= thumb.y + ts + tg:
    	    	if i == 0 and x < thumb.x:
		    pos = -1
		    break
    	    	if i > 0 and thumb.y != self.model.thumbs[i-1].y:
		    if x < thumb.x:
			pos = i - 1
			break
	    	if x >= thumb.x and x <= thumb.x + ts + tg:
		    pos = i
		    break
    	return pos

    def drag_begin(self, *args):
	self.internal_drag = 0

    def drag_motion(self, *args):
	area, context, x, y, timestamp = args
	y = y + self.view.virtual_pixel_row_visible
	pos = self.find_drop_neighbor(x, y)
    	self.view.draw_drop_indicator(pos)
	return gtk.TRUE
	
    def drag_leave(self, *args):
	self.view.undraw_drop_indicator()

    def drag_data_get(self, *args):
	area, context, selectiondata, info, data = args
	str = self.model.get_drag_data()
	selectiondata.set(selectiondata.target, 8, str.encode("utf-8"))

    def drag_data_delete(self, *args):
	if not self.internal_drag:
	    print "TA: drag data delete"
	    self.model.remove_selection_from_storage()
	else:
	    self.model.remove_selection_but_not_from_storage()

    def drag_end(self, *args):
    	self.internal_drag = 0
	self.model.unselect_all()
	self.view.undraw_drop_indicator()
	self.view.recompute_positions()
	self.view.update_scrollbar()
	self.view.draw()
	self.view.set_sensitivity()
	self.view.update_metadata_editor()
	
    def drag_data_received(self, *args):
    	print "TA: drag data received"
	area, context, x, y, selectiondata, info, timestamp = args
	p = MetaData.Parser()
	p.feed(selectiondata.data)
	photos = p.close()
	y = y + self.view.virtual_pixel_row_visible
	pos = self.find_drop_neighbor(x, y)
	if not self.view.window.drag_source_is_within_window(context):
	    self.view.window.folder_tree.model.set_unique_ids(photos.get())
    	else:
	    self.internal_drag = 1
	self.model.insert(pos, photos.get())
	if not self.view.window.drag_source_is_within_window(context):
	    self.model.copy_from_other_lodju(photos.get())
	context = args[1]
	if context.source_window != context.dest_window:
	    self.view.recompute_positions()
	    self.view.update_scrollbar()
	    self.view.draw()

    def on_configure(self, *args):
    	if not self.events_have_been_set:
	    e = self.view.widget.window.get_events()
	    e = e | gtk.gdk.BUTTON_PRESS_MASK
	    e = e | gtk.gdk.BUTTON_RELEASE_MASK
	    e = e | gtk.gdk.KEY_PRESS_MASK
	    e = e | gtk.gdk.KEY_RELEASE_MASK
	    self.view.widget.window.set_events(e)
	    self.events_have_been_set = 1
	width = float(self.view.actual_width())
	height = float(self.view.actual_height())
	thumb = float(self.view.thumb_size_in_pixels)
	self.view.thumb_adjustment.value = thumb
        self.view.thumb_adjustment.upper = min(width, height)
    	self.view.thumb_adjustment.changed()
	self.view.update_scrollbar()
	self.view.recompute_positions()

    def on_expose(self, *args):
    	self.view.draw()

    def scrolled(self, *args):
	self.view.virtual_pixel_row_visible = self.view.adjustment.value
	self.view.draw()

    def scale_thumbnails(self, *args):
    	t = self.view.thumb_adjustment
	self.view.thumb_size_in_pixels = \
	    int(self.view.thumb_adjustment.value)
	self.view.recompute_positions()
	self.view.update_scrollbar()
	self.view.draw()

    def on_press(self, *args):
    	self.view.widget.grab_focus()

    	e = args[1]
	self.button_press_position = (e.x, e.y)
    	thumb = self.view.find_thumbnail(e.x, e.y)
	
    	self.model.set_focused(thumb)
	
    	shift = e.state & gtk.gdk.SHIFT_MASK
	control = e.state & gtk.gdk.CONTROL_MASK
	
	if not shift and not control:
	    selection = self.model.get_selection()
	    if not selection:
	    	self.model.select(thumb)
	    elif thumb in selection:
	    	self.select_on_release_if_click = thumb
	    else:
	    	self.model.unselect_all()
		self.model.select(thumb)
    	elif control and not shift:
	    self.model.select(thumb)
    	elif shift and not control:
	    self.model.extend_selection_to(thumb)

    	self.view.set_sensitivity()
	self.view.update_metadata_editor()
    	self.view.draw()
	return gtk.FALSE

    def on_release(self, *args):
	e = args[1]
	if self.button_press_position == (e.x, e.y):
	    if self.select_on_release_if_click:
	    	self.model.unselect_all()
		self.model.select(self.select_on_release_if_click)
		self.view.draw()
		self.view.update_metadata_editor()
		self.view.set_sensitivity()
    	self.button_press_position = None
	self.select_on_release_if_click = None
    	return gtk.FALSE

    def handle_endpoint_key(self, index, shift, control):
	self.view.undraw_selection()
	self.view.undraw_thumb_focus()
	if not control and not shift:
	    self.model.unselect_all()
	    if self.model.thumbs:
		self.model.select(self.model.thumbs[index])
		self.model.set_focused(self.model.thumbs[index])
	elif control and not shift:
	    if self.model.thumbs:
		self.model.set_focused(self.model.thumbs[index])
	elif shift and not control:
	    if self.model.thumbs:
		self.model.extend_selection_to(self.model.thumbs[index])
		self.model.set_focused(self.model.thumbs[index])
	self.view.scroll_to_focused()
	self.view.draw_selection()
	self.view.draw_thumb_focus()
	self.view.update_metadata_editor()
    	self.view.set_sensitivity()

    def handle_movement(self, shift, control, move_focus):
	self.view.undraw_thumb_focus()
	self.view.undraw_selection()
    	move_focus()

    	focused = self.model.get_focused()
	if focused:
	    if shift:
	    	self.model.select(focused)
	    elif not control:
		self.model.unselect_all()
		self.model.select(focused)

	self.view.scroll_to_focused()
	self.view.draw_selection()
	self.view.draw_thumb_focus()
    	self.view.set_sensitivity()
	self.view.update_metadata_editor()

    def handle_page_movement(self, shift, control, move_focus):
	self.view.undraw_thumb_focus()
	self.view.undraw_selection()
	move_focus(self.view.adjustment.page_size)
	focused = self.model.get_focused()
	if focused:
	    if shift and not control:
		self.model.extend_selection_to(focused)
	    elif not shift and not control:
		self.model.unselect_all()
		self.model.select(focused)
	self.view.scroll_to_focused()
	self.view.draw_selection()
	self.view.draw_thumb_focus()
    	self.view.set_sensitivity()
	self.view.update_metadata_editor()

    def on_key_press(self, *args):
    	e = args[1]
	name = gtk.gdk.keyval_name(e.keyval)
    	shift = e.state & gtk.gdk.SHIFT_MASK
	control = e.state & gtk.gdk.CONTROL_MASK

	if name == "Left":
	    self.handle_movement(shift, control, self.model.focus_previous)
	    return gtk.TRUE

	elif name == "Right":
	    self.handle_movement(shift, control, self.model.focus_next)
	    return gtk.TRUE

    	elif name == "Up":
    	    self.handle_movement(shift, control, self.model.focus_up)
	    return gtk.TRUE

    	elif name == "Down":
    	    self.handle_movement(shift, control, self.model.focus_down)
	    return gtk.TRUE

    	elif name == "Home":
	    self.handle_endpoint_key(0, shift, control)
	    return gtk.TRUE
	
	elif name == "End":
	    self.handle_endpoint_key(-1, shift, control)
	    return gtk.TRUE

    	elif name == "Page_Up":
	    self.handle_page_movement(shift, control, 
	    	    	    	      self.model.focus_page_up)
	    return gtk.TRUE

    	elif name == "Page_Down":
	    self.handle_page_movement(shift, control, 
	    	    	    	      self.model.focus_page_down)
	    return gtk.TRUE

    	elif name == "space":
	    self.view.undraw_selection()
	    if not control and not shift:
		self.model.unselect_all()
		self.model.select(self.model.get_focused())
    	    elif control and not shift:
	    	thumb = self.model.get_focused()
		if thumb:
		    if self.model.is_selected(thumb):
			self.model.unselect(thumb)
		    else:
		    	self.model.select(thumb)
    	    elif shift and not control:
	    	self.model.extend_selection_to(self.model.get_focused())

	    self.view.draw_selection()
	    self.view.set_sensitivity()
	    self.view.update_metadata_editor()
	    return gtk.TRUE

	return gtk.FALSE

    def focus_changed(self, *args):
	self.view.draw_focus()

    def on_select_all_activate(self, *args):
    	self.view.undraw_selection()
	self.model.select_all()
	self.view.draw_selection()
    	self.view.set_sensitivity()
	self.view.update_metadata_editor()

    def on_unselect_all_activate(self, *args):
    	self.view.undraw_selection()
	self.model.unselect_all()
	self.view.draw_selection()
    	self.view.set_sensitivity()
	self.view.update_metadata_editor()

    def on_invert_selection_activate(self, *args):
    	self.view.undraw_selection()
	self.model.invert_selection()
	self.view.draw_selection()
    	self.view.set_sensitivity()
	self.view.update_metadata_editor()

    def on_import_activate(self, *args):
    	Filesel.Filesel(u"Photo import", self.filesel_location, 
	    	    	gtk.FALSE, gtk.TRUE, 
	    	    	self.import_ok)

    def import_ok(self, filenames):
    	if len(filenames) > 0:
	    if os.path.isdir(filenames[0]):
	    	self.filesel_location = filenames[0]
    	    else:
	    	self.filesel_location = os.path.dirname(filenames[0])
	n = 0
	for filename in filenames:
	    if os.path.isfile(filename):
		self.model.import_photo(filename)
		n = n + 1
	self.view.recompute_positions()
	self.view.update_scrollbar()
	self.view.draw()

    def on_import_missing_activate(self, *args):
    	st = self.model.doc.storage
    	missing = filter(lambda t: not st.get_original_filename(t.photo[u"id"]),
	    	    	 self.model.thumbs)
	if not self.model.import_task.unimported:
	    gtk.idle_add(self.model.import_task.work)
    	for thumb in missing:
	    self.model.import_task.add(self.model.doc, thumb)

    def rotate(self, diff):
    	for thumb in self.model.get_selection():
	    angle = thumb.photo.get_angle() + diff
	    while angle < 0:
	    	angle = angle + 360
	    while angle >= 360:
	    	angle = angle - 360
    	    assert angle >= 0
	    assert angle < 360
	    thumb.photo.set_angle(angle)
	    self.view.draw_thumb(thumb)

    def on_rotate_left_activate(self, *args):
	self.rotate(90)

    def on_rotate_right_activate(self, *args):
	self.rotate(-90)

    def on_rotate_left_button_clicked(self, *args):
    	self.rotate(90)

    def on_rotate_right_button_clicked(self, *args):
    	self.rotate(-90)

    def on_photo_properties_activate(self, *args):
    	thumbs = self.model.get_selection()
	if len(thumbs) == 1:
	    photo = thumbs[0].photo
	    self.view.dialog.show(self.model.doc.photoattrs, photo)

    def scroll(self, *args):
	e = args[1]
	if e.direction == gtk.gdk.SCROLL_UP:
	    self.view.scroll_vertically(- self.view.thumb_size_in_pixels)
    	elif e.direction == gtk.gdk.SCROLL_DOWN:
	    self.view.scroll_vertically(self.view.thumb_size_in_pixels)
	return gtk.FALSE

    def on_export_originals_activate(self, *args):
    	Filesel.Filesel(u"Export originals", self.export_filesel_location,
	    	    	gtk.TRUE, gtk.FALSE, self.export_ok)

    def export_error(self, primary, secondary):
	Alert.Alert(gtk.STOCK_DIALOG_ERROR, [(gtk.STOCK_OK,)], 
	    	    primary, secondary)

    def export_ok(self, filenames):
	if len(filenames) != 1:
	    self.export_error(u"Too many selected",
	    	    	      u"You selected too many files or directories." +
	    	    	      u" You may only select one.")
    	elif not  os.path.isdir(filenames[0]):
	    self.export_error(u"Not a directory: %s" % filenames[0],
	    	    	      u"You selected something that doesn't exist " +
			      u"or is not a directory.")
    	else:
    	    self.export_filesel_location = filenames[0]
	    photos = map(lambda t: t.photo, self.model.get_selection())
	    ExportOriginalsToSimpleFilesTask(self.view.window,
	    	    	    	    	     self.model.doc, 
	    	    	    	    	     photos, 
					     filenames[0])
