/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "collision.h"


#define MAP(x,y)	cd.map[x*cd.ysquares+y]


struct collision_detector
{
  char *map;
  int xsquares, ysquares;
} cd;


void init_collision_detector( int xsquares, int ysquares )
{
  cd.xsquares = xsquares;
  cd.ysquares = ysquares;
  cd.map = (char *) malloc( xsquares * ysquares );
  if( !cd.map ) {
    fprintf( stderr, "Unable to allocate memory for collision detection map.\n" );
    exit(1);
  }
  memset( cd.map, 0, xsquares*ysquares );
}


void destroy_collision_detector( void )
{
  cd.xsquares = 0;
  cd.ysquares = 0;
  free( cd.map );
}


char check_collisions( int x0, int y0, int w, int h, char obj_mask )
{
  int x_stop = (x0 + w) % cd.xsquares;
  int y_stop = (y0 + h) % cd.ysquares;
  int y;
  char result = MAZE_OBJECT_NONE;
  
  assert( x0 >= 0);
  assert( y0 >= 0);
  assert( x0 < cd.xsquares );
  assert( y0 < cd.ysquares );
  
  for( ; x0 != x_stop ; x0 = (x0 + 1) % cd.xsquares )
    for( y=y0; y != y_stop ; y = (y + 1) % cd.ysquares )
      result |= MAP(x0,y);

  return result & obj_mask;
}


char check_adjacent_collisions( int x0, int y0, int w, int h, char obj_mask )
{
  int xf = (x0 + w - 1) % cd.xsquares;
  int yf = (y0 + h - 1) % cd.ysquares;
  int x_l = (x0 + cd.xsquares - 1) % cd.xsquares;
  int y_t  = (y0 + cd.ysquares - 1) % cd.ysquares;
  int x_r = (x0 + w) % cd.xsquares;
  int y_b = (y0 + h) % cd.ysquares;

  int x_strt = (x0 + 1) % cd.xsquares;
  int y_strt = (y0 + 1) % cd.ysquares;
  int x_stop = (x0 + w - 1) % cd.xsquares;
  int y_stop = (y0 + h - 1) % cd.ysquares;

  int x, y;
  char result[8];
  char uresult, vresult;
  
  assert( x0 >= 0);
  assert( y0 >= 0);
  assert( x0 < cd.xsquares );
  assert( y0 < cd.ysquares );
  
  for( x=0; x<8; x++ )
    result[x] = MAZE_OBJECT_NONE;
  
  /* result matrix
   * -------------
   * 0  1  2
   * 3 obj 4
   * 5  6  7
   */
  
  result[0] = MAP(x_l,y_t);
  result[2] = MAP(x_r,y_t);
  result[5] = MAP(x_l,y_b);
  result[7] = MAP(x_r,y_b);
  
  result[3] |= MAP(x_l,y0);
  result[4] |= MAP(x_r,y0);

  if( h > 1 ) {
    result[3] |= MAP(x_l,yf);
    result[4] |= MAP(x_r,yf);

    result[0] |= MAP(x_l,y0);
    result[5] |= MAP(x_l,yf);

    result[2] |= MAP(x_r,y0);
    result[7] |= MAP(x_r,yf);
    
    if( h > 2 ) {
      uresult = vresult = MAZE_OBJECT_NONE;
      for( y=y_strt; y != y_stop; y = (y + 1) % cd.ysquares ) {
	uresult |= MAP(x_l,y);
	vresult |= MAP(x_r,y);
      }
      result[0] |= uresult;
      result[3] |= uresult;
      result[5] |= uresult;
      result[2] |= vresult;
      result[4] |= vresult;
      result[7] |= vresult;
    }
  }

  result[1] |= MAP(x0,y_t);
  result[6] |= MAP(x0,y_b);

  if( w > 1 ) {
    result[1] |= MAP(xf,y_t);
    result[6] |= MAP(xf,y_b);

    result[0] |= MAP(x0,y_t);
    result[2] |= MAP(xf,y_t);

    result[5] |= MAP(x0,y_b);
    result[7] |= MAP(xf,y_b);

    if( w > 2 ) {
      uresult = vresult = MAZE_OBJECT_NONE;
      for( x=x_strt; x != x_stop; x = (x + 1) % cd.xsquares ) {
	uresult |= MAP(x,y_t);
	vresult |= MAP(x,y_b);
      }
      result[0] |= uresult;
      result[1] |= uresult;
      result[2] |= uresult;
      result[5] |= vresult;
      result[6] |= vresult;
      result[7] |= vresult;
    }
  }

  uresult = 0;
  for( x=7; x>=0; x-- )
    uresult = (uresult << 1) | ( (result[x] & obj_mask) ? 1 : 0 );

  return uresult;
}


char check_line_collisions( int x0, int y0, int dx, int dy, int len, char obj_mask )
{
  char result = MAZE_OBJECT_NONE;
  
  assert( len >= 0);
  assert( x0 >= 0);
  assert( y0 >= 0);
  assert( x0 < cd.xsquares );
  assert( y0 < cd.ysquares );
  
  for( ; len!=0; len--, x0+=dx, y0+=dy )
    result |= MAP(x0,y0);

  return result & obj_mask;
}


void collision_map_place_obj( int x0, int y0, int w, int h, char obj_type )
{
  int x_stop = (x0 + w) % cd.xsquares;
  int y_stop = (y0 + h) % cd.ysquares;
  int y;

  assert( x0 >= 0);
  assert( y0 >= 0);
  assert( x0 < cd.xsquares );
  assert( y0 < cd.ysquares );
  
  for( ; x0 != x_stop ; x0 = (x0 + 1) % cd.xsquares )
    for( y=y0; y != y_stop ; y = (y + 1) % cd.ysquares )
      MAP(x0,y) |= obj_type;
}


void collision_map_remove_obj( int x0, int y0, int w, int h, char obj_type )
{
  int x_stop = (x0 + w) % cd.xsquares;
  int y_stop = (y0 + h) % cd.ysquares;
  int y;

  assert( x0 >= 0);
  assert( y0 >= 0);
  assert( x0 < cd.xsquares );
  assert( y0 < cd.ysquares );
  
  for( ; x0 != x_stop ; x0 = (x0 + 1) % cd.xsquares )
    for( y=y0; y != y_stop ; y = (y + 1) % cd.ysquares )
      MAP(x0,y) &= ~obj_type;
}
