/*
 *  Linux snipes, a text-based maze-oriented game for linux.
 *  Copyright (C) 1997 Jeremy Boulton.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Jeremy Boulton is reachable via electronic mail at
 *  boultonj@ugcs.caltech.edu.
 */

/* walls.c */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <curses.h>
#include "snipes.h"
#include "mazegen.h"
#include "chars.h"
#include "collision.h"
#include "screen.h"
 
#define X_MAX	2000
#define Y_MAX	2000


#define TOP_WALL_MASK		0x8
#define LEFT_WALL_MASK		0x4
#define RIGHT_WALL_MASK		0x2
#define BOTTOM_WALL_MASK	0x1

chtype corner_char[]={
  BLANK_CHAR,			// 0
  VERTICAL_WALL_CHAR,		// 1
  HORIZONTAL_WALL_CHAR,		// 2
  UPPER_LEFT_CORNER_CHAR,	// 3
  HORIZONTAL_WALL_CHAR,		// 4
  UPPER_RIGHT_CORNER_CHAR,	// 5
  HORIZONTAL_WALL_CHAR,		// 6
  HORIZONTAL_T_DOWN_CHAR,	// 7
  VERTICAL_WALL_CHAR,		// 8
  VERTICAL_WALL_CHAR,		// 9
  LOWER_LEFT_CORNER_CHAR,	// 10
  VERTICAL_T_RIGHT_CHAR,	// 11
  LOWER_RIGHT_CORNER_CHAR,	// 12
  VERTICAL_T_LEFT_CHAR,		// 13
  HORIZONTAL_T_UP_CHAR,		// 14
  FOUR_WAY_INTERSECTION		// 15
};


/* WRAP macro for enforcing the range 0 <= t <= m.  Assumes that t will
 * always less than (m+1) out of range.
 */

#define WRAP(t,m)	( (t>m)?t-m-1:((t<0)?t+m+1:t) )

/* 2 more wrapping macros.  The range is 0 <= t <= m. t should be
 * in range to start with.
 */

#define INCWRAP(t,m)	( (t==m)?0:t+1 )
#define DECWRAP(t,m)	( (t==0)?m:t-1 )


/*
 * Cell width is the wall-to-wall width:
 * 
 * |--- 12 ---|
 * |          |
 * +----------+
 * 
 * and addressing of the map begins at (0,0) in the upper
 * left hand corner.  Use normalize_x_coord() and
 * normalize_y_coord() to make sure that (x,y) references
 * are not outsize of the bounds of the maze.  i.e., you
 * use these to handle wrap-around.
 * 
 */


typedef struct wall_info {
  char *wall_array;
  
  int cell_width;
  int cell_height;

  long max_x_cell;
  long max_y_cell;

  long max_x;
  long max_y;
} wall_info;


long normalize_x_coord( wall_info *w, long x );
long normalize_y_coord( wall_info *w, long y );
int init_walls( wall_info *w, int x, int y, int width, int height );
int close_walls( wall_info *w );
void draw_walls( wall_info *w, screen_coords *sc );


long normalize_x_coord( wall_info *w, long x )
{
  return WRAP(x, w->max_x);
}


long normalize_y_coord( wall_info *w, long y )
{
  return WRAP(y, w->max_y);
}


long next_x_cell( wall_info *w, long xcell )
{
  return INCWRAP( xcell, w->max_x_cell );
}


long next_y_cell( wall_info *w, long ycell )
{
  return INCWRAP( ycell, w->max_y_cell );
}


long prev_x_cell( wall_info *w, long xcell )
{
  return DECWRAP( xcell, w->max_x_cell );
}


long prev_y_cell( wall_info *w, long ycell )
{
  return DECWRAP( ycell, w->max_y_cell );
}


/* Functions for building the maze. */

/* set_wall_segment( wall_info *wi, int x, int y, int type )
 * Record for both display and collision detection purposes
 * that there is a wall of a particular corner type at the
 * given location.
 * 
 * x, y: maze coordinate at which to add wall
 *       Note: assumed to be in the correct range.
 * type: wall (corner) type number
 */

void set_wall_segment( wall_info *wi, int x, int y, int type )
{
  *(wi->wall_array+y*(wi->max_x+1)+x) = type;

  if( type )
    collision_map_place_obj( x, y, 1, 1, MAZE_OBJECT_WALL );
  else
    collision_map_remove_obj( x, y, 1, 1, MAZE_OBJECT_WALL );
}


/* check_wall_segment( wall_info *wi, int x, int y )
 * x, y: maze coordinate at which to add wall
 *       Note: assumed to be in the correct range.
 * Returns: (bool) wall exists at that location.
 */

int check_wall_segment( wall_info *wi, int x, int y )
{
  return *(wi->wall_array+y*(wi->max_x+1)+x);
}


/* add_h_wall( void *wi, long y, long x1, long x2 )
 * x1, x2, y: maze block coordinates
 * This function is called by the maze generator.
 */

void add_h_wall( void *wi, long y, long x1, long x2 )
{
  wall_info *w=wi;
  long i,x;
  
  for( x=x1; x!=x2; x=next_x_cell(w,x) )
    for( i=0; i<w->cell_width; i++ )
      set_wall_segment( w, x*w->cell_width+i, y*w->cell_height, 2 );
}


/* add_v_wall( void *wi, long x, long y1, long y2 )
 * y1, y2, x: maze block coordinates
 * This function is called by the maze generator.
 */

void add_v_wall( void *wi, long x, long y1, long y2 )
{
  wall_info *w=wi;
  long i,y;
  
  for( y=y1; y!=y2; y=next_y_cell(w,y) )
    for( i=0; i<w->cell_height; i++ )
      set_wall_segment( w, x*w->cell_width, y*w->cell_height+i, 1 );
}


int init_walls( wall_info *w, int x, int y, int width, int height )
{
  char temp;
  long i,j,cx,cy;
  maze *mymaze;
  
  if( x < 0 || x > X_MAX || y < 0 || y > Y_MAX )
    return 1;
  
  if( (w->wall_array = (char *)malloc( x*y*(width-1)*(height-1) )) == NULL )
    return 1;
  
  memset( w->wall_array, 0, x*y*(width-1)*(height-1)*sizeof(char) );

  w->cell_width=width-1;
  w->cell_height=height-1;

  w->max_x_cell=x-1;
  w->max_y_cell=y-1;

  w->max_x=x*( width-1)-1;
  w->max_y=y*(height-1)-1;

  mymaze = maze_init( w, (voidfunc)add_h_wall, (voidfunc)add_v_wall, x, y );
  maze_finish( mymaze );
  
  /* Set corner characters now that all the walls are drawn. */
  for( cx=0; cx<=w->max_x_cell; cx++ )
    for( cy=0; cy<=w->max_y_cell; cy++ ) {
      i = cx*w->cell_width;
      j = cy*w->cell_height;
      temp = 0;
      if( check_wall_segment( w, normalize_x_coord(w,i-1), j ) )
	temp |= LEFT_WALL_MASK;
      if( check_wall_segment( w, normalize_x_coord(w,i+1), j ) )
	temp |= RIGHT_WALL_MASK;
      if( check_wall_segment( w, i, normalize_y_coord(w,j-1) ) )
	temp |= TOP_WALL_MASK;
      if( check_wall_segment( w, i, normalize_y_coord(w,j+1) ) )
	temp |= BOTTOM_WALL_MASK;
      set_wall_segment( w, i, j, temp );
    }
  
  return 0;
}



int close_walls( wall_info *w )
{
  free( w->wall_array );
  return 0;
}


void draw_walls( wall_info *w, screen_coords *sc )
{
  long x, y;
  long final_x, final_y;
  long start2_x, start2_y;
  long final2_x, final2_y;
  int sx, sy;
  int index;
  char *t;
  
  screen_setnormalattr();
  
  start2_x = final2_x = final_x = normalize_x_coord( w, sc->maze_x2+1 );
  start2_y = final2_y = final_y = normalize_y_coord( w, sc->maze_y2+1 );
  
  if( final_x < sc->maze_x1 ) {
    start2_x = 0;
    final_x = w->max_x+1;
  }
  if( final_y < sc->maze_y1 ) {
    start2_y = 0;
    final_y = w->max_y+1;
  }
  
  sy = sc->screen_y1;
  for( y=sc->maze_y1; y!=final_y; y++, sy++ ) {
    sx = sc->screen_x1;
    t = w->wall_array+y*(w->max_x+1);
    for( x=sc->maze_x1; x!=final_x; x++, sx++ ) {
      index = *(t+x);
      if( corner_char[index] != BLANK_CHAR )
	screen_mvaddch( sy, sx, corner_char[index] );
    }
    for( x=start2_x; x!=final2_x; x++, sx++ ) {
      index = *(t+x);
      if( corner_char[index] != BLANK_CHAR )
	screen_mvaddch( sy, sx, corner_char[index] );
    }
  }
  for( y=start2_y; y!=final2_y; y++, sy++ ) {
    sx = sc->screen_x1;
    t = w->wall_array+y*(w->max_x+1);
    for( x=sc->maze_x1; x!=final_x; x++, sx++ ) {
      index = *(t+x);
      if( corner_char[index] != BLANK_CHAR )
	screen_mvaddch( sy, sx, corner_char[index] );
    }
    for( x=start2_x; x!=final2_x; x++, sx++ ) {
      index = *(t+x);
      if( corner_char[index] != BLANK_CHAR )
	screen_mvaddch( sy, sx, corner_char[index] );
    }
  }
}
