/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
//! @file inti/atk/implementor.h
//! @brief An AtkImplementor C++ wrapper interface.
//!
//! Provides Implementor, an interface implemented by G::Object instances wanting to access their Atk::Object implementation.

#ifndef INTI_ATK_IMPLEMENTOR_H
#define INTI_ATK_IMPLEMENTOR_H

#ifndef INTI_G_TYPE_H
#include <inti/glib/type.h>
#endif

#ifndef __ATK_OBJECT_H__
#include <atk/atkobject.h>
#endif

namespace Inti {

namespace Atk {

class ImplementorIface;
class Object;

//! @class Implementor implementor.h inti/atk/implementor.h
//! @brief An AtkImplementor C++ wrapper class.
//!
//! Implementor is an interface providing access to a G::Object's Atk::Object implementation.
//! This interface is implemented by Gtk::Widget (see ref_accessible()).

class Implementor : public virtual G::TypeInterface
{
	friend class ImplementorIface;

protected:
//! @name Constructors
//! @{

	virtual ~Implementor() = 0;
	//!< Destructor.
	
//! @}
//  Override these do_ methods when you want to change the default behaviour of the AtkImplementor.
	
	virtual AtkObject* do_ref_accessible();

public:
//! @name Accessors
//! @{

	AtkImplementor* atk_implementor() const { return (AtkImplementor*)instance; }
	//!< Get a pointer to the AtkImplementor structure.

	AtkImplementorIface* atk_implementor_iface() const;
	//!< Get a pointer to the AtkImplementorIface structure.

	operator AtkImplementor* () const;
	//!< Conversion operator; safely converts an Implementor to an AtkImplementor pointer.

	Pointer<Object> ref_accessible() const;
	//!< Gets a reference to an object's Atk::Object implementation,
	//!< if the object implements the Implementor interface.
	//!< @return A smart pointer to an object's Atk::Object implementation.
	//!<
	//!< <BR>Gtk::Widget implements this interface. Calling Gtk::Widget::get_accessible()
	//!< is the same calling this method on a widget, except this method  returns a
	//!< smart pointer because it increments the Atk::Object's reference count,
	//!< get_accessible() does not.

//! @}
};

} // namespace Atk

} // namespace Inti

#endif // INTI_ATK_IMPLEMENTOR_H

