/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  object.cc - AtkObject C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "object.h"
#include "private/object_p.h"
#include "relation.h"
#include "stateset.h"

using namespace Inti;

/*  Atk::role_register
 */

Atk::Role
Atk::role_register(const String& name)
{
	return (Role)atk_role_register(name.c_str());
}

/*  Atk::role_get_name
 */

String
Atk::role_get_name(Role role)
{
	return atk_role_get_name((AtkRole)role);
}

/*  Atk::role_get_localized_name
 */

String 
Atk::role_get_localized_name(Role role)
{
	return atk_role_get_localized_name((AtkRole)role);
}

/*  Atk::role_for_name
 */

Atk::Role
Atk::role_for_name(const String& name)
{
	return (Role)atk_role_for_name(name.c_str());
}

/*  Atk::Object
 */

Atk::Object::Object(AtkObject *object, bool reference)
: G::Object((GObject*)object, reference)
{
}

Atk::Object::~Object()
{
}

AtkObjectClass*
Atk::Object::atk_object_class() const
{ 
	return get_class<AtkObjectClass>(); 
}
	
Atk::Object::operator AtkObject* () const 
{ 
	return this ? atk_object() : 0; 
}

String
Atk::Object::get_name() const
{
	return atk_object_get_name(atk_object());
}

String
Atk::Object::get_description() const
{
	return atk_object_get_description(atk_object());
}

Atk::Object*
Atk::Object::get_parent() const
{
	AtkObject *parent = atk_object_get_parent(atk_object());
	return parent ? G::Object::wrap<Object>(parent) : 0;
}
	
int
Atk::Object::get_n_accessible_children() const
{
	return atk_object_get_n_accessible_children(atk_object());
}

Atk::Object*
Atk::Object::ref_accessible_child(int index) const
{
	AtkObject *child = atk_object_ref_accessible_child(atk_object(), index);
	return child ? G::Object::wrap<Object>(child) : 0;
}

Atk::RelationSet*
Atk::Object::ref_relation_set() const
{
	AtkRelationSet *set = atk_object_ref_relation_set(atk_object());
	return set ? G::Object::wrap<RelationSet>(set) : 0;
}

Atk::Role
Atk::Object::get_role() const
{
	return (Role)atk_object_get_role(atk_object());
}

Pointer<Atk::StateSet>
Atk::Object::ref_state_set() const
{
	AtkStateSet *set = atk_object_ref_state_set(atk_object());
	return set ? G::Object::wrap<StateSet>(set, true) : 0;
}

int 
Atk::Object::get_index_in_parent() const
{
	return atk_object_get_index_in_parent(atk_object());
}

void
Atk::Object::set_name(const String& name)
{
	atk_object_set_name(atk_object(), name.c_str());
}

void 
Atk::Object::set_description(const String& description)
{
	atk_object_set_description(atk_object(), description.c_str());
}

void 
Atk::Object::set_parent(Object& parent)
{
	atk_object_set_parent(atk_object(), parent.atk_object());
}

void 
Atk::Object::set_role(Role role)
{
	atk_object_set_role(atk_object(), (AtkRole)role);
}

void
Atk::Object::notify_state_change(State state, bool value)
{
	atk_object_notify_state_change(atk_object(), state, value);
}

bool 
Atk::Object::add_relationship(RelationType relationship, const Object& target)
{
	return atk_object_add_relationship(atk_object(), (AtkRelationType)relationship, target.atk_object());
}
	
bool 
Atk::Object::remove_relationship(RelationType relationship, const Object& target)
{
	return atk_object_remove_relationship(atk_object(), (AtkRelationType)relationship, target.atk_object());
}

/*  Atk::ObjectClass
 */

void
Atk::ObjectClass::init(AtkObjectClass *g_class)
{
	G::ObjectClass::init((GObjectClass*)g_class);
	g_class->get_name = &get_name_proxy;
	g_class->get_description = &get_description_proxy;
	g_class->get_parent = &get_parent_proxy;
	g_class->get_n_children = &get_n_children_proxy;
	g_class->ref_child = &ref_child_proxy;
	g_class->get_index_in_parent = &get_index_in_parent_proxy;
	g_class->ref_relation_set = &ref_relation_set_proxy;
	g_class->get_role = &get_role_proxy;
	g_class->get_layer = &get_layer_proxy;
	g_class->get_mdi_zorder = &get_mdi_zorder_proxy;
	g_class->ref_state_set = &ref_state_set_proxy;
	g_class->set_name = &set_name_proxy;
	g_class->set_description = &set_description_proxy;
	g_class->set_parent = &set_parent_proxy;
	g_class->set_role = &set_role_proxy;
	g_class->children_changed = &children_changed_proxy;
	g_class->focus_event = &focus_event_proxy;
	g_class->property_change = &property_change_proxy;
	g_class->state_change = &state_change_proxy;
	g_class->visible_data_changed = &visible_data_changed_proxy;
	g_class->active_descendant_changed = &active_descendant_changed_proxy;
}

GType
Atk::ObjectClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(ATK_TYPE_OBJECT, (GClassInitFunc)init);
	}
	return type;
}

void*
Atk::ObjectClass::create()
{
	return g_object_new(get_type(), 0);
}

const gchar*
Atk::ObjectClass::get_name_proxy(AtkObject *accessible)
{
	const gchar *result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_name();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_name)
			result = g_class->get_name(accessible);
	}
	return result;
}

const gchar*
Atk::ObjectClass::get_description_proxy(AtkObject *accessible)
{
	const gchar *result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_description();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_description)
			result = g_class->get_description(accessible);
	}
	return result;
}

AtkObject*
Atk::ObjectClass::get_parent_proxy(AtkObject *accessible)
{
	AtkObject *result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_parent();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_parent)
			result = g_class->get_parent(accessible);
	}
	return result;
}

gint
Atk::ObjectClass::get_n_children_proxy(AtkObject *accessible)
{
	gint result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_n_children();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_n_children)
			result = g_class->get_n_children(accessible);
	}
	return result;
}

AtkObject*
Atk::ObjectClass::ref_child_proxy(AtkObject *accessible, gint i)
{
	AtkObject *result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_ref_child(i);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->ref_child)
			result = g_class->ref_child(accessible, i);
	}
	return result;
}

gint
Atk::ObjectClass::get_index_in_parent_proxy(AtkObject *accessible)
{
	gint result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_index_in_parent();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_index_in_parent)
			result = g_class->get_index_in_parent(accessible);
	}
	return result;
}

AtkRelationSet*
Atk::ObjectClass::ref_relation_set_proxy(AtkObject *accessible)
{
	AtkRelationSet *result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_ref_relation_set();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->ref_relation_set)
			result = g_class->ref_relation_set(accessible);
	}
	return result;
}

AtkRole
Atk::ObjectClass::get_role_proxy(AtkObject *accessible)
{
	AtkRole result = ATK_ROLE_INVALID;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_role();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_role)
			result = g_class->get_role(accessible);
	}
	return result;
}

AtkLayer
Atk::ObjectClass::get_layer_proxy(AtkObject *accessible)
{
	AtkLayer result = ATK_LAYER_INVALID;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_layer();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_layer)
			result = g_class->get_layer(accessible);
	}
	return result;
}

gint
Atk::ObjectClass::get_mdi_zorder_proxy(AtkObject *accessible)
{
	gint result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_get_mdi_zorder();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->get_mdi_zorder)
			result = g_class->get_mdi_zorder(accessible);
	}
	return result;
}

AtkStateSet*
Atk::ObjectClass::ref_state_set_proxy(AtkObject *accessible)
{
	AtkStateSet *result = 0;
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		result = tmp_accessible->do_ref_state_set();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->ref_state_set)
			result = g_class->ref_state_set(accessible);
	}
	return result;
}

void
Atk::ObjectClass::set_name_proxy(AtkObject *accessible, const gchar *name)
{
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		tmp_accessible->do_set_name(name);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->set_name)
			g_class->set_name(accessible, name);
	}
}

void
Atk::ObjectClass::set_description_proxy(AtkObject *accessible, const gchar *description)
{
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		tmp_accessible->do_set_description(description);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->set_description)
			g_class->set_description(accessible, description);
	}
}

void
Atk::ObjectClass::set_parent_proxy(AtkObject *accessible, AtkObject *parent)
{
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		tmp_accessible->do_set_parent(parent);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->set_parent)
			g_class->set_parent(accessible, parent);
	}
}

void
Atk::ObjectClass::set_role_proxy(AtkObject *accessible, AtkRole role)
{
	Atk::Object *tmp_accessible = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_accessible)
		tmp_accessible->do_set_role(role);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->set_role)
			g_class->set_role(accessible, role);
	}
}

void
Atk::ObjectClass::children_changed_proxy(AtkObject *accessible, guint change_index, gpointer changed_child)
{
	Atk::Object *tmp_object = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_object)
		tmp_object->on_children_changed(change_index, changed_child);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->children_changed)
			g_class->children_changed(accessible, change_index, changed_child);
	}
}

void
Atk::ObjectClass::focus_event_proxy(AtkObject *accessible, gboolean focus_in)
{
	Atk::Object *tmp_object = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_object)
		tmp_object->on_focus_event(focus_in);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->focus_event)
			g_class->focus_event(accessible, focus_in);
	}
}

void
Atk::ObjectClass::property_change_proxy(AtkObject *accessible, AtkPropertyValues *values)
{
	Atk::Object *tmp_object = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_object)
		tmp_object->on_property_change(values);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->property_change)
			g_class->property_change(accessible, values);
	}
}

void
Atk::ObjectClass::state_change_proxy(AtkObject *accessible, const gchar *name, gboolean state_set)
{
	Atk::Object *tmp_object = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_object)
	{
		String tmp_name(name);
		tmp_object->on_state_change(tmp_name, state_set);
	}
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->state_change)
			g_class->state_change(accessible, name, state_set);
	}
}

void
Atk::ObjectClass::visible_data_changed_proxy(AtkObject *accessible)
{
	Atk::Object *tmp_object = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_object)
		tmp_object->on_visible_data_changed();
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->visible_data_changed)
			g_class->visible_data_changed(accessible);
	}
}

void
Atk::ObjectClass::active_descendant_changed_proxy(AtkObject *accessible, gpointer *child)
{
	Atk::Object *tmp_object = G::Object::pointer<Atk::Object>(accessible);
	if (tmp_object)
		tmp_object->on_active_descendant_changed(child);
	else
	{
		AtkObjectClass *g_class = G::TypeInstance::class_peek_parent<AtkObjectClass>(ATK_OBJECT_GET_CLASS(accessible));
		if (g_class->active_descendant_changed)
			g_class->active_descendant_changed(accessible, child);
	}
}

/*  Atk::Object overridable methods
 */

const char*
Atk::Object::do_get_name()
{
	const char *result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_name)
		result = g_class->get_name(atk_object());
	return result;
}

const char*
Atk::Object::do_get_description()
{
	const char *result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_description)
		result = g_class->get_description(atk_object());
	return result;
}

AtkObject*
Atk::Object::do_get_parent()
{
	AtkObject *result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_parent)
		result = g_class->get_parent(atk_object());
	return result;
}

int
Atk::Object::do_get_n_children()
{
	int result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_n_children)
		result = g_class->get_n_children(atk_object());
	return result;
}

AtkObject*
Atk::Object::do_ref_child(int i)
{
	AtkObject *result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->ref_child)
		result = g_class->ref_child(atk_object(), i);
	return result;
}

int
Atk::Object::do_get_index_in_parent()
{
	int result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_index_in_parent)
		result = g_class->get_index_in_parent(atk_object());
	return result;
}

AtkRelationSet*
Atk::Object::do_ref_relation_set()
{
	AtkRelationSet *result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->ref_relation_set)
		result = g_class->ref_relation_set(atk_object());
	return result;
}

AtkRole
Atk::Object::do_get_role()
{
	AtkRole result = ATK_ROLE_INVALID;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_role)
		result = g_class->get_role(atk_object());
	return result;
}

AtkLayer
Atk::Object::do_get_layer()
{
	AtkLayer result = ATK_LAYER_INVALID;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_layer)
		result = g_class->get_layer(atk_object());
	return result;
}

int
Atk::Object::do_get_mdi_zorder()
{
	int result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->get_mdi_zorder)
		result = g_class->get_mdi_zorder(atk_object());
	return result;
}

AtkStateSet*
Atk::Object::do_ref_state_set()
{
	AtkStateSet *result = 0;
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->ref_state_set)
		result = g_class->ref_state_set(atk_object());
	return result;
}

void
Atk::Object::do_set_name(const char *name)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->set_name)
		g_class->set_name(atk_object(), name);
}

void
Atk::Object::do_set_description(const char *description)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->set_description)
		g_class->set_description(atk_object(), description);
}

void
Atk::Object::do_set_parent(AtkObject *parent)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->set_parent)
		g_class->set_parent(atk_object(), parent);
}

void
Atk::Object::do_set_role(AtkRole role)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->set_role)
		g_class->set_role(atk_object(), role);
}

/*  Atk::Object signal handlers
 */

void
Atk::Object::on_children_changed(unsigned int change_index, gpointer changed_child)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->children_changed)
		g_class->children_changed(atk_object(), change_index, changed_child);
}

void
Atk::Object::on_focus_event(bool focus_in)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->focus_event)
		g_class->focus_event(atk_object(), focus_in);
}

void
Atk::Object::on_property_change(AtkPropertyValues *values)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->property_change)
		g_class->property_change(atk_object(), values);
}

void
Atk::Object::on_state_change(const String& name, bool state_set)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->state_change)
		g_class->state_change(atk_object(), name.c_str(), state_set);
}

void
Atk::Object::on_visible_data_changed()
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->visible_data_changed)
		g_class->visible_data_changed(atk_object());
}

void
Atk::Object::on_active_descendant_changed(void **child)
{
	AtkObjectClass *g_class = class_peek_parent<AtkObjectClass>(atk_object_class());
	if (g_class->active_descendant_changed)
		g_class->active_descendant_changed(atk_object(), child);
}

// Atk::Object properties

const Atk::Object::AccessibleNamePropertyType Atk::Object::accessible_name_property("accessible-name");

const Atk::Object::AccessibleDescriptionPropertyType Atk::Object::accessible_description_property("accessible-description");

const Atk::Object::AccessibleParentPropertyType Atk::Object::accessible_parent_property("accessible-parent");

const Atk::Object::AccessibleValuePropertyType Atk::Object::accessible_value_property("accessible-value");

const Atk::Object::AccessibleRolePropertyType Atk::Object::accessible_role_property("accessible-role");

const Atk::Object::AccessibleComponentLayerPropertyType Atk::Object::accessible_component_layer_property("accessible-component-layer");

const Atk::Object::AccessibleComponentMdiZorderPropertyType Atk::Object::accessible_component_mdi_zorder_property("accessible-component-mdi-zorder");

const Atk::Object::AccessibleTableCaptionPropertyType Atk::Object::accessible_table_caption_property("accessible-table-caption");

const Atk::Object::AccessibleTableColumnDescriptionPropertyType Atk::Object::accessible_table_column_description_property("accessible-table-column-description");

const Atk::Object::AccessibleTableColumnHeaderPropertyType Atk::Object::accessible_table_column_header_property("accessible-table-column-header");

const Atk::Object::AccessibleTableRowDescriptionPropertyType Atk::Object::accessible_table_row_description_property("accessible-table-row-description");

const Atk::Object::AccessibleTableRowHeaderPropertyType Atk::Object::accessible_table_row_header_property("accessible-table-row-header");

const Atk::Object::AccessibleTableSummaryPropertyType Atk::Object::accessible_table_summary_property("accessible-table-summary");

// Atk::Object signals

const Atk::Object::ChildrenChangedSignalType Atk::Object::children_changed_signal("children_changed");

const Atk::Object::FocusEventSignalType Atk::Object::focus_event_signal("focus_event");

const Atk::Object::PropertyChangeSignalType Atk::Object::property_change_signal("property_change");

const Atk::Object::StateChangeSignalType Atk::Object::state_change_signal("state_change");

const Atk::Object::VisibleDataChangedSignalType Atk::Object::visible_data_changed_signal("visible_data_changed");

const Atk::Object::ActiveDescendantChangedSignalType Atk::Object::active_descendant_changed_signal("active_descendant_changed");

